package com.mini.framework.third.express.model;

import java.util.Comparator;
import java.util.Map;
import java.util.Optional;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import com.mini.framework.model.flag.EnumTypeShower;
import com.mini.framework.util.function.PredicateSpread;
import com.mini.framework.util.string.RegexUtil;

/**
 * 快递运输的阶段
 * @author jayheo
 *
 */
public enum ExpressTransportStage implements EnumTypeShower<ExpressTransportStage>{

	//快递单当前状态，包括0在途，1揽收，2疑难，3签收，4退签，5派件，6退回，7转投 等7个状态
	
	/**
	 * 快递100的退回
	 */
	sendBack("退回",10,new int[]{6}),
	
	/**
	 * 快递100的退回件已被发件人签收
	 */
	sendBackSign("退回",10,new int[]{4}),
	
	/**
	 *  转投其它的
	 */
	turnOther("转投其它的",9,new int[]{7}),

	/**
	 * 疑难位置
	 */
	unknown("疑难位置",8,new int[]{2},"疑难"),
	
	/**
	 * 已签收
	 */
	received("已签收",5,new int[]{3},"已签收","代签收"),
	
	
	/**
	 * 待取件
	 */
	requireTake("待取件",4,new int[]{},"请及时取件"),
	
	/**
	 * 派送中
	 */
	deliver("派送中",3,new int[]{5},"派送","派件"),
	
	/**
	 * 运输中
	 */
	progress("运输中",2,new int[]{0},"已发往","已到达","离开","中转部"),
	
	
	/**
	 * 快递公司已收货
	 */
	activation("已揽件",1,new int[]{1},"已收件","已揽收","已揽件"),
	
	
	;
	
	
	private String[] guessRegexs;
	
	private String title;
	
	/**
	 * 与快递100对应的code
	 */
	private int[] kuaidi100Codes;
	
	private int index;
	
	public String[] guessRegexs() {
		return guessRegexs;
	}

	public int index() {
		return index;
	}

	public String getTitle() {
		return title;
	}

	public void setTitle(String title) {
		this.title = title;
	}
	
	public int[] getKuaidi100Codes() {
		return kuaidi100Codes;
	}

	public void setKuaidi100Codes(int[] kuaidi100Codes) {
		this.kuaidi100Codes = kuaidi100Codes;
	}

	private static Comparator<ExpressTransportStage> guessMatchSorter(){
		return Comparator.comparing(ExpressTransportStage::index);
	}

	private ExpressTransportStage( String title,int index,int[] kuaidi100Codes,String... guessRegexs) {
		this.guessRegexs = guessRegexs;
		this.kuaidi100Codes = kuaidi100Codes;
		this.title = title;
		this.index = index;
	}
	
	public boolean guessMatch(String context) {
		if( context!=null){
			return Stream.of(guessRegexs).anyMatch(gr->RegexUtil.getMatch(gr, context)!=null);
		}else{
			return false;
		}
	}
	
	public static Optional<ExpressTransportStage> guessMatchOne(String context) {
		return Stream.of(values()).sorted(guessMatchSorter()).filter(PredicateSpread.applySecondParam(context,ExpressTransportStage:: guessMatch)).findFirst();
	}

	@Override
	public void fillFieldToMap(Map<String, Object> map) {
		map.put("title", title);
		map.put("guessRegexs", guessRegexs);
		map.put("index", index);
	}
	
	public boolean matchKudidi100Code(int code){
		return IntStream.of(kuaidi100Codes).anyMatch(kuaidiCode->kuaidiCode==code);
	}
	
	
	public static Optional<ExpressTransportStage> fromKuaidi100Code(int code){
		return Stream.of(values()).filter(stage->stage.matchKudidi100Code(code)).findFirst();
	}
	
	
	
}
