package com.mini.framework.third.weixin.mapp.model;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;


import com.mini.framework.third.weixin.mapp.model.state.MappCodePhotoUseEnv;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.gson.annotations.SerializedName;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.params.MapParams;
import com.mini.framework.util.string.GsonUtil;

/**
 * 小程序码图片
 * @author jayheo
 * <a href="https://developers.weixin.qq.com/miniprogram/dev/OpenApiDoc/qrcode-link/qr-code/getQRCode.html">文档地址</a>
 */
public class MappCodePhotoRequest {

    private static final Logger logger = LogManager.getLogger(MappCodePhotoRequest.class);

	/**
	 * 最大32个可见字符，只支持数字，大小写英文以及部分特殊字符：!#$&'()*+,/:;=?@-._~，其它字符请自行编码为合法字符（因不支持%，
	 * 中文无法使用 urlencode 处理，请使用其他编码方式）
	 */
	@SerializedName("scene")
	private String scene;

	/**
	 * 必须是已经发布的小程序存在的页面（否则报错），例如 pages/index/index, 根路径前不要填加
	 * /,不能携带参数（参数请放在scene字段里），如果不填写这个字段，默认跳主页面
	 */
	@SerializedName("page")
	private String page;

	/**
	 * 二维码的宽度，单位 px，最小 280px，最大 1280px
	 */
	@SerializedName("width")
	private Integer size;

	/**
	 * 自动配置线条颜色，如果颜色依然是黑色，则说明不建议配置主色调，默认 false
	 */
	@SerializedName("auto_color")
	private Boolean authColor;

	/**
	 * auto_color 为 false 时生效，使用 rgb 设置颜色 例如 {"r":"xxx","g":"xxx","b":"xxx"}
	 * 十进制表示
	 */
	@SerializedName("line_color")
	private Map<String, Integer> lineColor;

	/**
	 * 是否需要透明底色，为 true 时，生成透明底色的小程序
	 */
	@SerializedName("is_hyaline")
	private Boolean hyaline;


	/**
	 * 要不要检查路径是不是存在。
	 * <a href='https://developers.weixin.qq.com/miniprogram/dev/OpenApiDoc/qrcode-link/qr-code/getUnlimitedQRCode.html'> 文档说明 </a>
	 */
	@SerializedName("check_path")
	private Boolean strictPath = false;


	/**
	 * 部署的环境
	 * 要打开的小程序版本。正式版为 "release"，体验版为 "trial"，开发版为 "develop"。默认是正式版
	 */
	@SerializedName("env_version")
	private MappCodePhotoUseEnv useEnv;



	public void fillUseEnvRelease(){
		this.useEnv = MappCodePhotoUseEnv.release;
	}

	public void fillUseEnvTrial(){
		this.useEnv = MappCodePhotoUseEnv.trial;
	}

	public void fillUseEnvDevelop(){
		this.useEnv = MappCodePhotoUseEnv.develop;
	}



	public String getScene() {
		return scene;
	}

	/**
	 * 最大32个可见字符，只支持数字，大小写英文以及部分特殊字符：!#$&'()*+,/:;=?@-._~，其它字符请自行编码为合法字符（因不支持%，中文无法使用 urlencode 处理，请使用其他编码方式）
	 * @param scene
	 */
	public void setScene(String scene) {
		AssertUtil.assertSupport(scene==null || scene.length()<=32, "场景码不能大于32位", scene);
		this.scene = scene;
	}

	public String getPage() {
		return page;
	}

	public void setPage(String page) {
		this.page = page;
	}

	public Integer getSize() {
		return size;
	}

	public void setSize(Integer size) {
		this.size = size;
	}

	public Boolean getAuthColor() {
		return authColor;
	}

	public void setAuthColor(Boolean authColor) {
		this.authColor = authColor;
	}

	public Map<String, Integer> getLineColor() {
		return lineColor;
	}

	public void setLineColor(Map<String, Integer> lineColor) {
		this.lineColor = lineColor;
	}

	public void setLineColor(int red,int green,int blue) {
		HashMap<String, Integer> map = new HashMap<>();
		map.put("r", red);
		map.put("g", green);
		map.put("b", blue);
		lineColor = map;
	}

	public Boolean getHyaline() {
		return hyaline;
	}

	public void setHyaline(Boolean hyaline) {
		this.hyaline = hyaline;
	}

	public Boolean getStrictPath() {
		return strictPath;
	}

	public void setStrictPath(Boolean strictPath) {
		this.strictPath = strictPath;
	}

	public MappCodePhotoUseEnv getUseEnv() {
		return useEnv;
	}

	public void setUseEnv(MappCodePhotoUseEnv useEnv) {
		this.useEnv = useEnv;
	}

	public String toJson(){
		String json = GsonUtil.buildMilliSecondDateGson().toJson(this);
		return json;
	}
	
	
	/**
	 * 生成简易的小程序命令唤醒
	 * @param params 参数
	 * @param hitLengthDeposit 触发长度限制的时候可以通过这里来寄存。
	 * @return
	 */
	public static MappCodePhotoRequest createSimpleCommandAwake(MapParams params,Function<String, String> hitLengthDeposit){
		MappCodePhotoRequest request = new MappCodePhotoRequest();
		request.setPage(WeixinMappConstant.commandAwake01Pages);
		request.setHyaline(false);
		request.setAuthColor(false);
		request.setLineColor(0, 0, 0);//TODO 颜色是待处理的
		String scene = params.toUrlParams();
		if(scene.length()>32){//如果scene大于32位，不能生成了，所以要相一个代替32位的方法 
			logger.debug("scene:[{}]大于32位走commandDeposit寄存到数据库",scene);
			//见说明文档 http://wiki.xxggpt.com/docs/wykjdev/mapp-awake
			String key = hitLengthDeposit.apply(scene);
			params = MapParams.build("commandDeposit",key);
			scene = params.toUrlParams();
		}
		request.setScene(scene);
		return request;
	}

}
