package com.mini.framework.third.weixin.notice.server.handler;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.mini.framework.util.log.MiniLogLevel;
import com.mini.framework.util.log.Event5WBuilder;
import com.mini.framework.core.exception.HandleStringException;
import com.mini.framework.core.exception.standard.CustomException;
import com.mini.framework.third.weixin.notice.server.model.WeixinNoticeType;
import com.mini.framework.third.weixin.notice.server.model.WeixinOpenidSureResult;
import com.mini.framework.third.weixin.notice.server.model.resp.EventClickNotice;
import com.mini.framework.third.weixin.notice.server.model.resp.EventScanCodeNotice;
import com.mini.framework.third.weixin.notice.server.model.resp.EventScanSubNotice;
import com.mini.framework.third.weixin.notice.server.model.resp.EventSubscribeNotice;
import com.mini.framework.third.weixin.notice.server.model.resp.EventTemplateResultNotice;
import com.mini.framework.third.weixin.notice.server.model.resp.EventUnSubscribeNotice;
import com.mini.framework.third.weixin.notice.server.model.resp.EventViewNotice;
import com.mini.framework.third.weixin.notice.server.model.resp.EventWeixinAwakeSessionNotice;
import com.mini.framework.third.weixin.notice.server.model.resp.FileNotice;
import com.mini.framework.third.weixin.notice.server.model.resp.LocationNotice;
import com.mini.framework.third.weixin.notice.server.model.resp.TextNotice;
import com.mini.framework.third.weixin.notice.server.model.resp.VoiceNotice;
import com.mini.framework.third.weixin.notice.server.model.resp.WeixinAskMsg;
import com.mini.framework.third.weixin.notice.server.model.resp.WeixinNotice;

/**
 * 微信通知处理分发器
 * @author jayheo
 *
 */
//https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140453
public abstract class AbstractWeixinServerNoticeHandler implements WeixinServerNoticeHandler{

	private Logger logger = LogManager.getLogger(AbstractWeixinServerNoticeHandler.class);
	
	
	@Override
	public WeixinAskMsg handle(WeixinNotice weixinNotice){
		logger.info("收到微信通知:{}",weixinNotice);
		logger.debug("收到微信通知:{}",weixinNotice);
		//创建一个响应体，目前用text响应体
		WeixinAskMsg weixinAskMsg = new WeixinAskMsg(weixinNotice,WeixinNoticeType.text.getType());
		try{
			//默认提示如果后面没有作什么操作就会显示 这个默认的
			//weixinAskMsg.setContent("感谢您的关注，祝您购物愉快！");
			weixinAskMsg.setNoneContent();
			WeixinNoticeType type = weixinNotice.getWeixinNoticeType();
			this.dispatcher(type, weixinNotice, weixinAskMsg);
		}catch(HandleStringException e){
			logger.log(MiniLogLevel.getKeyBizLog(), "发现一个weixin notice异常:{}",e.getMessage());
			weixinAskMsg.setContent("数据处理异常");
			logger.error("处理微信任务失败",e);
		}catch(CustomException e){
			logger.log(MiniLogLevel.getKeyBizLog(), "发现一个weixin notice异常:{}",e.getMessage());
			logger.error("处理微信任务失败",e);
			weixinAskMsg.setContent(e.getMessage());
		}catch(RuntimeException e){
			logger.log(MiniLogLevel.getKeyBizLog(), "发现一个weixin notice未知异常,具体见后面的日志");
			logger.error("处理微信任务失败",e);
			weixinAskMsg.setContent("系统繁忙...");
		}
		return weixinAskMsg;
		
	}
	private void dispatcher(WeixinNoticeType type,WeixinNotice weixinNotice,WeixinAskMsg weixinAskMsg){
		logger.debug("收到 weixinNotice:{}",weixinNotice);
		switch (type) {
		case file:
			this.handleFile((FileNotice)weixinNotice,weixinAskMsg);
			break;
		case text:
			this.handleText((TextNotice)weixinNotice,weixinAskMsg);
			break;
		case voice:
			this.handleVoice((VoiceNotice)weixinNotice,weixinAskMsg);
			break;
		case location:
			this.handleLocation((LocationNotice)weixinNotice,weixinAskMsg);
			break;
		case eventScancodeWaitmsg:
			this.handleEvent((EventScanCodeNotice)weixinNotice,weixinAskMsg);
			break;
		case eventView:
			this.handleEvent((EventViewNotice)weixinNotice,weixinAskMsg);
			break;
		case eventSubscribe:
			this.handleEvent((EventSubscribeNotice)weixinNotice,weixinAskMsg);
			break;
		case eventUnsubscribe:
			this.handleEvent((EventUnSubscribeNotice)weixinNotice,weixinAskMsg);
			break;	
		case eventScan:
			this.handleEvent((EventScanSubNotice)weixinNotice,weixinAskMsg);
			break;
		case eventTemplateResult:
			this.handleEvent((EventTemplateResultNotice)weixinNotice,weixinAskMsg);
			break;
		case eventMappAwake:
			this.handleEvent((EventWeixinAwakeSessionNotice)weixinNotice,weixinAskMsg);
			break;
		case eventClick:
			this.handleEvent((EventClickNotice)weixinNotice,weixinAskMsg);
			break;
			//TODO 还有部分情况没有处理
		default:
			logger.info("收到微信事件还没有定义处理的办法,事件:{}",weixinNotice);
			weixinAskMsg.setContent(String.format("收到%s消息",weixinNotice.getType()));
		}
		logger.debug("响应 weixinAskMsg:{}",weixinAskMsg);
	}
	
	
	
	protected void handleEvent(EventClickNotice weixinNotice, WeixinAskMsg weixinAskMsg) {
		weixinAskMsg.setContent("点击了key:%s" , weixinNotice.showButtonFlagKey());
	}
	/**
	 * 微信唤醒对话，比如说小程序中点客服消息进入客服聊天
	 * @param weixinNotice
	 * @param weixinAskMsg
	 */
	protected void handleEvent(EventWeixinAwakeSessionNotice weixinNotice, WeixinAskMsg weixinAskMsg) {
		weixinAskMsg.setContent("知道你打开会话了sessionParam:" + weixinNotice.getSessionParam());
	}

	
	
	protected abstract void handleEvent(EventTemplateResultNotice weixinNotice, WeixinAskMsg weixinAskMsg) ;

	/**
	 * 处理微信扫码关注公众号的的请求
	 * @param weixinNotice
	 * @param weixinAskMsg
	 */
	protected void handleEvent(EventScanSubNotice weixinNotice, WeixinAskMsg weixinAskMsg){
		//参考报文 event.scan.xml
		String sceneCode = weixinNotice.getEventKey();
		WeixinOpenidSureResult sureUser = weixinNotice.getOpenidUser();
		this.handleSubscribe(weixinNotice.getOpenidUser());			
		Event5WBuilder event = Event5WBuilder.event(2, "inviteSubWeixin").who("用户:[%s]", sureUser.getUserId())
				.what("扫码(带code:[%s])关注了:[%s]",sceneCode, weixinNotice.getServAcc()).when("非首次关注");//怎么知道是不是首页关注 见 EventSubscribeNotice
		logger.debug(event);
		this.handleSceneInvite(weixinNotice.getWeixinNoticeType(),weixinNotice.getOpenidUser(), sceneCode);
	}
	
	
	
	
	/**
	 * 处理用户发送地址的事件
	 * @param weixinNotice
	 * @param weixinAskMsg
	 */
	protected void handleLocation(LocationNotice weixinNotice, WeixinAskMsg weixinAskMsg) {
		logger.info("用户发送一个地址:{}",weixinNotice);
		weixinAskMsg.setContent("原来你在 %s ,我马上飞过来",weixinNotice.getLabel());
	}
	/**
	 * 处理用户发送文件的事件
	 * @param weixinNotice
	 * @param weixinAskMsg
	 */
	protected  void handleFile(FileNotice weixinNotice, WeixinAskMsg weixinAskMsg) {
		logger.info("用户上传了一个文件:{}",weixinNotice);
		weixinAskMsg.setContent(String.format("收到文件:%s,%s",weixinNotice.getTitle(),weixinNotice.getDesc()));
	}
	
	/**
	 * 处理用户取消关注
	 * 这种数据可能拿不到了微信用户信息了所以不要去拿了
	 * @param weixinNotice
	 * @param weixinAskMsg
	 */
	protected abstract  void handleEvent(EventUnSubscribeNotice weixinNotice, WeixinAskMsg weixinAskMsg) ;
	/**
	 * https://developers.weixin.qq.com/doc/offiaccount/Message_Management/Receiving_event_pushes.html
	 * 处理用户关注
	 * @param weixinNotice
	 * @param weixinAskMsg
	 */
	protected  void handleEvent(EventSubscribeNotice weixinNotice, WeixinAskMsg weixinAskMsg) {
		weixinAskMsg.setContent("感谢你的关注");
		WeixinOpenidSureResult sureUser = weixinNotice.getOpenidUser();
		this.handleSubscribe(sureUser);
		String sceneCode = weixinNotice.showRightInviteCode();
		if(sceneCode!=null){//根据微信文档如果有ticket就是扫码关注
			Event5WBuilder event = Event5WBuilder.event(2, "inviteSubWeixin").who("用户:[%s]", sureUser.getUserId())
					.what("扫码(带code:[%s])关注了:[%s]",sceneCode, weixinNotice.getServAcc()).when("首次关注");//怎么知道是不是首页关注 见 EventSubscribeNotice
			logger.debug(event);
			this.handleSceneInvite(weixinNotice.getWeixinNoticeType(), sureUser, sceneCode);
		}
	}
	
	
	/**
	 * 处理菜单点击事件
	 * @param weixinNotice
	 * @param weixinAskMsg
	 */
	protected  void handleEvent(EventViewNotice eventNotice, WeixinAskMsg weixinAskMsg) {
		String url = eventNotice.getViewUrl();
		logger.info("用户使用按钮:{}打开了url:{}",eventNotice.getMenuId(),url);
	}
	
	/**
	 * 处理扫码事件
	 * @param weixinNotice
	 * @param weixinAskMsg
	 */
	protected  void handleEvent(EventScanCodeNotice weixinNotice, WeixinAskMsg weixinAskMsg) {
		weixinAskMsg.setContent("你干嘛拿我扫！");
		weixinNotice.getScanInfo().getType();
	}
	
	
	

	/**
	 * 处理text通知
	 * @param textNotice
	 * @param weixinAskMsg
	 * @return
	 */
	protected  void handleText(TextNotice textNotice, WeixinAskMsg weixinAskMsg){
		weixinAskMsg.setContent("已收到一个文本消息:" + textNotice.getContent());
	}
	/**
	 * https://mp.weixin.qq.com/advanced/advanced?action=table&token=738852136&lang=zh_CN
	 * 处理收到声音通知 这个功能默认没有被开启需要去开启。
	 * @param voiceNotice
	 * @param weixinAskMsg
	 * @return
	 */
	protected  void handleVoice(VoiceNotice voiceNotice, WeixinAskMsg weixinAskMsg){
		//https://mp.weixin.qq.com/advanced/advanced?action=table&token=738852136&lang=zh_CN
		weixinAskMsg.setContent(String.format("我听到的是:[%s]",voiceNotice.getTranText()));
	}
	
	
	/**
	 * 处理扫码推荐
	 * @param weixinNoticeType 
	 * @param openid
	 */
	protected abstract void handleSceneInvite(WeixinNoticeType weixinNoticeType, WeixinOpenidSureResult openid,String inviteCode);
	

	/**
	 * 处理关注事件
	 * @param openid
	 * @param inviteCode
	 */
	protected abstract void handleSubscribe(WeixinOpenidSureResult openid);
	
}
