package com.mini.framework.third.weixin.pay.client;

import java.io.IOException;

import javax.net.ssl.SSLContext;

import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.RequestBody;
import okhttp3.logging.HttpLoggingInterceptor;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import retrofit2.Call;
import retrofit2.Response;
import retrofit2.Retrofit;
import retrofit2.converter.simplexml.SimpleXmlConverterFactory;

import com.mini.framework.core.exception.HandleIOException;
import com.mini.framework.core.exception.ThirdException;
import com.mini.framework.core.status.Status;
import com.mini.framework.third.weixin.pay.model.SitePayStatusType;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.log.MiniLogLevel;
import com.mini.framework.util.string.XStreamUtil;


public class PayApiClient {

	private static Logger logger = LogManager.getLogger(PayApiClient.class);
	
	private static final String wxpayServerUrl = "https://api.mch.weixin.qq.com/";

    /**
     * 微信支付的服务
     */
    public static Retrofit wxpayServer = new Retrofit.Builder()
    .baseUrl(wxpayServerUrl)
    //TODO 如何把换行去掉呢
    .addConverterFactory(SimpleXmlConverterFactory.createNonStrict())
    .client(new OkHttpClient.Builder()
            .addInterceptor((new HttpLoggingInterceptor().setLevel(HttpLoggingInterceptor.Level.BODY)))
            .build()
    )
    .build();
    
    
    public static  Retrofit wxpayCertServer (SSLContext sslContext){
    	return new Retrofit.Builder()
	    .baseUrl(wxpayServerUrl)
	    //TODO 如何把换行去掉呢
	    .addConverterFactory(SimpleXmlConverterFactory.createNonStrict())
	    .client(new OkHttpClient.Builder()
	    		.sslSocketFactory(sslContext.getSocketFactory())
	            .addInterceptor((new HttpLoggingInterceptor().setLevel(HttpLoggingInterceptor.Level.BODY)))
	            .build()
	    )
	    .build();
    }
    


    /**
     * 微信公众号的h5支付
     * @param request
     * @return
     */
    public static PublicPrePayResponse publicPay(PublicPrePayRequest request){
		String xmlReq = XStreamUtil.toXml(request);
		logger.log(MiniLogLevel.getKeyBizLog(), "准备发到腾讯支付报文xml:{}", xmlReq);
		RequestBody body = RequestBody.create(MediaType.parse("application/xml; charset=utf-8"), xmlReq);
		Call<PublicPrePayResponse> call = PayApiClient.wxpayServer.create(PayApi.class).publicPay(body);
		try {
			Response<PublicPrePayResponse> retrofitResponse = call.execute();
			AssertUtil.simpleAssertByStatus(retrofitResponse.isSuccessful(), Status.Third.weixin, "支付失败");
			PublicPrePayResponse response = retrofitResponse.body();
			checkStatus(response);
			return response;
		} catch (IOException e) {
			//TODO 这里还要处理一下
			throw new HandleIOException(e, "微信支付网络异常");
		}
    }
    
    
    
    /**
     * 微信公众号的h5支付退款
     * @param request
     * @return
     */
    public static PublicRefundApplyResponse publicRefundApply(SSLContext sslContext,PublicRefundApplyRequest request){
		String xmlReq = XStreamUtil.toXml(request);
		logger.log(MiniLogLevel.getKeyBizLog(), "准备发到腾讯退款报文xml:{}", xmlReq);
		RequestBody body = RequestBody.create(MediaType.parse("application/xml; charset=utf-8"), xmlReq);
		Call<PublicRefundApplyResponse> call = wxpayCertServer(sslContext  ).create(PayApi.class).refoundApply(body);
		try {
			Response<PublicRefundApplyResponse> retrofitResponse = call.execute();
			AssertUtil.simpleAssertByStatus(retrofitResponse.isSuccessful(), Status.Third.weixin, "支付失败");
			PublicRefundApplyResponse response = retrofitResponse.body();
			checkStatus(response);
			return response;
		} catch (IOException e) {
			//TODO 这里还要处理一下
			throw new HandleIOException(e, "微信支付网络异常");
		}
    }
    
    
    
    
    //--------------------------



    /**
     * 小程序支付
     * @param request
     * @return
     */
    public static MappPrePayResponse mappPay(MappPrePayRequest request){
		String xmlReq = XStreamUtil.toXml(request);
		logger.log(MiniLogLevel.getKeyBizLog(), "准备发到腾讯支付报文xml:{}", xmlReq);
		RequestBody body = RequestBody.create(MediaType.parse("application/xml; charset=utf-8"), xmlReq);
		Call<MappPrePayResponse> call = PayApiClient.wxpayServer.create(PayApi.class).mappPay(body);
		try {
			Response<MappPrePayResponse> retrofitResponse = call.execute();
			AssertUtil.simpleAssertByStatus(retrofitResponse.isSuccessful(), Status.Third.weixin, "支付失败");
			MappPrePayResponse response = retrofitResponse.body();
			checkStatus(response);
			return response;
		} catch (IOException e) {
			//TODO 这里还要处理一下
			throw new HandleIOException(e, "微信支付网络异常");
		}
    }


    public static QueryOrderResponse queryOrder(QueryOrderRequest request){
		String xmlReq = XStreamUtil.toXml(request);
		logger.log(MiniLogLevel.getKeyBizLog(), "准备发到腾讯支付的{}报文xml:{}",request.messageType(), xmlReq);
		RequestBody body = RequestBody.create(MediaType.parse("application/xml; charset=utf-8"), xmlReq);
		Call<QueryOrderResponse> call = PayApiClient.wxpayServer.create(PayApi.class).orderQuery(body);
		try {
			Response<QueryOrderResponse> retrofitResponse = call.execute();
			AssertUtil.simpleAssertByStatus(retrofitResponse.isSuccessful(), Status.Third.weixin, "支付失败");
			QueryOrderResponse response = retrofitResponse.body();
			checkStatus(response);
			return response;
		} catch (IOException e) {
			//TODO 这里还要处理一下
			throw new HandleIOException(e, "微信支付网络异常");
		}
    }

    public static MappSiteTransferInfoResponse mappTransferInfo(SiteQueryTransferInfoPreRequest request, SSLContext cert) {

        Retrofit payServer = getWxPayServer(cert);
        String xmlReq = XStreamUtil.toXml(request);
        logger.log(MiniLogLevel.getKeyBizLog(), "准备发到腾讯支付的{}报文xml:{}", request.messageType(), xmlReq);
        RequestBody body = RequestBody.create(MediaType.parse("application/xml; charset=utf-8"), xmlReq);
        Call<MappSiteTransferInfoResponse> call = payServer.create(PayApi.class).transferInfoQuery(body);

        try {
            Response<MappSiteTransferInfoResponse> retrofitResponse = call.execute();
            AssertUtil.simpleAssertByStatus(retrofitResponse.isSuccessful(), Status.Third.weixin, "支付失败");
            MappSiteTransferInfoResponse response = retrofitResponse.body();
            checkStatus(response);
            return response;
        } catch (IOException e) {

            throw new HandleIOException(e, "微信支付网络异常");
        }
    }

    /**
     * 企业付款到用户零钱
     *
     * @param request 向用户支付请求
     * @return MappTransferResponse
     */
    public static MappTransferResponse wechatPayTransfer(UserMappTransferPreRequest request, SSLContext cert) {

        Retrofit payServer = getWxPayServer(cert);
        String xmlReq = XStreamUtil.toXml(request);
        logger.log(MiniLogLevel.getKeyBizLog(), "准备发到腾讯支付的{}报文xml:{}", request.messageType(), xmlReq);
        RequestBody requestBody = RequestBody.create(MediaType.parse("application/xml; charset=utf-8"), xmlReq);
        Call<MappTransferResponse> call = payServer.create(PayApi.class).transfers(requestBody);

        try {

            Response<MappTransferResponse> retrofitResponse = call.execute();
            AssertUtil.simpleAssertByStatus(retrofitResponse.isSuccessful(), Status.Third.weixin, "支付失败");
            MappTransferResponse response = retrofitResponse.body();
            checkStatus(response);
            return response;

        } catch (IOException e) {

            throw new HandleIOException(e, "微信支付网络异常");
        }


    }

    private static Retrofit getWxPayServer(SSLContext sslContext) {
        return new Retrofit.Builder()
                .baseUrl("https://api.mch.weixin.qq.com/")
                .addConverterFactory(SimpleXmlConverterFactory.createNonStrict())
                .client(new OkHttpClient.Builder().hostnameVerifier((s, sslSession) -> true)
                        .retryOnConnectionFailure(true)
                        .sslSocketFactory(sslContext.getSocketFactory())
                        .addInterceptor((new HttpLoggingInterceptor().setLevel(HttpLoggingInterceptor.Level.BODY)))
                        .build()
                ).build();
    }


    public static MappRefundResponse mappRefund(UserMappRefundPreRequest request) {

        String xmlReq = XStreamUtil.toXml(request);
        logger.log(MiniLogLevel.getKeyBizLog(), "准备发到腾讯支付的{}报文xml:{}", request.messageType(), xmlReq);
        RequestBody requestBody = RequestBody.create(MediaType.parse("application/xml; charset=utf-8"), xmlReq);
        Call<MappRefundResponse> call = PayApiClient.wxpayServer.create(PayApi.class).refund(requestBody);

        try {
            Response<MappRefundResponse> retrofitResponse = call.execute();
            AssertUtil.simpleAssertByStatus(retrofitResponse.isSuccessful(), Status.Third.weixin, "退款失败");
            MappRefundResponse response = retrofitResponse.body();
            checkStatus(response);
            return response;
        } catch (IOException e) {

            throw new HandleIOException(e, "微信支付网络异常");
        }
    }


    private static void checkStatus(StatusAbleResponse response) {
        String bizType = response.bizType();
        // 这里通讯级别的错误判断。
        if (!"SUCCESS".equals(response.connCode())) {
        	if("订单已全额退款".equals(response.connDesc())){
        		throw new ThirdException(Status.Third.refundFullOfOrder, "%s订单已全部退款了code:%s,msg:%s", bizType, response.connCode(), response.connDesc());
        	}else if("refund_fee大于可退金额".equals(response.connDesc())){
        		throw new ThirdException(Status.Third.refundOverOrderLimit, "%s退款订单大于可退的钱code:%s,msg:%s", bizType, response.connCode(), response.connDesc());
        	}else if("你的操作过于频繁，请稍后再试。".equals(response.connDesc())){
        		throw new ThirdException(Status.Third.refundIllegalOperateQpsLimit, "%s退款操作过于频繁code:%s,msg:%s", bizType, response.connCode(), response.connDesc());
        	}else if("同一笔订单的退款数量超过限制".equals(response.connDesc())){
        		throw new ThirdException(Status.Third.refundIllegalOperateTimeLimit, "%s退款操作超过最大次数code:%s,msg:%s", bizType, response.connCode(), response.connDesc());
        	}else if("appid和mch_id不匹配".equals(response.connDesc())){
                //公众号支付通讯失败code:FAIL,msg:appid和mch_id不匹配
                throw new ThirdException(Status.Third.configAppidUnFoundError, "%s商户号中没有对应的微信appid,code:%s,msg:%s", bizType, response.connCode(), response.connDesc());
            }
            //TODO 以后可以精细化错误码
            throw new ThirdException(Status.Third.wxConn, "%s通讯失败code:%s,msg:%s", bizType, response.connCode(), response.connDesc());
        }
        // 这里业务级别的错误判断。
        if (!response.success()) {
            SitePayStatusType payStatus = SitePayStatusType.weixinPublicStatus(response.bizCode());
            if (payStatus != null) {
                throw new ThirdException(payStatus.getMap(), "%s业务错误code:%s,msg:%s", bizType, response.bizCode(), response.bizDesc());
            }
            //TODO 以后可以精细化错误码
            throw new ThirdException(Status.Third.weixin, "%s业务错误code:%s,msg:%s", bizType, response.bizCode(), response.bizDesc());
        }
    }
}
