package com.mini.framework.third.weixin.wxpay.util.certificate;

import com.mini.framework.core.exception.HandleIOException;
import com.mini.framework.third.weixin.wxpay.v3.exception.WxpaySignatureConfigException;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.params.MapParams;
import org.apache.commons.lang3.RandomStringUtils;
import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Base64;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 微信支付 商户号证书 容器
 * @author jayheo
 */
public class WxpayMarketCertificateContainer {

    private final String marketKey;
    private final String certificateSerialNumber;
    private final String certificateContentString;

    public WxpayMarketCertificateContainer(WxpayMarketCertificateConfig certificateConfig) {
        this(certificateConfig.sureHaveMarketKeyForSignature(),certificateConfig.sureHaveSerialNumberForSignature(),certificateConfig.sureHaveContentByteStringForSignature());
    }

    public WxpayMarketCertificateContainer(String marketKey, String certificateSerialNumber, String certificateContentString) {
        AssertUtil.assertMethodRequire("marketKey",marketKey);
        AssertUtil.assertMethodRequire("certificateSerialNumber",certificateSerialNumber);
        AssertUtil.assertMethodRequire("certificateContentString",certificateContentString);

        // TODO  做适当的预检查有例如减少错误错误的概率。
        createPrivateKeyByContentString(certificateContentString);
        this.marketKey = marketKey;
        this.certificateSerialNumber = certificateSerialNumber;
        this.certificateContentString = certificateContentString;
    }

    public static WxpayMarketCertificateContainer createByCertificateConfig(WxpayMarketCertificateConfig certificateConfig){
        return new WxpayMarketCertificateContainer(certificateConfig);
    }

    public String getMarketKey() {
        return marketKey;
    }

    public String getCertificateSerialNumber() {
        return certificateSerialNumber;
    }

    public String getCertificateContentString() {
        return certificateContentString;
    }




    public String sureHaveMarketKeyForSignature()throws WxpaySignatureConfigException {
        if(marketKey==null){
            throw new WxpaySignatureConfigException("配置参数:marketKey 不能为空");
        }
        return marketKey;
    }



    public String sureHaveCertificateSerialNumberForSignature()throws WxpaySignatureConfigException {
        if(certificateSerialNumber==null){
            throw new WxpaySignatureConfigException("配置参数:serialNumber 不能为空");
        }
        return certificateSerialNumber;
    }



    public String sureHaveCertificateContentStringForSignature()throws WxpaySignatureConfigException {
        if(certificateContentString==null){
            throw new WxpaySignatureConfigException("配置参数:contentByteString 不能为空");
        }
        return certificateContentString;
    }



    //------------------------------------------------------------------------------


    /**
     * 签名然后回填的
     * @param params
     * @param target
     */
    public void signatureThenFillBack(CertificateSignatureRequireParams params, HandlerAfterCertificateSignature target){
        String nonceStr = RandomStringUtils.randomAlphanumeric(20);
        String timestamp = String.valueOf(System.currentTimeMillis()/1000);
        String signature = countSignature(params.sureHaveApiMethodForSignature(), params.sureHaveApiUrlPathForSignature(), timestamp, nonceStr, params.sureHaveRequestBodyForSignature());
        String authToken = countAuthToken(params.sureHaveApiMethodForSignature(),params.sureHaveApiUrlPathForSignature(),params.sureHaveRequestBodyForSignature(),nonceStr,timestamp,signature);
        target.foundSignature(signature);
        target.foundHeaderAuthToken(authToken);
        
    }

    /**
     * 计算授权的token
     * https://wechatpay-api.gitbook.io/wechatpay-api-v3/qian-ming-zhi-nan-1/qian-ming-sheng-cheng
     * https://wechatpay-api.gitbook.io/wechatpay-api-v3/chang-jian-wen-ti/qian-ming-xiang-guan
     * 发现一个比较好的postman的例子，测试过可以用
     * https://github.com/wechatpay-apiv3/wechatpay-postman-script
     * @param method
     * @param apiUrl
     * @param body
     * @param nonceStr
     * @param timestamp
     * @param signature
     * @return
     */
    private String countAuthToken(String method, String apiUrl, String body, String nonceStr, String timestamp, String signature) {
        MapParams params = MapParams.build()
                .param("mchid", marketKey)
                .param("nonce_str", nonceStr)
                .param("timestamp", timestamp)
                .param("serial_no", certificateSerialNumber)
                .param("signature", signature);

        String token = params.entrySet().stream()
                .map(entry->String.format("%s=\"%s\"", entry.getKey(),entry.getValue()))
                .collect(Collectors.joining(","));
        return "WECHATPAY2-SHA256-RSA2048 " + token;
    }


    /**
     * 计算签名
     * @param method
     * @param apiUrl
     * @param timestamp
     * @param nonceStr
     * @param body
     * @return
     */
    private String countSignature(String method, String apiUrl, String timestamp, String nonceStr, String body) {
        String massage = Stream.of(method,apiUrl,timestamp,nonceStr,body)
                .map(Objects::toString)
                .collect(Collectors.joining("\n","","\n"));
        return signatureThenBase64Message(massage, certificateContentString);
    }


    /**
     * 绝大多数编程语言提供的签名函数支持对签名数据 进行签名。强烈建议商户调用该类函数，使用商户私钥对待签名串进行SHA256 with RSA签名，并对签名结果进行Base64编码得到签名值。
     * 给一条信息进行签名 再 base64
     * @param message
     * @param certDataString
     * @return
     */
    public static String signatureThenBase64Message(String message, String certDataString) {
        try{
            Signature signatureTool = createSHA256withRSASignature();
            PrivateKey yourPrivateKey = createPrivateKeyByContentString(certDataString);
            signatureTool.initSign(yourPrivateKey);
            signatureTool.update(message.getBytes(StandardCharsets.UTF_8));
            return Base64.getEncoder().encodeToString(signatureTool.sign());
        }catch (InvalidKeyException | SignatureException e){
            throw new HandleIOException(e,"签名的时候出错");
        }
    }


    /**
     * 创建一个 SHA256withRSA 实体
     * @return
     */
    public static Signature createSHA256withRSASignature(){
        try {
            return Signature.getInstance("SHA256withRSA");
        } catch (NoSuchAlgorithmException e) {
            throw new HandleIOException(e,"创建一个 SHA256withRSA 签名实例失败");
        }
    }

    /**
     * 获取私钥。
     *
     * @param contentString 证书内容
     * @return 私钥对象
     */
    public static PrivateKey createPrivateKeyByContentString(String contentString)  {
        AssertUtil.assertMethodRequire(contentString,"certificateContentString");
        String clearContent = contentString
                .replace("-----BEGIN PRIVATE KEY-----", "")
                .replace("-----END PRIVATE KEY-----", "")
                .replaceAll("\\s+", "");
        try {
            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
            return keyFactory.generatePrivate(
                    new PKCS8EncodedKeySpec(Base64.getDecoder().decode(clearContent)));
        } catch (NoSuchAlgorithmException e) {
            throw new HandleIOException(e,"当前Java环境不支持RSA");
        } catch (InvalidKeySpecException e) {
            //TODO 碰到这个问题时好好检查一下，如果如何提示更好
            throw new HandleIOException(e,"无效的密钥格");
        }
    }
}
