package com.mini.framework.third.weixin.console.server;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.mini.framework.core.exception.BadReqException;
import com.mini.framework.core.exception.standard.CustomException;
import com.mini.framework.core.status.Status;
import com.mini.framework.third.weixin.console.ConsoleProxyCommandHelper;
import com.mini.framework.third.weixin.console.model.WeixinCurlCommand;
import com.mini.framework.third.weixin.console.model.WeixinCurlCommandSectionable;
import com.mini.framework.third.weixin.console.server.handler.TemplateSyncHandler;
import com.mini.framework.third.weixin.console.server.model.template.Document;
import com.mini.framework.third.weixin.console.server.model.template.DocumentListResponse;
import com.mini.framework.third.weixin.console.server.model.template.SyncItemResultType;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.log.Event5WBuilder;
import com.mini.framework.util.log.MiniLogLevel;
import com.mini.framework.util.string.RegexUtil;

public class WeixinServerTemplateHelper {
	
	private static Logger logger = LogManager.getLogger(WeixinServerTemplateHelper.class);
	
	public static void syncTemplateDocument(String consoleCommand,int beginSyncOffset, int syncLimit, TemplateSyncHandler handler){
		syncTemplateDocument(consoleCommand, beginSyncOffset, syncLimit, handler, (msg)-> new BadReqException(Status.BadReq.illParam, "同步微信模板文档时出错:%s", msg));
	}
	/**
	 * 同步微信文档
	 * @param syncLimit 
	 * @param beginSyncOffset 
	 * @return
	 */
	public static< E extends CustomException> void syncTemplateDocument(String consoleCommand,int beginSyncOffset, int syncLimit, TemplateSyncHandler handler,Function<String, E> exceptionSupplier) throws E{
		
		int limitIndex = beginSyncOffset+syncLimit;
		
		int sectionSize = 100;
		WeixinCurlCommandSectionable command = new WeixinCurlCommandSectionable(ConsoleProxyCommandHelper.parseBachCurl(consoleCommand))
		.setSection(beginSyncOffset, Math.min(syncLimit,sectionSize), sectionSize);

		String apiUrl = "https://mp.weixin.qq.com/advanced/tmplmsg";
		AssertUtil.assertNoBadReq(command.getUrl().startsWith(apiUrl),Status.BadReq.illParam,"从命令中得到url:[%s]必须以:[%s]为起始值",command.getUrl(),apiUrl);
		Integer count = null;
		boolean firstReq = true;
		do{
			//查询模板消息列表必须以这个为起始值
			String responseBody = ConsoleProxyCommandHelper.exeUseUrlAndCookie(command.getUrl(), command.getCookie());
			logger.log(MiniLogLevel.getKeyBizLog(),Event5WBuilder.event(2, "syncWeixinServerTemplateDocument").what("得到微信端的响应:%s", responseBody));
			DocumentListResponse response =DocumentListResponse.fromJson(responseBody).loadData(exceptionSupplier);
			count = response.getDocumentCount();
			AssertUtil.simpleAssertByStatus(count!=null, Status.Third.wxTemplate,"同步模板的时请求中分页响应没有发现必要的count");
			if(firstReq && !handler.readyIfExecute(response.getDocumentCount())){
				// 如果不开始就结束了
				return ;
			}
			
			List<Document> reomteDocumentSimples = response.getDocuments();
			Set<String> keys = reomteDocumentSimples.stream().map(Document::getKey).filter(Objects::nonNull).collect(Collectors.toSet());
			List<Document> localDocuments = handler.queryLocalDocuments(keys);

			for (Document remoteDocumentSimple : reomteDocumentSimples) {
				SyncItemResultType oneResult = SyncItemResultType.undo;
				//遍历所以微信端的模板
				//情况1，如果本地没有。直接添加。
				//情况2，如果本地有，但是本地的更新时间小于微信端的时间，直接要求替换(至于替换前要不要备份，由handler决定)。
				//情况3，如果与本地内容不一致，直接要求替换。
				final String key = remoteDocumentSimple.getKey();
				handler.beforeOneHandle(key);
				Optional<Document> localDocumentOptional = localDocuments.stream().filter(d->d.matchKey(key)).findFirst();
				
				boolean todoNewRecord = !localDocumentOptional.isPresent();
				boolean todoUpdate = localDocumentOptional.filter(localDocument->localDocument.ifRequireResyncFromSimple(remoteDocumentSimple)).isPresent();
				if(todoNewRecord||todoUpdate){
					Event5WBuilder event = Event5WBuilder.event(2, "syncTemplateDocumentDiscoverChange").who("发现需要同步的文档:[%s]", key)
							.what("todoNewRecord:[%s],todoUpdate:[%s]", todoNewRecord,todoUpdate)
							.how("检查远程和本地的文件 remoteSameKey:[%s],localSameKey:[%s]"
									,remoteDocumentSimple.simpleKeyForUpdate(),localDocumentOptional.map(Document::simpleKeyForUpdate).orElse("没有本地内容"));
					logger.info(event);
					Optional<Document> fullRemoteDocmentOptional = queryTemplateDocument(key, command);
					if(fullRemoteDocmentOptional.isPresent()){
						Document fullRemoteDocment = fullRemoteDocmentOptional.get();
						if(todoUpdate){
							handler.setDocument(key,fullRemoteDocment);
							oneResult = SyncItemResultType.update;
						}
						if(todoNewRecord){
							handler.addDocument(fullRemoteDocment);
							oneResult = SyncItemResultType.newRecord;
						}
					}else{
						//TODO 没有拿到数据是什么原因要怎么办呢
						oneResult = SyncItemResultType.reqError;
					}
				}
				handler.afterOneHandle(key,oneResult);
			}
			firstReq = false;
		}while( command.nextSection(Math.min(limitIndex,count)));
		handler.endHandle();
	}
	
	/**
	 * 查询某个模板的文档
	 * @param key
	 * @param command
	 * @return
	 */
	public static Optional<Document> queryTemplateDocument(String key,WeixinCurlCommand command){
		String urlTemp = "https://mp.weixin.qq.com/advanced/tmplmsg?action=tmpl_preview&t=tmplmsg/preview&id=%s&token=%s&lang=zh_CN";
		String token = command.getTokenParam();
		String url = String.format(urlTemp, key,token);
		String responseBody = ConsoleProxyCommandHelper.exeUseUrlAndCookie(url, command.getCookie());
		List<String> lines = Stream.of(responseBody.split("\r\n")).collect(Collectors.toList());
		Collections.reverse(lines);
		String foundJson = null;
		for (String line : lines) {
			String matchJson = RegexUtil.getMatch("tmplmsg:(.*)", line, 1);
			if(matchJson!=null){
				foundJson = matchJson;
				break;
			}
		}
		//TODO 要怎么识别当前token没有登录呢？
		return Optional.ofNullable(foundJson).map(Document::fromJson);
		
	}
	
	
}
