package com.mini.framework.third.weixin.wxpay.v3.client;

import com.mini.framework.core.exception.ServerException;
import com.mini.framework.core.exception.standard.CustomException;
import com.mini.framework.third.weixin.wxpay.util.FeignUtil;
import com.mini.framework.third.weixin.wxpay.util.cache.MemoryBeanCache;
import com.mini.framework.third.weixin.wxpay.util.certificate.WxpayMarketCertificateContainer;
import com.mini.framework.third.weixin.wxpay.util.decode.WxpayResponseGsonDecoder;
import com.mini.framework.third.weixin.wxpay.v3.exception.WxpayBadRequestException;
import com.mini.framework.third.weixin.wxpay.v3.exception.WxpayDailyOrderNotExistException;
import com.mini.framework.third.weixin.wxpay.v3.exception.WxpayThirdException;
import com.mini.framework.third.weixin.wxpay.v3.model.weixin.AccountResultFileResponse;
import com.mini.framework.third.weixin.wxpay.v3.model.weixin.Html5CreatePayOrderResponse;
import com.mini.framework.third.weixin.wxpay.v3.model.weixin.JsApiCreatePayOrderResponse;
import com.mini.framework.third.weixin.wxpay.v3.model.weixin.TradeOrderHistoryPackage;
import com.mini.framework.third.weixin.wxpay.v3.model.weixin.WxpayCreatePayOrderRequest;
import com.mini.framework.third.weixin.wxpay.v3.model.weixin.WxpayQueryPayOrderResponse;
import com.mini.framework.third.weixin.wxpay.v3.model.weixin.WxpayRefundPayOrderRequest;
import com.mini.framework.third.weixin.wxpay.v3.model.weixin.WxpayRefundPayOrderResponse;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.date.DateUtil;
import com.mini.framework.util.date.TimeSectionType;
import com.mini.framework.util.log.MiniLogLevel;
import com.mini.framework.util.string.GsonUtil;
import feign.Feign;
import feign.RequestTemplate;
import feign.codec.Encoder;
import feign.codec.ErrorDecoder;
import feign.gson.GsonEncoder;
import feign.jaxrs.JAXRSContract;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.QueryParam;
import java.lang.reflect.Type;
import java.util.Calendar;
import java.util.Date;
import java.util.Optional;
import java.util.function.Supplier;

/**
 * https://www.jianshu.com/p/3d597e9d2d67
 * @author jayheo
 */
@Consumes("application/json")
public interface MarketDirectFeignHttpClient {

    ResponseHeaderResolver headerResolver = new ResponseHeaderResolver();

    Logger logger = LogManager.getLogger(MarketDirectFeignHttpClient.class);

    /**
     * 应用配置，
     */
    default void applyConfigIfRequire(ClientConfigAttach clientConfigAttach){
        throw new ServerException("子类必须实现这个方法");
    }

    MemoryBeanCache<MarketDirectFeignHttpClient> defaultCache = MemoryBeanCache.createMemory(100);


    /**
     * 创建或者从默认缓存中拿
     * 如果担心默认缓存不好用，可以直接自己弄一个缓存。
     * @param host
     * @param key
     * @param certificateContainerSupplier
     * @return
     */
    static MarketDirectFeignHttpClient createOrFromDefaultCache(String host, String key
            , Supplier<WxpayMarketCertificateContainer> certificateContainerSupplier){
        return defaultCache.get(key,()-> create(host, certificateContainerSupplier.get()));
    }



    static MarketDirectFeignHttpClient create(String host, WxpayMarketCertificateContainer certificateContainer){
        MarketDirectFeignHttpClient client = Feign.builder()
                .contract(new JAXRSContract())
                .logLevel(feign.Logger.Level.FULL)
                .logger(logger())
                .requestInterceptor(new SignatureRequestInterceptor(certificateContainer))
                .errorDecoder(errorDecoder())
                .encoder(encoder())
                .decoder(new WxpayResponseGsonDecoder(headerResolver))
                .target(MarketDirectFeignHttpClient.class,host);
        return new MarketDirectFeignHttpClientDelegate(client,certificateContainer);
    }

    static Encoder encoder() {
        return new GsonEncoder(GsonUtil.buildMilliSecondDateGson()){
            @Override
            public void encode(Object object, Type bodyType, RequestTemplate template) {
                if(object instanceof ParamBeanValidate){
                    ((ParamBeanValidate) object).beforeRequestValidate();
                }
                super.encode(object, bodyType, template);
            }
        };
    }


    static ErrorDecoder errorDecoder() {
        return (methodKey, response) -> {
            headerResolver.resolveHeaders(response.headers());
            Optional<String> bodyOptional = FeignUtil.findResponseBody(response);
            return bodyOptional.map(ErrorResponse::fromJson)
                    .map(errorResponse-> {
                        switch (errorResponse.getStatus()){
                            // 查询某天数据时，如果那天没有数据，会抛出错误，所以需要单独处理
                            case "NO_STATEMENT_EXIST":
                                return new WxpayDailyOrderNotExistException("当前时间没有对应的订单");
                            case "PARAM_ERROR":
                                return new WxpayDailyOrderNotExistException("参数有错误");
                            default:
                                return new WxpayBadRequestException(errorResponse,"微信服务端发现请求错误码:%s,错误信息:%s",errorResponse.getStatus(),errorResponse.getMessage());
                        }
                    })
                    .map(e->(CustomException)e)
                    .orElseGet(()->new WxpayThirdException("微信支付 %s 中出现未知异常:%s,%s",methodKey,response.status(),response.reason()));
        };
        
    }

    static feign.Logger logger() {
        return new feign.Logger() {
            @Override
            protected void log(String configKey, String format, Object... args) {
                logger.log(MiniLogLevel.getFrameWorkLog(),"微信支付v3:" + String.format(format, args));
            }
        };
    }



    //------------------------------------------------------------------


    /**
     * 显示当前的商户key
     * @return
     */
    default String showCurrentHadMarketKey(){
        //IllegalStateException: Method showCurrentHadMarketKey not annotated with HTTP method type (ex. GET, POST)
        throw new ServerException("子类必须实现这个方法");
    }


    //---------------------------公用接口。---------------------------


    /**
     * 查询支付订单通过商家的订单号
     * @param marketKey
     * @param siteOrderNumber
     * @return
     */
    @GET
    @Path("/v3/pay/transactions/out-trade-no/{siteOrderNumber}")
    WxpayQueryPayOrderResponse queryPayOrderBySiteOrder(@QueryParam("mchid") String marketKey, @PathParam("siteOrderNumber") String siteOrderNumber);



    /**
     * 查询支付订单通过微信侧的订单号
     * @param marketKey
     * @param wxpayOrderNumber
     * @return
     */
    @GET
    @Path("/v3/pay/transactions/id/{wxpayOrderNumber}")
    WxpayQueryPayOrderResponse queryPayOrderByWxpayOrder(@QueryParam("mchid") String marketKey, @PathParam("wxpayOrderNumber") String wxpayOrderNumber);

    /**
     * 关闭支付订单通过微信侧的订单号
     * @param marketKey
     * @param siteOrderNumber
     * @return
     */
    @GET
    @Path("/v3/pay/transactions/out-trade-no/{siteOrderNumber}/close")
    WxpayQueryPayOrderResponse closePayOrderBySiteOrder(@QueryParam("mchid") String marketKey, @PathParam("siteOrderNumber") String siteOrderNumber);


    /**
     * 执行退款
     * @param request
     * @return
     */
    @POST
    @Path("/v3/refund/domestic/refunds")
    WxpayRefundPayOrderResponse executeRefundPayOrder(WxpayRefundPayOrderRequest request);


    /**
     * 查询单笔退款API
     * 根据商家端的订单号
     * @param siteRefundOrderNumber
     * @return
     */
    @GET
    @Path("/v3/refund/domestic/refunds/{siteRefundOrderNumber}")
    WxpayRefundPayOrderResponse queryRefundPayOrderBySiteOrder(@PathParam("siteRefundOrderNumber") String siteRefundOrderNumber);


    /**
     * https://pay.weixin.qq.com/wiki/doc/apiv3/apis/chapter3_3_6.shtml
     * 申请交易账单API
     * @param day
     * @param tradeType
     * @return
     */
    @GET
    @Path("/v3/bill/tradebill")
    AccountResultFileResponse queryDailyHistoryOrdersFile(@QueryParam("bill_date")String day, @QueryParam("bill_type")String tradeType) throws WxpayDailyOrderNotExistException;


    /**
     * TODO 还差一点点，应该转成文件，然后解析文件。
     * 下载账单文件
     * @param path
     * @return
     */
    @GET
    @Path("/{path}")
    byte[] downloadByteArrayResultFileByTokenTypePath(@PathParam("path") String path, @QueryParam("token") String token);


    /**
     * 从交易结果中解析出内容
     * @param response
     * @return
     */
    default TradeOrderHistoryPackage readTradeResultContentByTokenTypePath(AccountResultFileResponse response){
        String path = response.getPathOfResourceUrl();
        path = path.substring(1);
        byte[] fileBytes = this.downloadByteArrayResultFileByTokenTypePath(path
                , response.queryTokenOfResourceUrl().orElseThrow(()->new ServerException("找不到token")));
        String content = new String(fileBytes);

        return TradeOrderHistoryPackage.fromOriginContent(content);

    }


    /**
     * 查询某一天有支付订单列表文件
     * @param dayDate
     * @return
     * @throws WxpayDailyOrderNotExistException
     */
    default AccountResultFileResponse queryDailyPayOrdersFile(Date dayDate) throws WxpayDailyOrderNotExistException {
        AssertUtil.assertMethodRequire(dayDate,"dayDate");
        validateHistoryOrdersParamsDay(dayDate);
        String dayString = DateFormatUtils.format(dayDate,"yyyy-MM-dd");
        return queryDailyHistoryOrdersFile(dayString,"SUCCESS");
    }

    /**
     * 查询某一天有退款订单列表文件
     * @param dayDate
     * @return
     * @throws WxpayDailyOrderNotExistException
     */
    default AccountResultFileResponse queryDailyRefundOrdersFile(Date dayDate) throws WxpayDailyOrderNotExistException {
        AssertUtil.assertMethodRequire(dayDate,"dayDate");
        validateHistoryOrdersParamsDay(dayDate);
        String dayString = DateFormatUtils.format(dayDate,"yyyy-MM-dd");
        return queryDailyHistoryOrdersFile(dayString,"REFUND");
    }


    /**
     * 验证历史订单参数之日期
     * @param dayDate
     */
    static void validateHistoryOrdersParamsDay(Date dayDate){
        Date now = new Date();
        Date nowDay = DateUtil.getCleanedDateUnit(TimeSectionType.day,now);
        Date minDayDate = DateUtil.addTimeUnit(TimeSectionType.day, nowDay, -90);
        Date maxDayDate = DateUtil.addTimeUnit(TimeSectionType.day, nowDay, -1);
        if(!minDayDate.before(dayDate)){
            //检查只允许查三个月内的
            throw new WxpayThirdException("查询时间:%s不能早于当前以前90天:%s",dayDate.toInstant(),minDayDate.toInstant());
        }

        if(!maxDayDate.after(dayDate)){
            //TODO 以前要换成3点以后可以查询昨天的。3点以前可以查询前天的。
            throw new WxpayThirdException("查询时间:%s不能迟于当前以前2天:%s",dayDate.toInstant(),maxDayDate.toInstant());
        }
    }




    /**
     * 查询某一天有支付订单包
     * @param dayDate
     * @return
     */
    default TradeOrderHistoryPackage queryDailyPayOrdersPackage(Date dayDate) {
        try{
            AccountResultFileResponse fileResponse = queryDailyPayOrdersFile(dayDate);
            return readTradeResultContentByTokenTypePath(fileResponse);
        }catch (WxpayDailyOrderNotExistException e){
            logger.warn("发现一个异常:没有找到对接订单列表",e);
            return TradeOrderHistoryPackage.createEmpty();
        }
    }

    /**
     * 查询某一天有退款订单包
     * @param dayDate
     * @return
     */
    default TradeOrderHistoryPackage queryDailyRefundOrdersPackage(Date dayDate){
        try{
            AccountResultFileResponse fileResponse = queryDailyRefundOrdersFile(dayDate);
            return readTradeResultContentByTokenTypePath(fileResponse);
        }catch (WxpayDailyOrderNotExistException e){
            logger.warn("发现一个异常:没有找到对接订单列表",e);
            return TradeOrderHistoryPackage.createEmpty();
        }
    }



    //--------------------------------html5的接口。---------------------------------------


    /**
     * H5下单API
     * https://pay.weixin.qq.com/wiki/doc/apiv3/apis/chapter3_3_1.shtml
     * @param request
     */
    @POST
    @Path("/v3/pay/transactions/h5")
    Html5CreatePayOrderResponse createHtml5PayOrder(WxpayCreatePayOrderRequest request);






    //----------------------------JSAPI的接口-----------------------------

    /**
     * JSAPI下单
     * https://pay.weixin.qq.com/wiki/doc/apiv3/apis/chapter3_1_1.shtml
     * @param request
     */
    @POST
    @Path("/v3/pay/transactions/jsapi")
    JsApiCreatePayOrderResponse createJsApiPayOrder(WxpayCreatePayOrderRequest request);


}
