package com.mini.framework.util.cache;

import com.mini.framework.core.exception.ServerException;
import com.mini.framework.core.exception.standard.CustomException;
import com.mini.framework.core.status.Status;

import java.util.Date;
import java.util.function.Supplier;


/**
 * 对象变量缓存
 * @param <O>
 */
public class ObjectVariableCache<O> {

    private final long expireMillis;
    private final CustomException unFoundException;

    final private Supplier<O> getter;

    private Date expireDate;

    private volatile O variable;



    private ObjectVariableCache(Supplier<O> getter, long expireMillis, CustomException unFoundException) {
        this.getter = getter;
        this.expireMillis = expireMillis;
        this.unFoundException = unFoundException;
    }

    public static <T> ObjectVariableCache<T> createInstance(long expireMillis,Supplier<T> getter){
        return createInstance(expireMillis,getter,new ServerException("没有定义出错时的错误"));
    }


    public static <T> ObjectVariableCache<T> createInstance(long expireMillis,Supplier<T> getter, CustomException unFoundException){
        if(unFoundException==null) {
            throw new ServerException(Status.Server.fatalBug, "必须定义异常");
        }
        return new ObjectVariableCache<T>(getter,expireMillis,unFoundException);
    }


    /**
     * 获取变量 同步的方式。这样防止最多次创建。
     * @return 缓存的值。
     */
    public synchronized O getVariableSync(){
        return getVariable();
    }


    /**
     * 获取变量
     * @return 缓存的值
     */
    public O getVariable(){
        Date now = new Date();
        if(expireDate!=null && expireDate.after(now) && variable!=null){
            return variable;
        }else{
            O newVariable = getter.get();
            if(newVariable==null){
                throw unFoundException;
            }
            //只缓存一个变量没有什么大影响，如果大对象要注意，定时清理。
            variable = newVariable;
            expireDate = new Date(System.currentTimeMillis() + expireMillis);
            return newVariable;
        }
    }
}
