package com.mini.framework.util.code;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;


/**
 * 映射编码的工具。
 * <br/>
 * 把一个数字映射成一个代码。这个代码有以下特征 。
 * <pre>
 *     1，看不出顺序
 *     2，一一对应
 *     3，可以设置防止猜测，即 N 个数中，只有一个是正确的，
 * </pre>
 * 一对一转化逻辑。(他是可逆转的)
 * <pre>
 *     1，把数字转成进制数
 *     731        -> 1011011011
 *     2，把进制数反排
 *     1011011011 -> 1101101101
 *     3, 把逆排转成数字
 *     1101101101 -> 877
 *     4，把数字进行移位
 *     877        ->  max % (877 + (slot + max/2))
 * </pre>
 * <pre>
 * 经过检查，当前工具没有做防猜测的功能。
 * </pre>
 */
public class MapperCodeUtil {

	public static final String chars62 = "oXrWOtAQgV6mzcaPG2LHdb4xJ7spFnShRk0q8iuYDlvKCZMyETIfwBUj3e1N95"; 
	public static final String chars36 = "ortg6mzca2db4x7spnhk0q8iulvyfwj3e195";

    //<editor-fold desc="配置参数">


	/**
	 * 重复的程度 
	 */
	public final int repeatDegree;
	
	/**
	 * 加盐值，等于是干扰值，不同的干扰值得到不同
	 */
	public final long salt;
	
	/**
	 * 目标结果的长度
	 */
	public final int targetLength;
	
	/**
	 * 可用的字符
	 */
	public final String charItems;
	
	public final int useRadix;



	//</editor-fold>


	//<editor-fold desc="内部使用的参数">

	
	
	/**
	 * 最大会使用，它不能大于字符串容量
	 * 也就是当前可用的容量
	 */
	private Long useMax;
	
	/**
	 * 计划使用的位数
	 */
	private Integer useOffset;

	/**
	 * 重复的次数
	 */
	private Integer repeatTime;

	/**
	 * 每次偏移的位数
	 */
	private Long shiftLength;


	//</editor-fold>

	public Long getUseMax() {
		return useMax;
	}

	public Integer getUseOffset() {
		return useOffset;
	}

	public Integer getRepeatDegree() {
		return repeatDegree;
	}

	public Integer getRepeatTime() {
		return repeatTime;
	}

	/**
	 * @see #createSimpleBinary(int, int)
	 * @param targetLength 目标结果的长度
	 * @param salt 盐值
	 * @return 映射工具
	 */
	@Deprecated
	public static MapperCodeUtil createSimpleBinaray( int targetLength,int salt){
		return createSimpleBinary(targetLength, salt);
	}
	/**
	 * 创建一个简单的62位映射工具。
	 * @param targetLength 目标结果的长度
	 * @param salt 盐值
	 * @return 映射工具
	 */
	public static MapperCodeUtil createSimpleBinary( int targetLength,int salt){
		return create(targetLength, chars62, 2, 10,salt);
	}


	/**
	 * 创建一个简单的36位映射工具
	 * @param targetLength 目标结果的长度
	 * @param salt 盐值 不同的盐值会得到不同的结果
	 * @return 映射工具
	 */
	public static MapperCodeUtil createSimple36Binary( int targetLength,int salt){
		return create(targetLength, chars36, 2, 10,salt);
	}


	/**
	 * 创建一个映射工具
	 * @param targetLength 目标结果的长度
	 * @param charItems 接受的字段。
	 * @param useRadix 过程中使用进制(进制越大，有一定的浪费)
	 * @param repeatDegree 重试程度。
	 * @param salt 盐值，干扰值。
	 * @return 映射工具。
	 */
	public static MapperCodeUtil create( int targetLength, String charItems,
			int useRadix,int repeatDegree,long salt) {
		return new MapperCodeUtil(targetLength, charItems, useRadix, repeatDegree,salt);
	}

	/**
	 * 创建一个32位的映射工具。
	 * @param targetLength 代码的长度
	 * @param useRadix 进制
	 * @param repeatDegree 重复的程度
	 * @param salt 干扰值
	 * @return 映射工具
	 */
	public static MapperCodeUtil createChar36(int targetLength, int useRadix, int repeatDegree, long salt){
		return new MapperCodeUtil(targetLength, chars36, useRadix, repeatDegree, salt);
	}

	
	private MapperCodeUtil( int targetLength, String charItems,
			int useRadix,int repeatDegree,long salt) {
		super();
		this.repeatDegree = repeatDegree;
		this.targetLength = targetLength;
		this.charItems = charItems;
		this.useRadix = useRadix;
		this.salt = salt;
		this.init();
	}
	
	private void init(){
		//计算出整个字符串的容量
		
		AssertUtil.assertNoBadReq(useRadix>1, Status.BadReq.illParam, "useRadix:[%s]必须大于1",useRadix);

		//容量
		long charCapacity = longPow(this.charItems.length(), this.targetLength);
		//<editor-fold desc="计算出偏移量">
		int offset = 0;
		while(longPow(this.useRadix, offset)<=charCapacity){
			offset++;
		}
		this.useOffset = offset - 1;
		//</editor-fold>
		this.repeatTime = primeNumber(repeatDegree);
		this.useMax = longPow(this.useRadix, this.useOffset);
		this.shiftLength = salt + useMax /2;
	}
	

	public String encode(long origin){
		AssertUtil.assertNoBadReq(origin<useMax, Status.BadReq.hitLimit, "目标:%s超过了最大限制:%s",origin,useMax);
		for (int index = 0; index < repeatTime; index++) {
			origin = stepCreate(origin);
		}
		return mapperCode(origin);
	}
	
	public long decode(String text){
		AssertUtil.assertMethodRequire(text, "text");
		AssertUtil.assertNoBadReq(text.length()==targetLength, Status.BadReq.noReq, "text:[%s]长度:[%s]与当前长度不符",text,targetLength);
		List<Integer> values = fromText(text);
		for (Integer value : values) {
			AssertUtil.assertNoBadReq(value>=0, Status.BadReq.illParam, "text[%s]中有字符不在[%s]中",text,charItems);
		}
		
		long origin = fromRadixOffsetValues(values, charItems.length());
		
		for (int index = 0; index < repeatTime; index++) {
			origin = stepCreateRevoke(origin);
		}
		return origin;
		
	}
	
	private List<Integer> fromText(String text){
		List<Integer> values = new ArrayList<>();
		for (int index = 0; index < text.length(); index++) {
			values.add(charItems.indexOf(text.charAt(index)));
		}
		return values;
	}
	
	
	protected long stepCreateRevoke(long origin){
		long result = shiftRevoke(origin, shiftLength);
		result = reverseAsRadix(result,useRadix,useOffset);
		return result;
	}
	
	protected long stepCreate(long origin){
		long result = reverseAsRadix(origin,useRadix,useOffset);
		result = shift(result, shiftLength);
		return result;
	}
	
	
	private String mapperCode(long origin){
		return mapperCode( origin,charItems);
	}
	
	private String mapperCode(long origin,String chars){
		List<Integer> offsetValues = toRadixOffsetValues(origin,chars.length());
		return offsetValues.stream().map(v->chars.charAt(v)+"").collect(Collectors.joining(""));
	}
	
	/**
	 * 使用2进制返回序
	 * @return 逆序后的结果
	 */
	protected long reverseAsRadix(long origin,int radix,int offset){
		List<Integer> values = toRadixOffsetValues(origin, radix);
		while(values.size()<offset){
			// 用0去填充
			values.add(0,0);
		}
		Collections.reverse(values);
		return fromRadixOffsetValues(values, radix);
	}
	protected long reverseAsRadixRevoke(long origin,int radix,int offset){
		return reverseAsRadix(origin, radix, offset);
	}

	/**
	 * 按 radix 进制得到算出数字表达式列表。
	 * @param origin 目标数
	 * @param radix 进制
	 * @return 数字表达式列表
	 */
	private List<Integer> toRadixOffsetValues(final long origin,final int radix){
		long tempOrigin = origin;
		List<Integer> offsetValues = new ArrayList<>();
		while(radix<=tempOrigin){
			offsetValues.add((int)(tempOrigin%radix));
			tempOrigin/=radix;
		}
		offsetValues.add((int)tempOrigin);
		while(offsetValues.size()<targetLength){
			offsetValues.add(0);
		}
		Collections.reverse(offsetValues);
		return offsetValues;
	}
	
	
	private long fromRadixOffsetValues(List<Integer>offsetValues,int radix){
		long value = 0;
		
		for (int index = 0; index < offsetValues.size(); index++) {
			int offset = offsetValues.size()- index - 1;
			value += offsetValues.get(index) * longPow(radix, offset);
		}
		return value;
	}

	/**
	 * 计算指数
	 * @param base 进制数
	 * @param times 次数/指数
	 * @return 结果
	 */
	private long longPow(int base,int times){
		long result = 1;
		for (int index = 0; index < times; index++) {
			result *= base;
		}
		double doublePow = Math.pow(base, times);
		//TODO 断言不应该放到这里。 风险还没有接触。
		AssertUtil.assertNotFatalBug(doublePow<Long.MAX_VALUE,"long型求指数发现base(%s)和index(%s)结果(%s)大于最大的long(%s)",base,times ,doublePow,Long.MAX_VALUE);
		AssertUtil.assertNotFatalBug(result>doublePow-10,"long型求指数发现base(%s)和index(%s)不满足 longPow(%s)>doublePow(%s)-10",base,times,result ,doublePow);
		return result;
	}
	
	/**
	 * 偏移一个指定的量
	 * @param offset 起始的值
	 * @param length 偏移的长度
	 * @return 结果
	 */
	private long shift(long offset,long length){
		return toRight(offset + length)%useMax;
	}
	
	/**
	 * 撤销(反向)偏移一个指定的量
	 * @param offset 起始的值
	 * @param length 偏移的长度
	 * @return 结果
	 */
	private long shiftRevoke(long offset,long length){
		return toRight((offset - length))%useMax;
	}

	/**
	 * 转换为正确的数
	 * @param input 输入的数值
	 * @return 转换后的结果
	 */
	private long toRight(long input){
		while(input<0){
			input+=useMax;
		}
		return (input)%useMax;
	}

	public void reverseAsRadixTest(long input) {
		System.out.println(input);
		long result = reverseAsRadix(input, useRadix, useOffset);
		System.out.println(result);
		long result1 = reverseAsRadixRevoke(result, useRadix, useOffset);
		System.out.println(result1);
		
	}
	
	/**
	 * 返回一个合适的质素
	 * @param index 索引第几个
	 * @return 质数结果
	 */
	private int primeNumber(int index){
		int[] primes = new int[]{5,7,11,13,17,19,23,29,31,37,41,43,47,53,59,61,67,71,73,79,83,87,97};
		AssertUtil.assertNoBadReq(index>=0, Status.BadReq.hitLimit, "index:[%s]不能小少0",index);
		AssertUtil.assertNoBadReq(index<primes.length, Status.BadReq.hitLimit, "index:[%s]必须小于可选质数数",primes.length);
		return primes[index];
	}
	
}
