package com.mini.framework.util.code;

import org.apache.commons.lang3.builder.ToStringBuilder;

import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;

/**
 * 将一个唯一进分区。
 * @author jayheo
 *
 */
public class PartitionCodeUtil {

	/**
	 * 猜测干扰
	 */
	public final int guessDisturbTime;
	
	/**
	 * 分区个数
	 */
	public final int partitionNumber;
	
	public final MapperCodeUtil mapperCodeUtil;

	/**
	 * 使用范围。
	 */
	private Integer useMax;
	
	
	
	private PartitionCodeUtil(int guessDisturbTime, int partitionNumber, MapperCodeUtil mapperCodeUtil) {
		super();
		this.guessDisturbTime = guessDisturbTime;
		this.partitionNumber = partitionNumber;
		this.mapperCodeUtil = mapperCodeUtil;
		init();
	}

	public static PartitionCodeUtil create(int guessDisturbTime,int partitionNumber, int targetLength, String charItems,
			int useRadix,int repeatDegree,long salt) {
		return new PartitionCodeUtil(guessDisturbTime, partitionNumber, MapperCodeUtil.create(targetLength, charItems, useRadix, repeatDegree, salt));
	}
	
	/**
	 * @see #createSimpleBinaray(int, int, int, int)
	 * @return 分区的分区码
	 */
	@Deprecated
	public static PartitionCodeUtil createSimpleBinaray(int guessDisturbTime,int partitionNumber, int targetLength,int salt) {
		return createSimpleBinary(guessDisturbTime, partitionNumber, targetLength, salt);
	}	
	/**
	 * 生成一个简单的二进制分区码
	 * @param guessDisturbTime 猜测难度，越大越好，但也容量越小，
	 * @param partitionNumber 分多少个区。
	 * @param targetLength 计算出来的code的目标长度。
	 * @param salt 盐值用来保证不一样。
	 * @return 分区的分区码
	 */
	public static PartitionCodeUtil createSimpleBinary(int guessDisturbTime,int partitionNumber, int targetLength,int salt) {
		return new PartitionCodeUtil(guessDisturbTime, partitionNumber,MapperCodeUtil.createSimpleBinary(targetLength,salt));
	}
	
	private void init(){
		long useMaxLong = (mapperCodeUtil.getUseMax()/guessDisturbTime/partitionNumber);
		AssertUtil.assertNoBadReq(useMaxLong<=Integer.MAX_VALUE, Status.BadReq.hitLimit, "useMaxLong:[%s]超过integer最大限制",useMaxLong);
		this.useMax = (int) useMaxLong;
	}
	
	
	/**
	 * @param partitionIndex  分区编号
	 * @param input 入参
	 * @return 分区编码
	 */
	public String encode(int partitionIndex,int input){
		AssertUtil.assertNoBadReq(partitionIndex>=0, Status.BadReq.illParam, "partitionIndex:[%s]必须大于0",partitionNumber);
		AssertUtil.assertNoBadReq(partitionIndex<partitionNumber, Status.BadReq.illParam, "partitionIndex:[%s]必须小于partitionNumber:[%s]",partitionIndex,partitionNumber);
		AssertUtil.assertNoBadReq(input<this.useMax, Status.BadReq.illParam, "input:[%s]必须小于useMax:[%s]",input,useMax);
		long realInput = (long)input * partitionNumber + partitionIndex;
		realInput*=guessDisturbTime;
		return mapperCodeUtil.encode(realInput);
	}
	public PartitionCode decode(String text){
		long result = mapperCodeUtil.decode(text);
		AssertUtil.assertNoBadReq(result%guessDisturbTime==0,Status.BadReq.illParam,"编码:[%s]不是正常的编码",text);
		result/=guessDisturbTime;
		long index = result/partitionNumber;
		//TODO断言结果在不在范围内
		long partition = result%partitionNumber;
		return PartitionCode.createByIndexValue((int)partition,(int) index);
	}

	
	public Integer getUseMax() {
		return useMax;
	}


	public static class PartitionCode{
		/**
		 * 分区的编号
		 */
		private Integer partition;
		
		private String code;
		
		/**
		 * 对应的数值
		 */
		private Integer index;
		
		private PartitionCode(Integer partition, String code, Integer index) {
			super();
			this.partition = partition;
			this.code = code;
			this.index = index;
		}

		public static PartitionCode createByCode(String code){
			return new PartitionCode(null,code,null);
		}
		
		public static PartitionCode createByIndexValue(int partition,int index){
			return new PartitionCode(partition, null, index);
		}
		
		public PartitionCode encode(PartitionCodeUtil util){
			this.code = util.encode(partition,index);
			return this;
			
		}
		
		public PartitionCode decode(PartitionCodeUtil util){
			PartitionCode result = util.decode(this.code);
			this.index = result.index;
			this.partition = result.partition;
			return this;
		}
		
		
		
	public Integer getPartition() {
			return partition;
		}

		public String getCode() {
			return code;
		}

		public Integer getIndex() {
			return index;
		}

	@Override
	public String toString() {
		return ToStringBuilder.reflectionToString(this);
	}	
		
	}
	
	
	
}
