package com.mini.framework.util.date;


import java.util.Arrays;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Stream;

import com.mini.framework.core.exception.ServerException;
import org.apache.commons.lang3.time.DateFormatUtils;

import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.type.EnumTypeShower;

/**
 * 日期范围
 * 比如一周的7天
 * @author jayheo
 *
 */
public enum DateScopeItems implements EnumTypeShower<DateScopeItems>{

	
	//TODO 刘仕强 把这里文档补全一下（已补充）
	allOfAll(TimeSectionType.all,TimeSectionType.all,"有史以来","有史以来"),
	yearOfAll(TimeSectionType.all,TimeSectionType.year,"yyyy年","每年"),
	seasonOfAll(TimeSectionType.all,TimeSectionType.season,null,"每季"),
	monthOfAll(TimeSectionType.all,TimeSectionType.month,"yyyy年MM月","每月"),
	weekOfAll(TimeSectionType.all,TimeSectionType.week,null,"每周"),//由于年没有完全覆盖周，所以首周不会出现尾周又会多了些数据
	dayOfAll(TimeSectionType.all,TimeSectionType.day,"yyyy年MM月dd日","每天"),
	hourOfAll(TimeSectionType.all,TimeSectionType.hour,"yyyy年MM月dd日HH点","每小时"),
	
	seasonOfYear(TimeSectionType.year,TimeSectionType.season,null,"年中各个季节"),
	monthOfYear(TimeSectionType.year,TimeSectionType.month,"MM月","年中各个月"),
	weekOfYear(TimeSectionType.year,TimeSectionType.week,null,"年中各周"),//由于年没有完全覆盖周，所以首周不会出现尾周又会多了些数据
	dayOfYear(TimeSectionType.year,TimeSectionType.day,"MM月dd日","年中各天"),
	hourOfYear(TimeSectionType.year,TimeSectionType.hour,"MM月dd日HH点","年中各个小时"),

	monthOfSeason(TimeSectionType.season,TimeSectionType.month,null,"季中各个月"),//由于年没有完全覆盖周，所以首周不会出现尾周又会多了些数据
	weekOfSeason(TimeSectionType.season,TimeSectionType.week,null,"季中各周"),//由于年没有完全覆盖周，所以首周不会出现尾周又会多了些数据
	dayOfSeason(TimeSectionType.season,TimeSectionType.day,null,"季中各天"),
	hourOfSeason(TimeSectionType.season,TimeSectionType.hour,null,"季中各个小时"),

	weekOfMonth(TimeSectionType.month,TimeSectionType.week,null,"月中各周"),//由于年没有完全覆盖周，所以首周不会出现尾周又会多了些数据
	dayOfMonth(TimeSectionType.month,TimeSectionType.day,"dd号","月中各天"),
	hourOfMonth(TimeSectionType.month,TimeSectionType.hour,"dd号HH点","月中各个小时"),

	
	
	
	dayOfWeek(TimeSectionType.week,TimeSectionType.day,null,"周中各天"),
	hourOfWeek(TimeSectionType.week,TimeSectionType.hour,null,"周中各个小时"),
	
	hourOfDay(TimeSectionType.day,TimeSectionType.hour,"HH点","天中各个小时"),
	;
	/**
	 * 组类型
	 */
	private TimeSectionType scope;
	/**
	 * 组元素
	 */
	private TimeSectionType item;
	
	private String desc;
	
	private String simpleItemTitleFormat;
	
	DateScopeItems(TimeSectionType scope, TimeSectionType item, String simpleNameFormat, String desc) {
		this.scope = scope;
		this.item = item;
		this.simpleItemTitleFormat = simpleNameFormat;
		this.desc = desc;
	}

	public TimeSectionType item() {
		return item;
	}

	public TimeSectionType scope() {
		return scope;
	}
	
	public String desc() {
		return desc;
	}

	public String simpleItemTitleFormat() {
		return simpleItemTitleFormat;
	}
	private static Map<String, DateScopeItems> createKeyMap() {
		
		Map<String, DateScopeItems> map = new ConcurrentHashMap<>();
		
		for (DateScopeItems pair : DateScopeItems.values()) {
			map.put(createKey(pair.scope, pair.item), pair);
		}
		
		return map ;
	}
	
	public static Map<String, DateScopeItems> keyMap = createKeyMap();
	
	private static String createKey(TimeSectionType scope, TimeSectionType item){
		AssertUtil.assertMethodRequire(scope, "scope");
		AssertUtil.assertMethodRequire(item, "item");
		return item + " in " + scope ;
	}
	
	public static DateScopeItems verifyPair(TimeSectionType scope, TimeSectionType item){
		DateScopeItems pair = matchPair(scope, item);
		AssertUtil.assertNoBadReq(pair!=null, Status.BadReq.unsupport,"不支持的分组日期:[%s]目前支持的分组日期是:%s",createKey(scope, item),Arrays.toString(values()));
		return pair;
	}


	public static DateScopeItems fromScopeItem(TimeSectionType scope, TimeSectionType item){
		return Stream.of(values())
				.filter(bean->bean.matchItem(item))
				.filter(bean->bean.matchScope(scope))
				.findFirst().orElseThrow(()->new ServerException("scope:[%s],item:[%s]没有匹配的DateScopeItems",scope,item));
	}

	public boolean matchScope(TimeSectionType scope){
		return this.scope == scope;
	}


	public boolean matchItem(TimeSectionType item){
		return this.item == item;
	}

	
	public static DateScopeItems matchPair(TimeSectionType scope, TimeSectionType item){
		return keyMap.get(createKey(scope, item));
	}
	
	/**
	 * 计算出简易时间item
	 * @param pointDate
	 * @return
	 */
	public String countSimpleItemTitle(Date pointDate){
		if(simpleItemTitleFormat !=null){
			return DateFormatUtils.format(pointDate, simpleItemTitleFormat);
		}else{
			DateScopeItems type = this;
			switch(type){
			case seasonOfAll:
				return DateFormatUtils.format(pointDate, "yy年") + DateUtil. indexSeasonNameQsort(pointDate);
			case weekOfYear:
				return String.format("第%s周", DateUtil.indexWeekOfYear(pointDate));
			case seasonOfYear:
				return DateUtil. indexSeasonNameQsort(pointDate);
			case weekOfMonth:
				return String.format("第%s周",DateUtil. indexWeekOfMonth(pointDate));
			case dayOfWeek:
				return DateUtil. indexDayOfWeekSimpleName(pointDate);
			case hourOfWeek:
				return hourOfDay.countSimpleItemTitle(pointDate) +"-"+ dayOfWeek.countSimpleItemTitle(pointDate) ;

					
			//	case dayOfWeek:
				//	return DateUtil.weekIndexOfYearSimpleName(pointDate);
		//		case hourOfWeek:
		//			return DateUtil.weekIndexOfYearSimpleName(pointDate) + DateFormatUtils.format(pointDate, " HH点");
					default :
			}
			
			return name();
		}
	}
	

	@Override
	public void fillFieldToMap(Map<String, Object> map) {
		map.put("scope", scope);
		map.put("item", item);
		map.put("desc", desc);
		map.put("simpleItemTitleFormat", simpleItemTitleFormat);
	}

}
