package com.mini.framework.util.oss;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;

import com.mini.framework.core.exception.ServerException;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.aliyun.oss.OSSClient;
import com.aliyun.oss.common.utils.BinaryUtil;
import com.aliyun.oss.model.MatchMode;
import com.aliyun.oss.model.ObjectMetadata;
import com.aliyun.oss.model.PolicyConditions;
import com.aliyun.oss.model.PutObjectResult;
import com.mini.framework.core.exception.BadReqException;
import com.mini.framework.core.exception.HandleIOException;
import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.params.MapParams;
import com.mini.framework.util.string.GsonUtil;
import com.mini.framework.util.string.RegexUtil;
import com.mini.framework.util.thread.ObjectThreadCachable;


/**
 * @author jayheo
 * 对象存储辅助类
 * 包含文件上传，文件流上传
 */
public class OssHelper {
	protected static Logger logger = LogManager.getLogger(OssHelper.class.getName());

	private ObjectThreadCachable<String> domainGetter;
	
	
	protected OssClientConfig occ;
	private String serverHome;
	
	public OssHelper(OssClientConfig occ,String serverHome){
		this.occ = occ;
		this.serverHome = serverHome;
	}
	
	public String getServerHome() {
		return serverHome;
	}
	
	/**
	 * 把oss路径解析出来
	 * 把serverHome从url中拿走
	 * 如果不是完整的地址什么都不要做
	 * @return
	 */
	public Function<String, String> decomposeServerHomeTestRun(){
		return resUrl ->{
			AssertUtil.assertMethodRequire(resUrl, "resUrl");
			String namespace = RegexUtil.getUrlNamespace(resUrl);
			if(namespace==null){
				return resUrl;
			}else{
				AssertUtil.assertNoBadReq(namespace.length()>0, Status.BadReq.illParam, "resUrl:[%s]的命名空间不能太短",resUrl);
				return namespace.substring(1);
			}
		};
	}
	
	
	/**
	 * 把oss路径解析出来
	 * 把serverHome从url中拿走
	 * @return
	 */
	public Function<String, String> decomposeServerHome(){
		return resUrl ->{
			AssertUtil.assertMethodRequire(resUrl, "resUrl");
			String namespace = RegexUtil.getUrlNamespace(resUrl);
			AssertUtil.assertNoBadReq(namespace!=null, Status.BadReq.illParam,"resUrl:[%s]不是正常的url不能解析出命名空间",resUrl);
			//TODO 如果这个出现数组越界应该修改一下 还要检查是不是匹配这个oss域名
			AssertUtil.assertNoBadReq(namespace.length()>0, Status.BadReq.illParam, "resUrl:[%s]的命名空间不能太短",resUrl);
			return namespace.substring(1);
		};
	}
	
	public String assemServerHome(String path){
		AssertUtil.assertMethodRequire(path, "path");
		return serverHome + "/" + path;
	}
	
	/**
	 * 检查url中的domain是不是本域的domain
	 * @param url
	 * @return
	 */
	public boolean matchServerHome(String url){
		AssertUtil.assertMethodRequire(url, "url");
		return RegexUtil.getUrlDomain(serverHome).equals(RegexUtil.getUrlDomain(url));
	}
	
	/**
	 * 确保这个url是当前使用的域的
	 * 返回一个带ServerHome的可用的url
	 * @return
	 */
	public Function<String, String> sureMatchServerHome(){
		return (url)-> {
			AssertUtil.assertMethodRequire(url, "url");
			if(matchServerHome(url)){
				return url;
			}else{
				return assemServerHome(uploadRes(url));
			}
		};
	}
	
	
	public ObjectThreadCachable<String> getDomainGetter() {
		return domainGetter;
	}

	public void setDomainGetter(ObjectThreadCachable<String> domainGetter) {
		this.domainGetter = domainGetter;
	}

	/**
	 * 安静的关闭客户端,什么事也不做
	 * */
	private void quietShutdown(OSSClient client) {
		if(client!=null){
			try{
				client.shutdown();
			}catch(Throwable e){
				logger.debug("关闭oss客户端时出现异常",e);
			}
		}
	}
	
	/**
	 * 上传base64的文件
	 * @param base64
	 * @param extName
	 * @return
	 */
	public String uploadBase64(String base64,String extName){
		AssertUtil.assertSupport(base64 != null, "base64字符串不能为空");
		Base64 base64Encoder = new Base64();
		try {
			return upload(base64Encoder.decode(base64),extName);
		} catch (RuntimeException e) {
			logger.error("在处理base64转byte的时候出错[{}],base64:[{}]",e.getMessage(),base64);
			throw new HandleIOException(e,"处理base64字符串错误");
		}
	}
	
	

	/**
	 * 上传资源返回的时候有host
	 * @param urlString
	 * @return
	 */
	public String uploadResWithHost(String urlString){
		return assemServerHome(uploadRes(urlString));
	}

	/**
	 * 根据url上传，直接使用它的后缀
	 * @param urlString
	 * @return
	 */
	public String uploadRes(String urlString){
		AssertUtil.assertMethodRequire(urlString, "urlString");
		String extName = StringUtils.substringAfterLast(urlString, ".");
		extName = StringUtils.substringBefore(extName, "?");
		if(extName.contains("/")){//如果这样取还会有/那么就理解成没有后缀。
			logger.debug("从url中解析不出后缀url:[%s]",urlString);
			extName= "file";
		}
		AssertUtil.assertNoBadReq(extName.length()>=0,Status.BadReq.illParam, "url:[%s]没有后缀不能使用这个方法", urlString);
		return uploadRes(urlString, extName);
	}
	
	
	/**
	 * 上传资源并保留原来的参数
	 * @param urlString
	 * @return
	 */
	public String uploadResRetainParams(String urlString){
		String url = StringUtils.substringBefore(urlString, "?");
		String params = StringUtils.substringAfter(urlString, "?");
		String newUrl = uploadRes(url);
		if(StringUtils.contains(urlString, "?")){
			newUrl = newUrl + "?" + params;
		}
		return newUrl;
	}
	

	/**
	 * 如果抛出异常，则表明操作失败，否则操作成功。抛出异常时，方法返回的数据无效
	 * 路径自动生成，文件后缀名是从 extName 中截取最后一个点后面的字符串
	 * */
	public String uploadRes(String urlString,String extName){
		try{
			AssertUtil.assertSupport(urlString != null, "上传资源时url不能为空");
			return this.upload(new URL(urlString).openStream(), extName);
		}catch(FileNotFoundException e){
			//Caused by: java.io.FileNotFoundException: https://www.bilibili.com/6d122ad0-0d18-4ae5-8268-9bad277bf872
			throw new BadReqException(e,"请求找不到url:[%s]类似404",urlString);
		}catch(IOException e){
			// TODO Caused by: java.io.IOException: Server returned HTTP response code: 403 for URL: https://www.bilibili.com/video/av75781138
			//TODO 需要处理一下 403的问题，进一步要处理一下其它的错误码
			throw new HandleIOException(e,"请求第三方数据源时出错,url:[%s]",urlString);
		}
	}
	
	/**
	 * 如果抛出异常，则表明操作失败，否则操作成功。抛出异常时，方法返回的数据无效
	 * 路径自动生成，文件后缀名是从 extName 中截取最后一个点后面的字符串
	 * */
	public String upload(byte[] buf,String extName){
		return this.upload(new ByteArrayInputStream(buf), extName);
	}
	
	/**
	 * 如果抛出异常，则表明操作失败，否则操作成功。抛出异常时，方法返回的数据无效
	 * 路径自动生成，文件后缀名是从文件名中截取最后一个点后面的字符串
	 * */
	public String upload(File file){
		//String extName = StringUtils.substringAfterLast(file.getAbsolutePath(), ".");
		return this.upload(file, file.getName());
	}

	/**
	 * 如果抛出异常，则表明操作失败，否则操作成功。抛出异常时，方法返回的数据无效
	 * 路径自动生成，文件后缀名是从 extName 中截取最后一个点后面的字符串
	 * */
	public String upload(File file,String extName){
		String pathPrefix = parsePathPrefix();
		OSSClient client = null;
		try{
			String path = createObjectKey(pathPrefix,extName);
			client = new OSSClient(occ.autoEndpoint(), occ.getAccessKeyId(), occ.getAccessKeySecret());
			PutObjectResult result = client.putObject(occ.getBucketName(), path, file);
			this.markPutObjectResult(result);
			return path;
		}finally{
			quietShutdown(client);
		}
	}
	
	private void markPutObjectResult(PutObjectResult result){
		logger.debug(String.format("PutObject requestId:%s",result.getRequestId()));
	}



	private String parsePathPrefix(){
		return findPathPrefix().orElse("file/");
	}


	/**
	 * 解析到路径的前缀
	 * @return
	 */
	private Optional<String> findPathPrefix(){
		return Optional.ofNullable(domainGetter).map(getter->{
			String prefix = getter.get();
			AssertUtil.assertNormal(prefix!=null,()->new ServerException("domainGetter.get()不能为空"));
			return prefix;
		});
	}
	
	
	/**
	 * 上传文件，带后缀
	 * @return
	 */
	public BiFunction<String,String, String> uploadResSuffixFunction(){
		return (res,suffix)->uploadRes(res,suffix);
	}
	
	
	/**
	 * 上传一般文件源
	 * @return
	 */
	public Function<String, String> uploadResFunction(){
		return (res)->uploadRes(res);
	}
	
	/**
	 * 上传在oss域下的文件源
	 * @return
	 */
	public Function<String, String> uploadOssResFunction(){
		return uploadResFunction().compose(assemServerHomeFunction());
	}
	
	/**
	 * 拼装oss域
	 * @return
	 */
	public Function<String, String> assemServerHomeFunction(){
		return (path) -> assemServerHome(path);
	}
	
	
	/**
	 * 如果抛出异常，则表明操作失败，否则操作成功。抛出异常时，方法返回的数据无效
	 * 路径自动生成，文件后缀名是从 extName 中截取最后一个点后面的字符串
	 * */
	public String upload(InputStream inputStream,String extName){
		String pathPrefix = parsePathPrefix();
		OSSClient client = null;
		try{
			String path = createObjectKey(pathPrefix,extName);
			client = new OSSClient(occ.autoEndpoint(), occ.getAccessKeyId(), occ.getAccessKeySecret());
			ObjectMetadata metadata = new ObjectMetadata();
			//TODO 这里应该提供一个方式允许用户修改 显示名称，这是用于下载的  
			//metadata.setContentDisposition("attachment; filename=" + pathPrefix +  "." +  extName);
			PutObjectResult result = client .putObject(occ.getBucketName(), path, inputStream,metadata );
			this.markPutObjectResult(result);
			return path;
		}finally{
			quietShutdown(client);
		}
	}
	
	private static String createObjectKey(String pathPrefix,String extName){
		AssertUtil.assertMethodRequire(pathPrefix!=null, "pathPrefix不能为空");
		AssertUtil.assertSupport(extName!=null, "文件后缀名不能为空");
		if(extName.contains(".")){
			logger.debug(String.format("extName:%s中含有.现去掉最后一个点以及以前的字符",extName));
			extName = StringUtils.substringAfterLast(extName, ".");
		}
		String random = DateFormatUtils.format(new Date(),"yyyyMM/dd/yyyyMMddHHmmssSSS"+RandomStringUtils.randomNumeric(6));
		String path = String.format("%s/%s.%s",pathPrefix, random,extName);
		logger.debug(String.format("OssHelper createRandomKey:%s",path));
		return path;
	}
	

	
	/**
	 * 一次性授权oss直传
	 * @param expireSecond
	 * @param callbackUrl
	 * @return
	 */
	public MapParams grantAuthUpload(int expireSecond,String callbackUrl){
		return grantAuthUpload(expireSecond, callbackUrl,(mp)->{});
	}
	

	/**
	 * @param expireSecond
	 * @param callbackUrl
	 * @param callbackParamHandle  允许在回调参数创建后修改一下参数
	 * @return
	 */
	public MapParams grantAuthUpload(int expireSecond,String callbackUrl,Consumer<MapParams> callbackParamHandle){
        String bucket = occ.getBucketName();//"aqukedev";
        String accessId = occ.getAccessKeyId();// "LTAIWTpKSJflylEY"; // 请填写您的AccessKeyId。
        String accessKey = occ.getAccessKeySecret();// "Ir8eBOSYwMmDW8sx7MSwGwMmjqsTHY"; // 请填写您的AccessKeySecret。
        String host = "https://" + bucket + "." + occ.getEndpoint(); // host的格式为 bucketname.endpoint  这里上要给用户上传的所以不能写内网地址。
        // callbackUrl为 上传回调服务器的URL，请将下面的IP和Port配置为您自己的真实信息。
		String datePath = DateFormatUtils.format(new Date(),"yyyyMM/dd/yyyyMMddHHmmssSSS");
		String randomPath  = RandomStringUtils.randomNumeric(6);
        String ossDirPath = String.format("%s%s%s/",parsePathPrefix(), datePath,randomPath);
        OSSClient client = new OSSClient(occ.autoEndpoint(), accessId, accessKey);
        Date expireDate = DateUtils.addSeconds(new Date(),expireSecond);
		long fileSizeLimit = 10485760000L;
        long expireDateLong = expireDate.getTime();
        PolicyConditions policyConds = new PolicyConditions();
		// 这里设置上限 为 10 个 G。
        policyConds.addConditionItem(PolicyConditions.COND_CONTENT_LENGTH_RANGE, 0, fileSizeLimit);
        policyConds.addConditionItem(MatchMode.StartWith, PolicyConditions.COND_KEY, ossDirPath);

        String postPolicy = client.generatePostPolicy(expireDate, policyConds);
        byte[] binaryData = postPolicy.getBytes(StandardCharsets.UTF_8);
        String encodedPolicy = BinaryUtil.toBase64String(binaryData);
        String postSignature = client.calculatePostSignature(postPolicy);

        //TODO 这里参数配置有问题。build("fileName","${object}")
        MapParams callbackBody = MapParams.build("fileName","${object}").param("size", "${size}").param("mimeType", "${mimeType}");
        callbackParamHandle.accept(callbackBody);
        MapParams resultMap = MapParams.build()
				//TODO 应该配置封装成一个 对象。
        .param("fileSizeLimit", fileSizeLimit)
        .param("expireDate", expireDate)
        .param("accessid", accessId)
        .param("policy", encodedPolicy)
        .param("signature", postSignature)
        .param("dir", ossDirPath)
        .param("host", host)
        .param("expire", expireDateLong);
        MapParams callbackParams =MapParams.build("callbackBodyType", "application/x-www-form-urlencoded")
        .param("callbackUrl", callbackUrl)//"http://111.231.88.14:8088/test"
        .param("callbackBody",callbackBody.toUrlParams());
        String base64CallbackBody = BinaryUtil.toBase64String(GsonUtil.buildMilliSecondDateGson().toJson(callbackParams).getBytes());
        resultMap.put("callback", base64CallbackBody);
        MapParams devDesc = MapParams.build("desc", "在使用这个接口的时候务必要将当前使用的bucket设置可以跨域post访问,否则会出来403错误")
        		.param("bucket", bucket)
        		.param("bucketHost", host)
        		.param("setPost", "进入阿里云控制台oss控制台进入对应bucket->基础设置->跨域设置添加post允许规则,以下是 上海 节点的设置地址")
        		.param("setPostUrl", String.format("https://oss.console.aliyun.com/bucket/oss-cn-shanghai/%s/settings/cors", bucket))
        		;
        resultMap.param("devDesc",devDesc);
        return resultMap;
	}

	/**
	 * 重置oss路径
	 * @param path
	 * @param newPath
	 */
	public void resetPath(String path, String newPath) {
		logger.debug("开始resetPath path:{},newPath:{}",path,newPath);
		OSSClient client = null;
		try{
			client = new OSSClient(occ.autoEndpoint(), occ.getAccessKeyId(), occ.getAccessKeySecret());
			client.copyObject(occ.getBucketName(), path, occ.getBucketName(), newPath);
			client.deleteObject(occ.getBucketName(), path);
		}finally{
			quietShutdown(client);
		}
		logger.debug("结束resetPath path:{},newPath:{}",path,newPath);
	}
	
	
	/**
	 * 换一个新的地址
	 * @param path
	 * @return
	 */
	public String resetPathAuto(String path) {
		//TODO 解析出后缀应该做成一个方法可以放到regexUtil里
		String extName = StringUtils.substringAfterLast(path, ".");
		String newPath = createObjectKey(parsePathPrefix(), extName);
		resetPath(path, newPath);
		return newPath;
	}
	
}
