package com.mini.framework.util.report.statistics.protocol.process;


import com.mini.framework.core.exception.ServerException;
import com.mini.framework.util.asserts.AssertUtil;
import org.apache.commons.lang3.builder.ToStringBuilder;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

/**
 * 分发查询的进度。
 */
public class ForkQueryProcess {

    /**
     * 查询点列表
     */
    private List<ForkQuerySpot<?>> querySpots = new ArrayList<>();


    //TODO 需要确认一下什么时候会被释放
    //TODO 这里会导致Gson 序列化失败栈溢出
    private ExecutorService fixedThreadPool;


    /**
     * 是不是使用新线程
     */
    private Boolean otherThread;

    public static ForkQueryProcess create(Optional<Integer> poolSizeOptional,ThreadFactory threadFactory){
        ForkQueryProcess instance = new ForkQueryProcess();
        poolSizeOptional.ifPresent(poolSize->{
            instance.fixedThreadPool = new ThreadPoolExecutor(1, poolSize,
                    3*60*1000L, TimeUnit.MILLISECONDS,
                    new LinkedBlockingQueue<>(),threadFactory);

            instance.otherThread = true;
        });

        if(!poolSizeOptional.isPresent()){
            instance.otherThread = false;
        }
        return instance;
    }


    /**
     * 推送任务并马上执行
     * @param queryFunction
     * @param <R>
     * @return
     */
    public <R> R pushQueryNowExecute(Supplier<R> queryFunction,Object processParams){
        return pushQuery(queryFunction,processParams).get();
    }






    /**
     * 推送任务
     * @param queryFunction
     * @param <R>
     */
    public <R> Supplier<R> pushQuery(Supplier<R> queryFunction,Object processParams){
        if(otherThread){
            AssertUtil.assertNotFatal(!fixedThreadPool.isShutdown(),"线程池已结束不能在添加任务");
            ForkQuerySpot<R> spot = ForkQuerySpot.createThenAsyncBegin(queryFunction, fixedThreadPool,processParams);
            querySpots.add(spot);
            return ()->{
                try {
                    return spot.showAsyncExecutor().get();
                } catch (InterruptedException e) {
                    throw new ServerException(e,"线程被异常打断");
                } catch (ExecutionException e) {
                    throw new ServerException(e,"执行一个查询的时候错误了");
                }
            };
        }else{
            ForkQuerySpot<R> spot = new ForkQuerySpot<>();
            querySpots.add(spot);
            return ()->spot.fillThenSyncBegin(queryFunction);
        }
    }


    /**
     * 合并其它的查询点
     * @param other
     */
    public ForkQueryProcess mergeOtherQuerySpots(ForkQueryProcess other){
        querySpots.addAll(other.querySpots);
        return this;
    }




    /**
     * 结束线程池，关闭资源
     */
    public void closeResource(){
        if(fixedThreadPool!=null){
            fixedThreadPool.shutdown();
        }
        fixedThreadPool = null;
    }


    public List<ForkQuerySpot<?>> getQuerySpots() {
        return querySpots;
    }

    public void setQuerySpots(List<ForkQuerySpot<?>> querySpots) {
        this.querySpots = querySpots;
    }

    public Boolean getOtherThread() {
        return otherThread;
    }

    public void setOtherThread(Boolean otherThread) {
        this.otherThread = otherThread;
    }
    
    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }
}
