package com.mini.framework.util.report.statistics.summation;

import com.mini.framework.core.exception.LogicException;
import com.mini.framework.core.exception.ServerException;
import com.mini.framework.core.exception.standard.CustomException;
import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.date.DateRange;
import com.mini.framework.util.function.FinalUsefulBean;
import com.mini.framework.util.report.statistics.protocol.StatisticsSerializeParams;
import com.mini.framework.util.report.statistics.protocol.SummationDimension;
import com.mini.framework.util.report.statistics.protocol.TimeRegionRange;
import com.mini.framework.util.report.statistics.protocol.TimeRegionUnit;
import com.mini.framework.util.report.statistics.protocol.params.QueryParamsOverview;
import com.mini.framework.util.report.statistics.protocol.process.ProcessContext;
import com.mini.framework.util.report.statistics.summation.bean.NativeRegionChildrenRegionSummationLine;
import com.mini.framework.util.report.statistics.summation.bean.NativeRegionSummationElementAmount;
import com.mini.framework.util.report.statistics.summation.bean.NativeRegionTimeOutNames;
import com.mini.framework.util.report.statistics.summation.bean.SummationElementAmount;
import com.mini.framework.util.report.statistics.summation.bean.array.MultiParamsNativeRegionSummationArray;
import com.mini.framework.util.report.statistics.summation.bean.array.NativeRegionArraySummationArray;
import com.mini.framework.util.report.statistics.summation.bean.compare.BothSameRoundSummationCompare;
import com.mini.framework.util.report.statistics.summation.bean.compare.BothSummationCompare;
import com.mini.framework.util.report.statistics.summation.bean.compare.NativeRegionChildrenRegionSummationLineCompare;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiPredicate;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

/**
 * 某个一个纬度的求和的特征。<BR>
 * 任何纬度都可能做求和。<br>
 * 这些求和有相同的地方，也有不同的地方，
 * 不同的地方就是他的特征，需要不同的维度是实现。
 * #标准名词说明#dimension:业务维度例如 订单量/推广员人数/总人数等;
 * #标准名词说明#summation:求和;
 * @param <P> 维度中的参数。
 */
public interface SummationDimensionFeature<P extends StatisticsSerializeParams> extends SummationStatistics<P> {


    /**
     * 显然当前的统计纬度
     * @return 统计之求和的一个维度
     */
    SummationDimension showStatisticsDimension();

    @Override
    default String showStatisticsFeatureKey(){
        return showStatisticsDimension().showDimensionStringKey();
    }

    /**
     * 查询自然时间的求和记录
     * @return 如果没有就返回empty
     */
    Optional<NativeRegionSummationElementAmount> queryNatureDateSummationRecord(P params, String regionUnionKey);


    /**
     * 批量查询自然时间的求和记录
     * @param params 一个可以序列化的自定义参数
     * @param regionRanges  多个一个自然时间区域的范围实体
     * @return 以regionRanges.unitOffsetUnionString()为key
     * @see TimeRegionRange#unitOffsetUnionString()
     */
    List<NativeRegionSummationElementAmount> queryBatchNatureDateSummationRecord(P params, List<TimeRegionRange> regionRanges);




    /**
     * 保存自然时间的求和记录
     * @param record 自然时候的求和记录
     */
    void saveNatureDateSummationRecord(NativeRegionSummationElementAmount record);


    /**
     * 批量自然时间的求和记录
     * @param paramsUniqueSerializeMd5 参数的唯一值
     * @param regionUnionKeys 区域的联合key 的集合
     * @return 自然时间求和数字的列表，每个自然时间区间的求和数字
     */
    default List<NativeRegionSummationElementAmount> queryBatchNatureDateSummationRecord(String paramsUniqueSerializeMd5, Set<String> regionUnionKeys){
        return queryMultiParamsBatchNatureDateSummationRecord(Stream.of(paramsUniqueSerializeMd5).collect(Collectors.toSet()), regionUnionKeys);
    }


    /**
     * 批量自然时间的求和记录
     * @param paramsUniqueSerializeMd5s 多个参数的唯一值
     * @param regionUnionKeys 区域的联合key 的集合
     * @return 自然时间求和数字的列表，每个自然时间区间的求和数字
     */
    List<NativeRegionSummationElementAmount> queryMultiParamsBatchNatureDateSummationRecord(Set<String> paramsUniqueSerializeMd5s, Set<String> regionUnionKeys);


    /**
     * 统计指定时间范围的的求和。
     * @param params 一个可以序列化的自定义参数
     * @param dateRange 时间范围。左闭右开原则。
     * @return 求和的数字
     */
    SummationElementAmount statisticsDateRangeSummationFromImpl(P params, DateRange dateRange);


    /**
     * 能接受的最大的统计 源数据 的时间范围<BR>
     * 在作统计的时间会按时间单元拆分统计，<BR>
     * 为了不让统计的范围太大导致一些问题，<BR>
     * 这里可以设置时间拆分的最大粒度。
     * @return 时间单元
     */
    TimeRegionUnit acceptStatisticsOriginMaxUnit();

    /**
     * 能接受的最小的缓存聚合数据的时间范围<BR>
     * 在作统计的时间会按时间单元拆分统计，这里设置，最小的聚合单元。
     * @return 时间单元
     */
    TimeRegionUnit acceptCacheStatisticsMinDateUnit();


    /**
     * 允许忽略的有缺陷的时间单元<br>
     * 例如 这里返回 hour 那么 12月13号5点15分16秒。<br>
     * 会被当作 12月13号5点 处理。而忽略 最后的15分16秒。这个忽略可以提高比较大的性能。
     * @return 可以接受的忽略的时间单元
     */
    Optional<TimeRegionUnit> acceptSkipStatisticsDefectDateUnit();
    

    /**
     * 在使用做一次前置验证。有问题早发现
     */
    void previewValidateImpl();

    //-------------------------------------以上是需要被实现的方法-----------------------------------------


    /**
     * 计算出可能使用的历史限制时间
     * @param historyLimitDate 上层调用输入的历史限制时间
     * @return 真正用于统计的历史限制时间。
     */
    default Date countUsefulHistoryLimitDate(Date historyLimitDate){
        return acceptSkipStatisticsDefectDateUnit()
                .map(unit->unit.nearLazyFenceRangeDate(historyLimitDate))
                .orElse(historyLimitDate);
    }



    /**
     * 这里定义 SummationElementAmount 的检查，如果有错误就会抛出这个地异常，此方法可以被重写。
     * @return 根据错误信息返回异常的function
     */
    default Function<String,CustomException> validateExceptionSupplierSummationAmountRight(){
        return error -> new ServerException(Status.Server.programConfigJava,"求和记录不完整,%s",error);
    }

    /**
     * TODO 后续 要做几个接口用于提供几个处理异常的通用方案<br>
     * 执行某个方法:给持久层插入数据的方法。<BR>
     * 当前方法控制插入数据时出现问题(联合唯一约束)时要做的事情。<br>
     * 例如可能有以下几种可选的做法:<BR>
     * 1，出现错误时，直接忽略什么都不做。他不会影响整体结果<BR>
     * 2，出现错误时，抛出异常中断程序的执行。
     * @param runnable 对应的方法。
     */
    void executePersistentAddRecord(Runnable runnable);

    //-------------------------------------------------------------------------------
    

    /**
     * 在使用做一次前置验证。有问题早发现
     */
    default void previewValidate(){
        Function<String, ServerException> supplier
                = message-> new ServerException(Status.Server.programConfigJava,"程序使用有问题需要检查utilFullyEmptyItem()出错,%s",message);
        TimeRegionUnit originMaxUnit = acceptStatisticsOriginMaxUnit();
        List<TimeRegionUnit> supportMaxUnits = Stream.of(TimeRegionUnit.year, TimeRegionUnit.season,TimeRegionUnit.month, TimeRegionUnit.day, TimeRegionUnit.week)
                .collect(Collectors.toList());
        if(!supportMaxUnits.contains(originMaxUnit)){
            throw supplier.apply(String.format("最大支持的统计点:[%s]应该在此范围内:%s",originMaxUnit,supportMaxUnits));
        }
        previewValidateImpl();
    }


    /**
     * 统计指定时间范围的的求和。
     * 标记一下泛型出错。方便排查。
     * @param params 一个可以序列化的自定义参数
     * @param dateRange 时间范围。左闭右开原则。
     * @return 求和的数字
     */
    default SummationElementAmount statisticsDateRangeSummationFromImplWithTypeValidate(P params, DateRange dateRange){
        //TODO 目前没有想到比较好的提示，希望这种方式能够协助排查问题。

        // 如果参数 params 对应的 泛型出错 可能会出现 异常 ClassCastException ... cannot be cast to  ... 等字样。
        //当你看到这个代码时建议检查一下，实现类中，泛型与运行中的是否一致。
        SummationElementAmount implSummation = statisticsDateRangeSummationFromImpl(params, dateRange);
        implSummation.validateRight(error-> validateExceptionSupplierSummationAmountRight().apply(error));
        return implSummation;
    }

    /**
     * 是不是需要存到缓存里面
     * @param statisticsMaxDate 当前统计的范围的最大时间
     * @param queryLimitDate 历史时间的极限，也就是可以进入历史数据的最大时间
     * @return 是不是需要保存到缓存中去
     */
    default boolean ifRequireSaveToCache(Date statisticsMaxDate, Date queryLimitDate){
        // 这个可以被子类重写
        return ! queryLimitDate.before(statisticsMaxDate);
    }


    /**
     * 验证自然时间区域序列参数是不是符合规范。
     * TODO 以后这个方法要放到地 TimeRegionUnit 中去用于检查参数合理性。
     * @param regionUnit 自然时间的单元
     * @param dateRange 时间的范围
     */
    default void validateNativeDateRegionRangeArray(TimeRegionUnit regionUnit, DateRange dateRange){
        regionUnit.validateNativeDateRightPoint(dateRange
                ,error->new ServerException( Status.Server.programConfigJava,"类型为:[%s],自然时间序列参数错误%s未作前置检查,%s",regionUnit,dateRange,error));
    }


    /**
     * 统计某一个时间范围求和
     * @param params 一个可以序列化的自定义参数
     * @param dateRange 时间范围。左闭右开原则。
     * @param queryLimitDate 历史时间的极限，也就是可以进入历史数据的最大时间
     * @return 求和的数字
     */
    default SummationElementAmount statisticsOneParamsDateRangeSummation(
            ProcessContext context
            ,P params
            , DateRange dateRange
            , Date queryLimitDate){
        previewValidate();
        Date realStatisticsLimitDate = countUsefulHistoryLimitDate(queryLimitDate);
        String paramsMd5Key = params.uniqueSerializeMd5();
        SummationElementAmount result = statisticsMultiParamsDateRangeSummation(
                context,Stream.of(params).collect(Collectors.toList()), dateRange, realStatisticsLimitDate)
                .get(params.uniqueSerializeMd5());
        AssertUtil.assertNormal(result!=null,()->new LogicException(Status.Logic.def,"前面数据处理错误,没有:[%s]对应结果",paramsMd5Key));
        result.markResultMeta(realStatisticsLimitDate);
        result.fillQueryProcess(context.getProcess());
        return result;

    }


    /**
     * 统计某一个时间范围求和
     * @param paramsList 多个可以序列化的自定义参数
     * @param dateRange 时间范围。左闭右开原则。
     * @param queryLimitDate 历史时间的极限，也就是可以进入历史数据的最大时间
     * @return 每个参数的序列md5值对应求和的数字
     */
    default Map<String,SummationElementAmount> statisticsMultiParamsDateRangeSummation(
            ProcessContext context
            ,List<P> paramsList
            , DateRange dateRange
            , Date queryLimitDate){
        //体现上 queryLimitDate 大于这个时间直接忽略。
        DateRange fullyStatisticsDateRange = dateRange.limitUpperDate(queryLimitDate);
        List<String> paramsMd5List = paramsList.stream().map(P::uniqueSerializeMd5).collect(Collectors.toList());

        // 无法切分的时间碎片时间段。
        List<DateRange> regionChipDateRanges = new ArrayList<>();
        //1 得到要计算的自然时间片段。
        List<TimeRegionRange> requireQueryRegions = TimeRegionRange.splitRangeLessPointHandleChip(fullyStatisticsDateRange
                , acceptCacheStatisticsMinDateUnit(), acceptStatisticsOriginMaxUnit()
                , regionChipDateRanges::add);


        //---------------------------参数已准备好----------------------------------

        //得到完全cache过的数据的自然时间区域的结果
        List<NativeRegionSummationElementAmount> cacheRegionSummations = queryMultiParamsBatchNatureDateSummationRecord(
                new HashSet<>(paramsMd5List), requireQueryRegions.stream().map(TimeRegionRange::unitOffsetUnionString).collect(Collectors.toSet()));


        //得到 参数md5+时间  与  统计结果的映射
        Map<String,List<NativeRegionSummationElementAmount>>cacheRegionParamsMd5DateSummationMapper = cacheRegionSummations.stream()
                .collect(Collectors.groupingByConcurrent(NativeRegionSummationElementAmount::showRegionAndParamsMd5UnionKey));

        //检查有没有重复的数据
        cacheRegionParamsMd5DateSummationMapper.forEach((key,list)->{
            // NativeRegionSummationElementAmount::showRegionAndParamsMd5UnionKey 返回的数据必然是唯一的，如果看到这里出现错误，基本要检查一下是不是忘了弄唯一约束导致有bug
            AssertUtil.assertDataComplete(list.size()==1,"统计之求和中发现cache中有:[%s]个重复(key:[%s])的数据:%s",list.size(),key,list);
        });

        //定义一个方法，检查 参数以及时间区间是不是在缓存中。
        BiPredicate<P,TimeRegionRange> cacheParamsMd5RegionSummationsFound = (params,region)->{
            String mapperKey = NativeRegionSummationElementAmount.showRegionAndParamsMd5UnionKey(region,params);
            return !cacheRegionParamsMd5DateSummationMapper.containsKey(mapperKey);
        };


        //得到 参数md5  与  缓存中统计了的列表  的映射。
        Map<String, List<NativeRegionSummationElementAmount>> cacheParamsMd5SummationsMapper = cacheRegionSummations.stream()
                .collect(Collectors.groupingByConcurrent(NativeRegionSummationElementAmount::getParamsSerializeMd5));


        //参数md5 与 未来要查询的映射。
        Map<String,List<Supplier<SummationElementAmount>>> futureQueriesParamsMd5SummationsMapper = new HashMap<>();
        paramsList.forEach(params-> {
                    List<Supplier<SummationElementAmount>> futureQueries = requireQueryRegions.stream()
                            .filter(region -> cacheParamsMd5RegionSummationsFound.test(params, region))
                            .sorted(TimeRegionRange.randomSorter())
                            .map(region -> context.getProcess()
                                    .pushQuery(statisticsNativeDateSummationFromCacheOrImplSupplier(context, params, region, queryLimitDate), QueryParamsOverview.create(params.uniqueSerializeMd5(),region))
                            ).collect(Collectors.toList());
                    futureQueriesParamsMd5SummationsMapper.put(params.uniqueSerializeMd5(),futureQueries);
                }
        );


        //参数md5 与 未来要查询的映射。
        Map<String,List<Supplier<SummationElementAmount>>> chipFutureParamsMd5SummationsMapper = new HashMap<>();
        paramsList.forEach(params->{
            List<Supplier<SummationElementAmount>> summations = regionChipDateRanges.stream()
                    .map(regionChip -> context.getProcess()
                            .pushQuery(() -> statisticsDateRangeSummationFromImplWithTypeValidate(params, regionChip), QueryParamsOverview.create(params.uniqueSerializeMd5(),regionChip)))
                    .collect(Collectors.toList());
            chipFutureParamsMd5SummationsMapper.put(params.uniqueSerializeMd5(),summations);
        });

        //执行的批量查询
        Map<String,List<SummationElementAmount>> resultQueriesParamsMd5SummationsMapper = new HashMap<>();
        futureQueriesParamsMd5SummationsMapper
                .forEach((key,list)->resultQueriesParamsMd5SummationsMapper.put(key,list.stream().map(Supplier::get).collect(Collectors.toList())));


        //执行的批量查询
        Map<String,List<SummationElementAmount>> chipResultParamsMd5SummationsMapper = new HashMap<>();
        chipFutureParamsMd5SummationsMapper.forEach((key,list)->chipResultParamsMd5SummationsMapper.put(key,list.stream().map(Supplier::get).collect(Collectors.toList())));

        Map<String, SummationElementAmount> resultMapper = new HashMap<>();
        paramsList.stream().map(P::uniqueSerializeMd5)
                .forEach(paramsMd5->{
                    SummationElementAmount paramsMd5AllSummation = Stream.of(
                            cacheParamsMd5SummationsMapper.get(paramsMd5)
                            , resultQueriesParamsMd5SummationsMapper.get(paramsMd5)
                            , chipResultParamsMd5SummationsMapper.get(paramsMd5)
                    ).filter(Objects::nonNull).flatMap(List::stream)
                    .map(SummationElementAmount::refAsSummationElementAmount)
                    .reduce(SummationElementAmount.createEmptyInstance(),SummationElementAmount::mergeAmount);
                    resultMapper.put(paramsMd5,paramsMd5AllSummation);
        });


/*
        
        paramsList.forEach(params->{
            //已缓存的区域的求和计数
            SummationElementAmount cacheRegionAmount = cacheRegionAmountRecords
                    .stream()
                    .filter(summation->summation.matchParamsMd5(params))
                    .map(a -> (SummationElementAmount) a)
                    .reduce(SummationElementAmount.createEmptyInstance(), SummationElementAmount::mergeAmount);

            SummationElementAmount requireQuerySummation = requireQueryRegions.stream()
                    .filter(region -> cacheParamsRegionAmountRecordsFound.test(params, region))
                    .map(region->resultQueriesParamsMd5SummationsMapper.get(NativeRegionSummationElementAmount.showRegionAndParamsMd5UnionKey(region,params)))
                    .reduce(SummationElementAmount.createEmptyInstance(),SummationElementAmount::mergeAmount);


            //区域之间的碎片时间的计数
            SummationElementAmount regionChipDateAmount = regionChipDateRanges.stream()
                    .map(regionChip -> statisticsDateRangeSummationFromImplWithTypeValidate(params, regionChip))
                    .reduce(SummationElementAmount.createEmptyInstance(), SummationElementAmount::mergeAmount);

            //当前这个参数下所有的计和的总和
            SummationElementAmount paramsAllAmount = Stream.of(cacheRegionAmount,requireQuerySummation,regionChipDateAmount)
                .reduce(SummationElementAmount.createEmptyInstance(), SummationElementAmount::mergeAmount);
            resultMapper.put(params.uniqueSerializeMd5(),paramsAllAmount);
        });*/
        return resultMapper;

    }



    default Supplier<SummationElementAmount> statisticsNativeDateSummationFromCacheOrImplSupplier(
            ProcessContext context,P params, TimeRegionRange regionRange, Date queryLimitDate){
        return ()->statisticsNativeDateSummationFromCacheOrImpl(context,params,regionRange,queryLimitDate);
    }

    default SummationElementAmount statisticsNativeDateSummationFromCacheOrImpl(
            ProcessContext context,P params, TimeRegionRange regionRange, Date queryLimitDate){
        Date realStatisticsLimitDate = countUsefulHistoryLimitDate(queryLimitDate);
        Optional<NativeRegionSummationElementAmount> implAmountOptional = queryNatureDateSummationRecord(params, regionRange.unitOffsetUnionString());
        DateRange requireDateRange = regionRange.showDateRange().limitUpperDate(realStatisticsLimitDate);
        return implAmountOptional.map(a->(SummationElementAmount)a)
                .orElseGet(()-> {
                    SummationElementAmount summation = statisticsDateRangeSummationFromImplWithTypeValidate(params, requireDateRange);
                    if(ifRequireSaveToCache(regionRange.mapperToDate(),queryLimitDate)){
                        //如果这个数据已成为历史，那么就可以储存了。
                        FinalUsefulBean<Boolean> bean = FinalUsefulBean.of(false);
                        executePersistentAddRecord(()->{
                            bean.apply(b->true);
                            saveNatureDateSummationRecord(NativeRegionSummationElementAmount.create(params,regionRange,summation));
                        });
                        AssertUtil.assertNotFatalBug(bean.get(),"工具使用异常 方法executePersistentAddRecord(Runnable)中Runnable参数必须是被run了的");
                    }
                    summation.markResultMeta(realStatisticsLimitDate);
                    summation.fillQueryProcess(context.getProcess());
                    return summation;
                });
    }

    /**
     * 统计某个自然时间的求和
     * @param params 一个可以序列化的自定义参数
     * @param regionRange 一个自然时间区域的范围实体
     * @param queryLimitDate 历史时间的极限，也就是可以进入历史数据的最大时间
     * @return 自然时间的求和数字
     */
    @Override
    default NativeRegionSummationElementAmount statisticsNativeDateSummation(
            ProcessContext context, P params, TimeRegionRange regionRange, Date queryLimitDate){
        previewValidate();
        Date realStatisticsLimitDate = countUsefulHistoryLimitDate(queryLimitDate);
        SummationElementAmount summationAmount = statisticsOneParamsDateRangeSummation(context,params,regionRange.showDateRange(),queryLimitDate);
        NativeRegionSummationElementAmount result = NativeRegionSummationElementAmount.create(params,regionRange,summationAmount);
        result.markResultMeta(realStatisticsLimitDate);
        result.fillQueryProcess(context.getProcess());
        return result;
    }


    @Override
    default MultiParamsNativeRegionSummationArray<P> statisticsMultiParamsNativeDateSummation(
            ProcessContext context,List<P> paramsList, TimeRegionRange regionRange, Date queryLimitDate){
        previewValidate();

        Date realStatisticsLimitDate = countUsefulHistoryLimitDate(queryLimitDate);
        Map<String, SummationElementAmount> mapper = statisticsMultiParamsDateRangeSummation(context,paramsList, regionRange.showDateRange(), realStatisticsLimitDate);
        MultiParamsNativeRegionSummationArray<P> result = new MultiParamsNativeRegionSummationArray<>();
        result.markResultMeta(realStatisticsLimitDate);
        result.fillQueryProcess(context.getProcess());
        paramsList.forEach(params->{
                    SummationElementAmount amount = mapper.get(params.uniqueSerializeMd5());
                    AssertUtil.assertNormal(amount!=null,()->new LogicException(Status.Logic.def,"前面数据处理错误,没有params:[%s]对应结果",params.uniqueSerialize()));
                    NativeRegionSummationElementAmount summation = NativeRegionSummationElementAmount.create(params, regionRange, amount);
                    result.add(params,summation);
                });
        return result;

    }

    /**
     * 统计某个自然时间的求和，并提供对比
     * @param params 一个可以序列化的自定义参数
     * @param regionRange 一个自然时间区域的范围实体
     * @param queryLimitDate 历史时间的极限，也就是可以进入历史数据的最大时间
     * @param compareIndexOffset 要对比的线的位置。
     * @return 两者求和的对比
     */
    @Override
    default BothSummationCompare statisticsNativeDateSummationWithBothCompare(
            ProcessContext context,P params, TimeRegionRange regionRange, Date queryLimitDate, int compareIndexOffset){
        previewValidate();
        Date realStatisticsLimitDate = countUsefulHistoryLimitDate(queryLimitDate);
        TimeRegionRange compareRange = regionRange.nextRegionRange(compareIndexOffset);
        Function<TimeRegionRange, NativeRegionSummationElementAmount> regionRangeSummationMapper =
                statisticsRegionRangeArraySummationArrayPromiseNotOut(context,params, Stream.of(regionRange, compareRange).collect(Collectors.toList()), realStatisticsLimitDate);
        BothSummationCompare result = BothSummationCompare.createSummationInstance(regionRangeSummationMapper.apply(regionRange), regionRangeSummationMapper.apply(compareRange));
        //TODO 在这里添加 类 WorkDispatcher 解决工作分解的问题。
        result.markResultMeta(realStatisticsLimitDate);
        result.fillQueryProcess(context.getProcess());
        return result;
    }

    @Override
    default BothSameRoundSummationCompare statisticsNativeDateSummationWithSameRoundCompare(
            ProcessContext context, P params, TimeRegionRange regionRange, Date queryLimitDate
            , int roundCompareIndexOffset, int sameCompareIndexOffset, Optional<TimeRegionUnit> sameCompareUnitOptional){
        Date realStatisticsLimitDate = countUsefulHistoryLimitDate(queryLimitDate);
        TimeRegionRange roundCompareRange = regionRange.nextRegionRange(roundCompareIndexOffset);
        //TODO 需要校验 sameCompareUnitOptional 是不是合理的。
        Optional<TimeRegionRange> sameCompareRangeOptional = sameCompareUnitOptional.map(scopeUnit -> regionRange.nextSameRegionRange(scopeUnit, sameCompareIndexOffset));
        List<TimeRegionRange> regions = Stream.of(roundCompareRange, regionRange).collect(Collectors.toList());
        sameCompareRangeOptional.ifPresent(regions::add);
        Function<TimeRegionRange, NativeRegionSummationElementAmount> regionRangeSummationMapper = statisticsRegionRangeArraySummationArrayPromiseNotOut(context,params,regions,realStatisticsLimitDate);
        BothSameRoundSummationCompare result = BothSameRoundSummationCompare.create(
                regionRangeSummationMapper.apply(regionRange)
                , regionRangeSummationMapper.apply(roundCompareRange)
                , sameCompareRangeOptional.map(regionRangeSummationMapper));
        result.markResultMeta(realStatisticsLimitDate);
        result.fillQueryProcess(context.getProcess());
        return result;

    }


    /**
     * 统计多个连续的自然时间的求和
     * @param params 一个可以序列化的自定义参数
     * @param regionUnit 自然时间的单元
     * @param dateRange 时间范围
     * @param queryLimitDate 历史时间的极限，也就是可以进入历史数据的最大时间
     * @return 求和的数字
     */
    @Override
    default SummationElementAmount statisticsNativeDateArraySummation(
            ProcessContext context,P params, TimeRegionUnit regionUnit, DateRange dateRange, Date queryLimitDate){
        previewValidate();
        validateNativeDateRegionRangeArray(regionUnit, dateRange);
        Date rangeMinDate = dateRange.getMinDate();
        //坚持左闭又左原则，不再加1，如果前端为了操作方便可以controller提供加1的方法。
        Date rangeMaxDate = dateRange.getMaxDate();
        //Date rangeMaxDate = regionUnit.addTimeUnit(dateRange.getMaxDate(),1);
        return statisticsOneParamsDateRangeSummation(context,params,DateRange.create(rangeMinDate,rangeMaxDate),queryLimitDate);
    }


    @Override
    default SummationElementAmount statisticsCustomDateRangeSummation(
            ProcessContext context, P params, DateRange dateRange, Date queryLimitDate){
        return statisticsOneParamsDateRangeSummation(context,params, dateRange, queryLimitDate);
    }

    /**
     * 统计多个连续自然时间的分别求和得到列表
     * @param params 一个可以序列化的自定义参数
     * @param regionUnit 自然时间的单元
     * @param dateRange 时间的范围，左闭右开
     * @param queryLimitDate 历史时间的极限，也就是可以进入历史数据的最大时间
     * @return 自然时间求和数字的列表，每个自然时间区间的求和数字
     */
    @Override
    default NativeRegionArraySummationArray statisticsNativeDateArrayScatterSummation(
            ProcessContext context,P params, TimeRegionUnit regionUnit, DateRange dateRange, Date queryLimitDate){
        previewValidate();
        Date realStatisticsLimitDate = countUsefulHistoryLimitDate(queryLimitDate);
        validateNativeDateRegionRangeArray(regionUnit,dateRange);
        List<TimeRegionRange> requireRanges = regionUnit.regionFences(dateRange.getMinDate(), dateRange.getMaxDate());
        NativeRegionArraySummationArray result = statisticsRegionRangeArraySummationArray(context, params, requireRanges, queryLimitDate);
        result.markResultMeta(realStatisticsLimitDate);
        result.fillQueryProcess(context.getProcess());
        result.fillArraySequenceAccumulates();
        return result;
    }



    /**
     * 统计一个时间区域数组的求和数据 的方法
     * @param params 一个可以序列化的自定义参数
     * @param regionRanges 个自然时间区域的范围实体列表
     * @param queryLimitDate 历史时间的极限，也就是可以进入历史数据的最大时间
     * @return 得到一个用自然时间区间换统计值的方法，并保证一定不会超出查询范围
     */
    default Function<TimeRegionRange, NativeRegionSummationElementAmount> statisticsRegionRangeArraySummationArrayPromiseNotOut(
            ProcessContext context,P params, List<TimeRegionRange> regionRanges, Date queryLimitDate){
        previewValidate();
        return statisticsRegionRangeArraySummationArray(context,params, regionRanges, queryLimitDate
                ,key-> {throw new ServerException("key:[%s]错误找不到数据超出的计划的数据",key);});
    }

    /**
     * 统计一个时间区域数组的求和数据
     * @param params 一个可以序列化的自定义参数
     * @param regionRanges 一个自然时间区域的范围实体
     * @param queryLimitDate 历史时间的极限，也就是可以进入历史数据的最大时间
     * @param ifOutKeySupplier 如果key不存在提供异常的方法
     * @return 得到一个方法查询的方法
     */
    default Function<TimeRegionRange, NativeRegionSummationElementAmount> statisticsRegionRangeArraySummationArray(
            ProcessContext context,P params, List<TimeRegionRange> regionRanges, Date queryLimitDate
            , Function<TimeRegionRange, NativeRegionSummationElementAmount> ifOutKeySupplier
            ){
        previewValidate();
        //执行批量统计得到每个时间点的求和。
        List<NativeRegionSummationElementAmount> allSummation = statisticsRegionRangeArraySummationArray(context,params, regionRanges, queryLimitDate).getElements();
        return regionRange-> allSummation.stream()
                .filter(summation->summation.mapperRegionRange().match(regionRange))
                .findFirst().orElseGet(()->ifOutKeySupplier.apply(regionRange));
    }


    /**
     * 统计一个时间区域数组的求和数据
     * @param params 一个可以序列化的自定义参数
     * @param regionRanges 自然时间区域的范围实体列表
     * @param queryLimitDate 历史时间的极限，也就是可以进入历史数据的最大时间
     * @return 自然时间求和数字的列表，每个自然时间区间的求和数字
     */
    default NativeRegionArraySummationArray statisticsRegionRangeArraySummationArray(
            ProcessContext context,P params, List<TimeRegionRange> regionRanges, Date queryLimitDate){
        previewValidate();
        Date realStatisticsLimitDate = countUsefulHistoryLimitDate(queryLimitDate);
        List<NativeRegionSummationElementAmount> cacheSummations = queryBatchNatureDateSummationRecord(params, regionRanges);

        Map<String, NativeRegionSummationElementAmount> regionUnionKeySummationMapper = cacheSummations.stream()
                .collect(Collectors.toMap(NativeRegionSummationElementAmount::getRegionUnionKey, Function.identity()));
        List<NativeRegionSummationElementAmount> allSummation = regionRanges.stream().map(requireRange ->
            Optional.ofNullable(regionUnionKeySummationMapper.get(requireRange.unitOffsetUnionString()))
            .orElseGet(() -> {
                SummationElementAmount thisRangeSummation = statisticsNativeDateSummation(context,params, requireRange, realStatisticsLimitDate);
                return NativeRegionSummationElementAmount.create(params, requireRange, thisRangeSummation);
            })
        ).collect(Collectors.toList());
        NativeRegionArraySummationArray result = NativeRegionArraySummationArray.create(allSummation);
        result.markResultMeta(realStatisticsLimitDate);
        result.fillQueryProcess(context.getProcess());
        result.fillArraySequenceAccumulates();
        return result;
    }


    /**
     * 统计某个自然时间内分子时间单元的各个小区间的求和。
     * @param params 一个可以序列化的自定义参数
     * @param regionRange 一个自然时间区域的范围实体
     * @param childrenRegionUnit 原子 时间区域单元
     * @param queryLimitDate 历史时间的极限，也就是可以进入历史数据的最大时间
     * @return 一个统计的列表
     */
    @Override
    default NativeRegionChildrenRegionSummationLine statisticsNativeDateChildrenArraySummationLine(
            ProcessContext context,P params, TimeRegionRange regionRange, TimeRegionUnit childrenRegionUnit, Date queryLimitDate){
        previewValidate();
        Date realStatisticsLimitDate = countUsefulHistoryLimitDate(queryLimitDate);
        List<TimeRegionRange> childrenRegionRanges = regionRange.findElementRegionFences(childrenRegionUnit);
        List<NativeRegionSummationElementAmount> summations = statisticsRegionRangeArraySummationArray(context,params, childrenRegionRanges, realStatisticsLimitDate).getElements();
        Function<Date, NativeRegionTimeOutNames> dateOutNameMapper = NativeRegionTimeOutNames.nativeUnitOutName(regionRange,childrenRegionUnit);
        summations.forEach(summation->summation.fillOutNames(dateOutNameMapper));
        NativeRegionChildrenRegionSummationLine result = NativeRegionChildrenRegionSummationLine.create(regionRange.getOffset(), regionRange.getUnit(), childrenRegionUnit, summations);
        result.markResultMeta(realStatisticsLimitDate);
        result.fillQueryProcess(context.getProcess());
        result.fillArraySequenceAccumulates();
        return result;
    }

    /**
     * 统计多个连续的自然时间的分别求和 并对比其它曲线
     * @param params 一个可以序列化的自定义参数
     * @param regionRange 一个自然时间区域的范围实体
     * @param childrenRegionUnit 原子 时间区域单元
     * @param queryLimitDate 历史时间的极限，也就是可以进入历史数据的最大时间
     * @param compareIndexOffsets 需要对比的索引偏移量，例如如果要看上一个周期的统计线，这里要要传 数组 [-2,-1]
     * @return 自然时间区域下面的自然时间区域的分布线的对比线
     */
    @Override
    default NativeRegionChildrenRegionSummationLineCompare statisticsNativeDateChildrenArraySummationLineWithCompare(
            ProcessContext context,P params, TimeRegionRange regionRange, TimeRegionUnit childrenRegionUnit, Date queryLimitDate, int[] compareIndexOffsets){
        previewValidate();

        Date realStatisticsLimitDate = countUsefulHistoryLimitDate(queryLimitDate);
        //所有的偏移量的集合。
        Set<Integer> allIndexOffsets = IntStream.of(compareIndexOffsets).boxed().collect(Collectors.toSet());
        //主线也算一个偏移量为0的特殊点
        allIndexOffsets.add(0);

        //得到所有需要统计的时间点。
        List<TimeRegionRange> requireSummationRegions = allIndexOffsets.stream().distinct().sorted().map(regionRange::nextRegionRange)
                .map(range -> range.findElementRegionFences(childrenRegionUnit))
                .flatMap(List::stream).collect(Collectors.toList());

        //根据所有的求和得到一个用时间点换求和的方法。
        Function<TimeRegionRange, NativeRegionSummationElementAmount>regionSummationLineFinder
                = statisticsRegionRangeArraySummationArrayPromiseNotOut(context,params, requireSummationRegions, realStatisticsLimitDate);

        //预备一个根据偏移量得到统计线的方法。
        Function<Integer,NativeRegionChildrenRegionSummationLine> indexOffsetSummationLineFinder = indexOffset->{
            TimeRegionRange indexRegionRange = regionRange.nextRegionRange(indexOffset);
            List<NativeRegionSummationElementAmount> summations = indexRegionRange.findElementRegionFences(childrenRegionUnit)
                    .stream().map(regionSummationLineFinder).collect(Collectors.toList());
            return NativeRegionChildrenRegionSummationLine.create(indexRegionRange.getOffset(),indexRegionRange.getUnit(),childrenRegionUnit,summations);
        };

        //计算出对比线。
        List<NativeRegionChildrenRegionSummationLine> contrastLines = IntStream.of(compareIndexOffsets).sorted().boxed()
                .map(indexOffsetSummationLineFinder).collect(Collectors.toList());
        //计算出主线。这里注意，偏移量为0就代表了主线。
        NativeRegionChildrenRegionSummationLine mainLine = indexOffsetSummationLineFinder.apply(0);
        NativeRegionChildrenRegionSummationLineCompare result = NativeRegionChildrenRegionSummationLineCompare.create(mainLine, contrastLines);
        result.markResultMeta(realStatisticsLimitDate);
        result.fillQueryProcess(context.getProcess());
        result.fillArraySequenceAccumulates();
        return result;
    }

}
