package com.mini.framework.util.surplus.reduce;

import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.function.PredicateSpread;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.core.ListOperations;
import org.springframework.data.redis.core.RedisOperations;
import org.springframework.data.redis.core.SessionCallback;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

/**
 *剩余量预先准备扣减的的队列的redis实现
 * @author jayheo
 */
public abstract class SurplusReduceCacheQueueRedisTemplateImpl<K> implements SurplusReduceCacheQueue<K>{

    /**
     * 得到当前使用的list操作器
     * @see RedisTemplate#TODO
     * */
    protected abstract ListOperations<String,ReduceQueueItem> currentListOperation();

    /**
     * 根据key生成redis的key
     * */
    protected abstract String createRedisKey(K key);

    private ListOperations<String,ReduceQueueItem> safeListOperation(){
        ListOperations<String,ReduceQueueItem> operation = currentListOperation();
        AssertUtil.assertNotFatalBug(operation!=null,"currentListOperation 方法返回了空指针");
        //TODO 检查一下。
        return operation;
    }


    @Override
    public Optional<ReduceQueueItem> popItem(K key) {
        String redisKey = createRedisKey(key);
        //TODO 为什么没任何结果的数据还会返回一个空的对应。而不是空指针
        ReduceQueueItem item = safeListOperation().rightPop(redisKey);
        return Optional.ofNullable(item)
                .filter(ReduceQueueItem::ifCompleteRecord);
    }



    @Override
    public Optional<List<ReduceQueueItem>> popSatisfiedItems(K key, int size, Predicate<ReduceQueueItem> satisfiedCondition){
        List<ReduceQueueItem> items = new ArrayList<>();
        while (items.size()<size){
            Optional<ReduceQueueItem> itemOptional = popItem(key);
            if(itemOptional.isPresent()){
                itemOptional.ifPresent(item->items.add(item));
            }else{
                break;
            }
        }

        items.stream().allMatch(satisfiedCondition);

        if(items.size()==size&&items.stream().allMatch(satisfiedCondition)){
            return Optional.of(items);
        }else{
            List<ReduceQueueItem> cancelItems = items.stream()
                    .filter(PredicateSpread.of(satisfiedCondition).negate())
                    .collect(Collectors.toList());
            batchPushItem(key,cancelItems);
            return Optional.empty();
        }
    }


    @Override
    public void pushItem(K key, ReduceQueueItem item) {
        pushItem(safeListOperation(),key,item);
    }


    public void pushItem(ListOperations listOperations,K key, ReduceQueueItem item) {
        String redisKey = createRedisKey(key);
        listOperations.leftPush(redisKey,item);
    }



    @Override
    public int queueSize(K key) {
        String redisKey = createRedisKey(key);
        long longSize = safeListOperation().size(redisKey);
        AssertUtil.assertNotFatalBug(longSize<=Integer.MAX_VALUE,"发现redisKey:[%s]的集合值:[%s]大于 最大的int",key,longSize);
        AssertUtil.assertNotFatalBug(longSize>=Integer.MIN_VALUE,"发现redisKey:[%s]的集合值:[%s]小于 最小的int",key,longSize);
        return (int) longSize;
    }


    @Override
    public void batchPushItem(K key, List<ReduceQueueItem> itemArray) {
        redisBatchPushItem(key,itemArray);
    }


    public void simpleBatchPushItem(K key, List<ReduceQueueItem> itemArray) {
        itemArray.forEach(item->{
            pushItem(key,item);
        });
    }


    /**
     * 使用redis的原生api批量插入，这个方法还没有实现
     * */
    public void redisBatchPushItem(K key, List<ReduceQueueItem> itemArray) {
        String redisKey = createRedisKey(key);
        if(!itemArray.isEmpty()){
            safeListOperation().leftPushAll(redisKey,itemArray);
        }
    }
}
