package com.mini.framework.util.asserts;

import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;

import com.mini.framework.core.exception.SoaException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.mini.framework.core.exception.BadDataException;
import com.mini.framework.core.exception.BadReqDelNoDataException;
import com.mini.framework.core.exception.BadReqException;
import com.mini.framework.core.exception.BadReqGetNoDataException;
import com.mini.framework.core.exception.BusyException;
import com.mini.framework.core.exception.HandleException;
import com.mini.framework.core.exception.HandleStoreException;
import com.mini.framework.core.exception.LogicException;
import com.mini.framework.core.exception.PermissionException;
import com.mini.framework.core.exception.ServerException;
import com.mini.framework.core.exception.ThirdException;
import com.mini.framework.core.exception.standard.CustomException;
import com.mini.framework.core.exception.standard.UnsupportedException;
import com.mini.framework.core.status.Status;
import com.mini.framework.core.status.StatusCode;

/**
 * 断言util
 * @author jayheo
 */
public class AssertUtil {
	
	private static Logger logger = LogManager.getLogger(AssertUtil.class);

	private static final Map<Class<? extends CustomException>, StatusCode> cacheExceptionStatusMap = createExceptionStatusMap();
	private static final Map<StatusCode,Class<? extends CustomException>> cacheStatusExceptionMap = createStatusExceptionMap();
	private static final Map<Integer,Class<? extends CustomException>> cacheStatusIntExceptionMap = createStatusIntExceptionMap();

	private static final int baseStatusCode = Status.Busy.def.code();
	private static Map<Class<? extends CustomException>, StatusCode> createExceptionStatusMap() {
		Map<Class<? extends CustomException>, StatusCode> statusExceptionMap = new ConcurrentHashMap<>();
		statusExceptionMap.put( BadReqException.class, Status.BadReq.def);
		statusExceptionMap.put( BadDataException.class, Status.BadData.def);
		statusExceptionMap.put( BusyException.class, Status.Busy.def);
		statusExceptionMap.put( LogicException.class, Status.Logic.def);
		statusExceptionMap.put( PermissionException.class, Status.Permission.def);
		statusExceptionMap.put( ThirdException.class, Status.Third.def);
		statusExceptionMap.put( HandleException.class, Status.Handle.def);
		statusExceptionMap.put( SoaException.class, Status.Soa.def);
		statusExceptionMap.put( ServerException.class, Status.Server.def);
		//TODO 子异常也要写出来 如果没有写出来会不会有bug，需要写测试用例验证
		return statusExceptionMap;
	}
	
	

	private static Map<Integer, Class<? extends CustomException>> createStatusIntExceptionMap() {
		Map<StatusCode, Class<? extends CustomException>> map = createStatusExceptionMap();
		Map<Integer,Class<? extends CustomException>> statusIntExceptionMap = new ConcurrentHashMap<>();
		for (Entry<StatusCode, Class<? extends CustomException>> entry : map.entrySet()) {
			statusIntExceptionMap.put(entry.getKey().code(),entry.getValue());
		}
		return statusIntExceptionMap;
	}



	private static Map<StatusCode,Class<? extends CustomException>> createStatusExceptionMap() {
		Map<StatusCode, Class<? extends CustomException>> statusExceptionMap= new ConcurrentHashMap<>();
		
		Map<Class<? extends CustomException>, StatusCode> map = createExceptionStatusMap();
		for (Entry<Class<? extends CustomException>, StatusCode> entry : map.entrySet()) {
			statusExceptionMap.put(entry.getValue(),entry.getKey());
		}
		return statusExceptionMap;
	}



	/**
	 * @param statusCode
	 * @param clazz
	 */
	public static void assertStatusSupport(StatusCode statusCode, Class<? extends CustomException> clazz) {
		StatusCode defStatusCode = cacheExceptionStatusMap.get(clazz);
		int subCode = statusCode.code() - defStatusCode.code();
		boolean right = JudgeUtil.match(subCode >= 0 && subCode < baseStatusCode, Require.isTrue);
		if( !right ){
			throw new UnsupportedException("致命错误状态类型%s和异常%s对不上,所以不支持",statusCode.code(),clazz);
		}
		
	}

	/**
	 * 要求每个值都不是错误的输入
	 * @param status 对应错误状态码
	 * @param require 需要检查的类型
	 * @param message 出错提示信息
	 * @param toAsserts  需要对比数据 ,也会当作message的参数输出到信息
	 */
	public static void assertAllNoBadReq(StatusCode status, Require require, String message, Object... toAsserts) {
		boolean right = true;
		for (Object toAssert : toAsserts) {
			right &= JudgeUtil.match(toAssert, require);
			if (!right) {
				throw new BadReqException(status, message,toAsserts);
			}
		}
	}
	/**
	 * toAssert 必须介于min max之间 [min,max) <br>
	 * 左闭又开原则<br>
	 * @param toAssert 受检查的值得
	 * @param min 最小值
	 * @param max 最大值
	 * @param message 错误提示
	 */
	public static <T> void assertBetween(Comparable<T> toAssert ,Comparable<T>  min,Comparable<T>  max,String message) {
		boolean right = JudgeUtil.match(toAssert, Require.between, min,max);
		if(!right){
			throw new BadReqException("%s:%s值域应该为[%s,%s)",message,toAssert,min,max);
		}
	}
	
	/**
	 * 查看被检查的值得在不在样本中
	 * @param toAssert  受检查的值得
	 * @param message 错误提示
	 * @param params 检查的样本
	 */
	public static <T> void assertValueIn(Object toAssert ,String message, Object... params) {
		if( params==null){
			throw new UnsupportedException("不能没有params参数",message);
		}
		
		for (Object param : params) {
			if((param.equals(toAssert))){
				return;
			}
		}
		throw new BadReqException(Status.BadReq.illParam,message,params);
	}
	
	/**
	 * 要求传的参数要么都为空 ，要么都不为空，不允许出现部分为空部分不 为空的情况
	 * @param message
	 * @param toAssert
	 * @return
	 */
	public static boolean assertSameStatus(String message,Object... toAssert){
		assertSupport(toAssert!=null && toAssert.length>0,"不能不传参需要立即修改代码");
		boolean full = true;
		boolean empty = true;
		for (Object object : toAssert) {
			full &= object != null;
			empty &= object == null;
		}
		assertNoBadReq(full||empty, Status.BadReq.illParam, message);
		return full;
	}
	
	
	/**
	 * 要求每个值都不是空
	 * @param status 对应错误状态码
	 * @param message 出错提示信息
	 * @param toAsserts 需要对比数据 ,也会当作message的参数输出到信息
	 */
	public static void assertAllNotNull(StatusCode status, String message, Object... toAsserts) {
		assertAllNoBadReq(status, Require.notNull, message, toAsserts);
	}
	/**
	 * 有任何不为 空的值
	 * 即至少有一个不为空的值
	 * @param status
	 * @param message
	 * @param toAsserts  需要对比数据 ,也会当作message的参数输出到信息
	 */
	public static void assertAnyNotNull( StatusCode status, String message, Object... toAsserts) {
		assertAnyNoBadReq(status, Require.notNull, message, toAsserts);
	}

	/**
	 * 至少有一个符合条件
	 * @param status 对应错误状态码
	 * @param require 需要检查的类型
	 * @param message 出错提示信息
	 * @param toAsserts  需要对比数据 ,也会当作message的参数输出到信息
	 */
	public static void assertAnyNoBadReq( StatusCode status,Require require, String message, Object... toAsserts) {
		boolean right = false;
		for (Object toAssert : toAsserts) {
			right |= JudgeUtil.match(toAssert, require);
			if (right) {
				return;
			}
		}
		throw new BadReqException(status, message,toAsserts);
	}

	

	public static void assertNoBadReq(boolean explain, StatusCode status, Supplier<String> supplier) {
		if (!explain) {
			throw new BadReqException(status, supplier.get());
		}
	}
	
	/**
	 * @param explain 判断表达式 false 抛异常 true不抛出
	 * @param status 对应错误状态码
	 * @param message 出错提示信息
	 * @param params message占位字符串 辅助判断参数
	 */
	public static void assertNoBadReq(boolean explain, StatusCode status, String message, Object... params) {
		assertStatusSupport(status, BadReqException.class);
		if (!explain) {
			throw new BadReqException(status, message, params);
		}
	}
	/**
	 * 不为为空指针
	 * @param obj 被检查的对象
	 * @param message 出错提示信息
	 * @param params message占位字符串 辅助判断参数
	 */
	public static void assertNotNull(Object obj, String message, Object... params) {
		assertNoBadReq(obj, Require.notNull, message, params);
	}
	/**
	 * 不为为空值
	 * @param obj 被检查的对象
	 * @param message 出错提示信息
	 * @param params message占位字符串 辅助判断参数
	 */
	public static void assertNotEmpty(Object obj, String message, Object... params) {
		assertNoBadReq(obj, Require.notEmpty, message, params);
	}
	/**
	 * 不为为空值
	 * @param coll 被检查的对象
	 * @param message 出错提示信息
	 * @param params message占位字符串 辅助判断参数
	 */
	public static <T> void assertNotEmpty(Collection<T> coll, String message, Object... params) {
		assertNoBadReq(coll!=null,Require.notNull, message, params);
		for (Object object : params) {
			assertNoBadReq(object, Require.notNull, message, params);
		}
	}

	/**
	 * 要求每个值都不是错误的输入
	 * @param obj 被检查的对象
	 * @param require 要求的数据形态
	 * @param message 出错提示信息
	 * @param params message占位字符串 辅助判断参数
	 */
	public static void assertNoBadReq(Object obj, Require require, String message, Object... params) {
		String messagePrefix;
		StatusCode status;
		switch (require) {
		case notEmpty:
			messagePrefix = "数据不能没有内容:";
			status = Status.BadReq.noReq;
			break;
		case notNull:
			messagePrefix = "不能没有数据:";
			status = Status.BadReq.illParam;
			break;
		default:
			messagePrefix = "";
			status = Status.BadReq.noReq;
			break;
		}

		assertNoBadReq(JudgeUtil.match(obj, require,params), status, messagePrefix + message, params);
	}
	
	
	/**
	 * 开关状态切换
	 * @param explain 被检查的对象
	 * @param message 出错提示信息
	 * @param params message占位字符串 辅助判断参数
	 */
	public static <T> void assertSwitchStatus(boolean explain, String message, Object... params) {
		if (!explain) {
			throw new BadReqException(Status.BadReq.conflictSwitchStatus, message, params);
		}
	}
	
	
	/**
	 * 判断有没有参数。
	 * @param object
	 * @param message
	 * @param params
	 */
	public static void haveParams(Object object,String message, Object... params){
		assertNoBadReq(object!=null, Status.BadReq.noReq, message, params);
	}
	
	/**
	 * @param explain
	 * @param message
	 * @param params
	 */
	public static void validateParams(boolean explain,String message, Object... params){
		assertNoBadReq(explain, Status.BadReq.illParam, message, params);
	}
	

	// ----------------------------------------------------

	public static void assertPermission(boolean explain, StatusCode status, String message, Object... params) {
		assertStatusSupport(status, PermissionException.class);
		if (!explain) {
			 throw new PermissionException(status, message, params);
		}
	}
	
	
	public static void assertPermission(boolean explain, StatusCode status, String message,Consumer<CustomException> exceptHandle, Object... params) {
		assertMethodRequire(exceptHandle, "exceptHandle");
		assertStatusSupport(status, PermissionException.class);
		if (!explain) {
			PermissionException except = new PermissionException(status, message, params);
			try{
				exceptHandle.accept(except);
			}catch(Throwable e){
				logger.error("在出现错误的时候再此出错",e);
			}
			throw except;
		}
	}
	
	
	/**
	 * 检查数据是不是被存成功了
	 * @param explain 判断表达式 false 抛异常 true不抛出
	 * @param message 出错提示信息
	 * @param params message占位字符串 辅助判断参数
	 */
	public static void assertStoreSuccess(boolean explain, String message, Object... params) {
		if (!explain) {
			throw new HandleStoreException( message, params);
		}
	}
	
	/**
	 * 检查取存在的数据库<br>
	 * 一般来说根据id取数据数据库必须存在，如果存在就是是一个合理的请求<br>
	 * 试想一下，如果这个id对应的数据都不存在，那么用户怎么有这个id呢<br>
	 * @param explain 判断表达式 false 抛异常 true不抛出
	 * @param message 出错提示信息
	 * @param params message占位字符串 辅助判断参数
	 */
	public static void assertGetHaveData(boolean explain, String message, Object... params) {
		if (!explain) {
			throw new BadReqGetNoDataException( message, params);
		}
	}

	/**
	 * 删除一个不存在<br>
	 * 一般来说根据id删除数据数据库必须存在，如果存在就是是一个合理的请求<br>
	 * 试想一下，如果这个id对应的数据都不存在，那么用户怎么有这个id呢<br>
	 * 对应删除个一不存的数据问题不会太大，但还是要明确报告异常<br>
	 * @param explain 判断表达式 false 抛异常 true不抛出
	 * @param message 出错提示信息
	 * @param params message占位字符串 辅助判断参数
	 */
	public static void assertDelHaveData(boolean explain, String message, Object... params) {
		if (!explain) {
			throw new BadReqDelNoDataException( message, params);
		}
	}
	
	/**
	 * 检查程序中有没有存在致命问题,比如配置没被加载，<br>
	 * 如果出现这样的致命问题必须立即停止程序，因为程序跑下去已没什么意义<br>
	 * @param explain 判断表达式 false 抛异常 true不抛出
	 * @param message 出错提示信息
	 * @param params message占位字符串 辅助判断参数
	 */
	public static void assertNotFatal(boolean explain, String message, Object... params) {
		if (!explain) {
			throw new BusyException(message, params);
		}
	}
	
	/**
	 * 检查程序中有没有存在致命问题,比如配置没被加载，<br>
	 * 如果出现这样的致命问题必须立即停止程序，因为程序跑下去已没什么意义<br>
	 * @param explain 判断表达式 false 抛异常 true不抛出
	 * @param message 出错提示信息
	 * @param params message占位字符串 辅助判断参数
	 */
	public static void assertNotFatal(boolean explain,StatusCode busyCode , String message, Object... params) {
		if (!explain) {
			 throw new BusyException(busyCode,message, params);
		}
	}
	

	/**
	 * 致命的bug
	 * @param explain
	 * @param message
	 * @param params
	 */
	public static void assertNotFatalBug(boolean explain , String message, Object... params) {
		message = "碰到这个问题请立即修改程序," + message;
		if (!explain) {
			 throw new ServerException(Status.Server.fatalBug,message, params);
		}
	}
	
	/**
	 * 服务错误之配置错误
	 * @param explain
	 * @param serverCode
	 * @param message
	 * @param params
	 */
	public static void assertNotFatalDataConfig(boolean explain,StatusCode serverCode , String message, Object... params) {
		assertNotFatalBug(serverCode.subOf(Status.Server.dataConfig), "serverCode必须是dataConfig下面子状态码", message);
		if (!explain) {
			throw new ServerException(serverCode,message, params);
		}
	}
	
	/**
	 * 服务错误之程序错误
	 * @param explain
	 * @param serverCode
	 * @param message
	 * @param params
	 */
	public static void assertNotFatalProgramConfig(boolean explain,StatusCode serverCode , String message, Object... params) {
		assertNotFatalBug(serverCode.subOf(Status.Server.programConfig), "serverCode必须是programConfig下面子状态码", message);
		if (!explain) {
			 throw new ServerException(serverCode,message, params);
		}
	}
	
	
	/**
	 * 代码不支持的写法，如果出现这个错误应该立即修改代码。
	 * 如果这里抛错了，直接检查代码吧。
	 * @param explain
	 * @param message
	 * @param params
	 */
	public static void assertSupport(boolean explain, String message, Object... params){
		if (!explain) {
			 throw new UnsupportedException(message, params);
		}
	}


	/**
	 * 断言一个方法必须有当前参数，
	 * TODO 这个断言以后要使用注解等方式来处理。
	 * @param param
	 * @param paramName
	 */
	public static void assertMethodRequire(Object param,String paramName){
		assertSupport(param!=null,"此方法必须传入:[%s]参数",paramName);
	}
	
	/**
	 * 自动根据注解的规则检查参数是不是符合要求
	 */
	public static void assertParamsRequire(){
		//TODO 
	}
	
	
	/**
	 * 断言扩展数据的完整性
	 * @param explain
	 * @param message
	 * @param params
	 */
	public static void assertMapDataComplete(boolean explain, String message, Object... params){
		if (!explain) {
			 throw new BadDataException(Status.BadData.noMapKeyData,message, params);
		}
	}
	
	
	/**
	 * 断言数据是正确的完整的，一般来说，数据库存的数据错了才会出这个问题。
	 * @param explain
	 * @param message
	 * @param params
	 */
	public static void assertDataComplete(boolean explain, String message, Object... params){
		if (!explain) {
			 throw new BadDataException(Status.BadData.DataIncomplete,message, params);
		}
	}


	/**
	 * 断言数据是正确的完整的，一般是来描述缺少非空的属性
	 * @param explain
	 * @param message
	 * @param params
	 */
	public static void assertDataMissFieldRequire(boolean explain, String message, Object... params){
		if(!explain){
			throw new BadDataException(Status.BadData.missRequireFieldData,message, params);
		}
	}

	
	/**
	 * 断言是正常的
	 * @param explain
	 * @param exceptionSupplier
	 */
	public static <E extends CustomException> void assertNormal(boolean explain,Supplier<E> exceptionSupplier){
		AssertUtil.assertMethodRequire(exceptionSupplier, "exceptionSupplier");
		if(!explain){
			throw exceptionSupplier.get();
		}
	}

	/**
	 * 断言是正常的
	 * @param explain
	 * @param exceptionSupplier
	 * @param message
	 * @param params
	 */
	public static <E extends CustomException> void assertNormal(boolean explain,Function<String,E> exceptionSupplier,String message ,Object... params) throws E{
		AssertUtil.assertMethodRequire(exceptionSupplier, "exceptionSupplier");
		if(!explain){
			final String messageFinal = CustomException.format(message,params);
			throw exceptionSupplier.apply(messageFinal);
		}
	}


	//TODO 根据状态码生成异常...........


	/**
	 * 必须不是空，否则是致命错误
	 * @param object
	 * @param message
	 * @param params
	 * @return
	 */
	public static <O> O requireNonNullAsFatal(O object,String message, Object... params){
		simpleAssertByStatus(object!=null, Status.Server.fatalBug, message, params);
		return object;
	}
	
	
	/**
	 * 必须不是空，否则是一个错误的请求
	 * @param object
	 * @param badreq
	 * @param message
	 * @param params
	 * @return
	 */
	public static <O> O requireNonNullAsBadReq(O object,StatusCode badreq, String message, Object... params){
		assertNotFatalBug(badreq.subOf(Status.BadReq.def), "badreq必须是BadReq下面子状态码", message);
		simpleAssertByStatus(object!=null, badreq, message, params);
		return object;
	}
	
	
	/**
	 * 依赖某个对象
	 * @param object
	 * @param status
	 * @param message
	 * @param params
	 * @return
	 */
	public static <O> O requireNonNull(O object,StatusCode status, String message, Object... params){
		simpleAssertByStatus(object!=null, status, message, params);
		return object;
	}
	
	
	/**
	 * 根据状态码生成相关异常
	 * @param explain 断表达式 false 抛异常 true不抛出
	 * @param status 状态码
	 * @param message 出错提示信息
	 * @param params message占位字符串 辅助判断参数
	 */
	public static void simpleAssertByStatus(boolean explain,StatusCode status, String message, Object... params){
		if(explain){
			return;
		}
		int superCode = status.code() - status.code() % baseStatusCode;
		Class<? extends CustomException> customExceptionClass = cacheStatusIntExceptionMap.get(superCode);
		if(BadReqException.class == customExceptionClass){
			throw new BadReqException(status,message, params);
		}else if(BadDataException.class == customExceptionClass){
			throw new BadDataException(status,message, params);
		}else if(BusyException.class == customExceptionClass){
			throw new BusyException(status,message, params);
		}else if(LogicException.class == customExceptionClass){
			throw new LogicException(status,message, params);
		}else if(PermissionException.class == customExceptionClass){
			throw new PermissionException(status,message, params);
		}else if(ThirdException.class == customExceptionClass){
			throw new ThirdException(status,message, params);
		}else if(HandleException.class == customExceptionClass){
			throw new HandleException(status,message, params);
		}else if(ServerException.class == customExceptionClass){
			throw new ServerException(status,message, params);
		}else{
			throw new UnsupportedException(customExceptionClass +  ",statusCode:" + status.code() + message, params);
		}
	}

	private static String format( String message, Object... paramsMsg){
		try{
			return String.format(message, paramsMsg);
		}catch(Throwable e){
			//TODO 这里要去把code修改一下。
			return "出现严重问题需要检查format:[" +message + "],和参数表:"+ Arrays.toString(paramsMsg);
		}
	}
	

	/**
	 * 处理 CustomException 类异常。
	 * */
	public static <E extends CustomException> void handleCustomException(boolean explain, Function<String,E> errorHandle, String message, Object... params) throws E{
		if(!explain){
			final String messageFinal = CustomException.format(message,params);
			throw errorHandle.apply(messageFinal);
		}
	}

	
}
