package com.mini.framework.util.bean;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.lang3.ArrayUtils;

import com.mini.framework.util.asserts.AssertUtil;

/**
 * 集合工具
 * @author jayheo
 */
public class ArrayUtil {
	

	public static Set<Integer> toSetFromInt(int[] ints){
		AssertUtil.assertMethodRequire(ints, "ints");
		return toSetFromInt(ints, ()->new HashSet<>());
	}
	
	/**
	 * 变成set，如果数据为空就使用后面的
	 * @param ints
	 * @param ifNull
	 * @return
	 */
	public static Set<Integer> toSetFromInt(int[] ints,Supplier<Set<Integer>> ifNull){
		if(ints==null || ints.length==0){
			return ifNull.get();
		}else{
			Integer[] arr = ArrayUtils.toObject(ints);
			return Stream.of(arr).collect(Collectors.toSet());
		}
	}

	public static Set<String> toSetFromString(String[] values,Supplier<Set<String>> ifNull){
		if(values==null || values.length==0){
			return ifNull.get();
		}else{
			return Stream.of(values).collect(Collectors.toSet());
		}
	}
	
	public static <T> Set<T> toSet(T[] values,Supplier<Set<T>> ifNull){
		if(values==null || values.length==0){
			return ifNull.get();
		}else{
			return Stream.of(values).collect(Collectors.toSet());
		}
	}
	
	public static <T,R> List<R> mapList(T[] array, Function<T, R> fun){
		return mapList(Stream.of(array).collect(Collectors.toList()),fun);
	}
	
	public static <T,R> List<R> mapList(Collection<T> array, Function<T, R> fun){
		return array.stream().map(fun).collect(Collectors.toList());
	}
	
	
	public static <T,R> Set<R> mapSet(Collection<T> array, Function<T, R> fun){
		return array.stream().map(fun).collect(Collectors.toSet());
	}
	
	
	/**
	 * 从集合中找一个元素
	 * @param condition
	 * @param target
	 * @param conditionGetter
	 * @return
	 */
	public static <C,E> E findElement(C condition,Collection<E> target,Function<E, C> conditionGetter){
		AssertUtil.assertMethodRequire(condition, "condition");
		return target.stream().filter(e->condition.equals(conditionGetter.apply(e))).findFirst().orElse(null);
	}
	
	/**
	 * 从集合中找一个元素的某个属性
	 * @param condition 条件
	 * @param target 目标
	 * @param conditionGetter 条件提取器
	 * @param fieldGetter 属性提取器
	 * @return
	 */
	public static <C,E,F> F findField(C condition,Collection<E> target,Function<E, C> conditionGetter,Function<E, F> fieldGetter){
		AssertUtil.assertMethodRequire(condition, "condition");
		E element = findElement(condition, target, conditionGetter);
		return element==null?null:fieldGetter.apply(element);
	}
	
	/**
	 * 从集合中找一个元素的某个属性
	 * @param conditions
	 * @param target
	 * @param conditionGetter
	 * @param fieldGetter
	 * @return
	 */
	public static <C,E,F> List<F> findFields(Collection<C> conditions,Collection<E> target,Function<E, C> conditionGetter,Function<E, F> fieldGetter){
		return conditions.stream().map(condition->findField(condition, target, conditionGetter, fieldGetter)).filter(f->f!=null).collect(Collectors.toList());
	}
	
	/**
	 * 根据条件找元素
	 * @param conditions 多个条件
	 * @param target 目标
	 * @param conditionGetter 条件提取器(从目标中提取)
	 * @return
	 */
	public static <C,E> List<E> findElements(Collection<C> conditions,Collection<E> target,Function<E, C> conditionGetter){
		return conditions.stream().map(condition->findElement(condition, target, conditionGetter)).filter(f->f!=null).collect(Collectors.toList());
	}
	
	/**
	 * 根据条件找元素
	 * @param reses 多个条件
	 * @param resConditionGetter
	 * @param target
	 * @param conditionGetter
	 * @return
	 */
	public static <R,C,E> List<E> findElements(Collection<R> reses,Function<R, C> resConditionGetter,Collection<E> target,Function<E, C> conditionGetter){
		Collection<C> conditions = reses.stream().map(resConditionGetter).collect(Collectors.toList());
		return conditions .stream().map(condition->findElement(condition, target, conditionGetter)).filter(f->f!=null).collect(Collectors.toList());
	}

	/**
	 * 删除符合条件的元素
	 * @param coll
	 * @param filter
	 */
	public static <T> void remove(Collection<T> coll, Predicate<T> filter) {
		Iterator<T> it = coll.iterator();
		while(it.hasNext()){
			T item = it.next();
			if(filter.test(item)){
				it.remove();
			}
		}
	}
	

	/**
	 * 参考另一个集合的排序规则，规则给出排序器
	 * @see #asListComparator(List, Function, Function)
	 * @param keyList 被参考的集合。他里面的元素必须实现了 Comparable 接口
	 * @param keyOfElement 需要排序的目标元素应该能提取一个与sortList元素类型一致的key，那么这里就是这个  key的提取器
	 * @return
	 */
	public static <E,K> Comparator<E> asListComparator(List<K> keyList,Function<E, K> keyOfElement){
		return asListComparator(keyList, keyOfElement,Function.identity());
	}
	
	
	/**
	 * <pre>
	 * 参考另一个集合的规则给出排序器
	 * 目标  E->Integer
	 * 过程  E->S   S->T(这里是 keyOfSorter的反向 ) T->int(index of list)
	 * </pre>
	 * @param targetList 被参考的集合。
	 * @param keyOfElement 需要排序key提取器，根据元素解析出key
	 * @param keyOfTarget 用作排序的集合的key提取器，根据排序元素得到key
	 * @return
	 */
	public static <E,T,K > Comparator<E> asListComparator(List<T> targetList,Function<E, K> keyOfElement,Function<T, K> keyOfTarget){
		Function<E, Integer> indexOfElement = (element)->{
			K key = keyOfElement.apply(element);
			List<K> keys = targetList.stream().map(keyOfTarget).collect(Collectors.toList());
			return keys.indexOf(key);
		};
		return Comparator.comparing(indexOfElement );
	}
	
	
	/**
	 * 保留符合条件的元素
	 * @param coll
	 * @param filter
	 */
	public static <T> void retain(Collection<T> coll, Predicate<T> filter) {
		remove(coll, filter.negate());
	}

	public static <T> Optional<Set<T>> toOptionalSet(T... ids){
		if(ids==null || ids.length==0){
			return Optional.empty();
		}else{
			return Optional.ofNullable(Stream.of(ids).collect(Collectors.toSet()));
		}
	}

	/**
	 * 找到任务两两相同状态的entry。
	 * @param list
	 * @param compareElement
	 * @param statusComparer
	 * @param sameConsumer
	 * @param <T>
	 * @param <C>
	 */
	public static <T,C> void findAnyBiSameStatusThenConsumer(List<T> list, Function<T,C> compareElement, BiFunction<C,C,Boolean> statusComparer, BiConsumer<T,T> sameConsumer) {
		findAnyBiSameStatus(list,compareElement,statusComparer).ifPresent(entry->sameConsumer.accept(entry.getKey(),entry.getValue()));
	}


	/**
	 * 找到任务两两相同状态的entry。
	 * @param compareElement
	 * @param statusComparer
	 * @param array
	 * @param <T>
	 * @param <C>
	 * @return
	 */
	public static <T,C> Optional<Map.Entry<T,T>> findAnyBiSameStatus( Function<T,C> compareElement, BiFunction<C,C,Boolean> statusComparer,T... array) {
		return findAnyBiSameStatus(Stream.of(array).collect(Collectors.toList()), compareElement,statusComparer);
	}


	/**
	 * 找到任务两两相同状态的entry。
	 * @param list
	 * @param compareElement
	 * @param statusComparer
	 * @param <T>
	 * @param <C>
	 * @return
	 */
	public static <T,C> Optional<Map.Entry<T,T>> findAnyBiSameStatus(List<T> list, Function<T,C> compareElement, BiFunction<C,C,Boolean> statusComparer) {
		if(list.size()>2){
			T first = list.get(0);
			List<T> otherItems = list.subList(1,list.size());
			return otherItems.stream()
					.map(item -> findAnyBiSameStatus(compareElement, statusComparer, first, item))
					.filter(Optional::isPresent).findFirst()
					.orElseGet(()->findAnyBiSameStatus(otherItems, compareElement, statusComparer));
		}else if(list.size()==2){
			C firstCompare = compareElement.apply(list.get(0));
			C secondCompare = compareElement.apply(list.get(1));
			if(statusComparer.apply(firstCompare,secondCompare)){
				return Optional.of(VarUtil.createEntry(list.get(0),list.get(1)));
			}else{
				return Optional.empty();
			}
		}else{
			return Optional.empty();
		}
	}
}
