package com.mini.framework.util.date;


import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.lang3.time.DateUtils;

import com.mini.framework.core.exception.BadReqException;
import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.asserts.Require;
import com.mini.framework.util.date.bean.RelativeDate;
import com.mini.framework.util.date.bean.RelativeDateAliasScheme;

/**
 * 时间工具类，这里禁止重复造轮子，除非在在源工具类中找不到才在这里写。
 * 
 * @see {@link DateUtils}
 * @see {@link DateFormatUtils}
 * @author jayheo
 *
 */
public class DateUtil {

	/**
	 * 生成一个时间栅栏，
	 * @param timeUnit 栅栏间隔。
	 * @param begin 开始时间
	 * @param end 结束时间
	 * @param timeUnit 
	 * @return
	 */
	public static List<TimeSection> createFence(TimeSectionType timeUnit, Date begin, Date end) {

		Date minDate = getCleanedDateUnit(timeUnit, begin);
		Date oneFence = new Date(minDate.getTime());
		List<TimeSection> TimeSectionDtos = new ArrayList<TimeSection>();
		while(oneFence.compareTo(end) <= 0){
			TimeSection TimeSectionDto = new TimeSection(timeUnit, oneFence);
			TimeSectionDtos.add(TimeSectionDto);
			oneFence = addTimeUnit(timeUnit, oneFence, 1);
		}
		return TimeSectionDtos;
	}
	

	public static boolean isDateUnitPoint(TimeSectionType timeSectionType,Date date){
		AssertUtil.assertMethodRequire(date, "date");
		AssertUtil.assertMethodRequire(timeSectionType, "timeSectionType");
		Date newDate = getCleanedDateUnit(timeSectionType, date);
		return newDate.getTime() == date.getTime();
	}
	
	

	/**
	 * 向上取整
	 * @param timeSectionType
	 * @param oldDate
	 * @return
	 */
	public static Date getCleanedTauchDateUnit(TimeSectionType timeSectionType,Date oldDate){
		Date newDate = getCleanedDateUnit(timeSectionType, oldDate);
		if(newDate.compareTo(oldDate)<0){
			newDate = addTimeUnit(timeSectionType, newDate, 1);
		}
		return newDate;
	}
	
	/**
	 * 如果现 是15日 10点，那么相对于天溢出的时间就是10小时对应的毫秒数10*60*60*1000
	 * 得到溢出的时间单位毫秒
	 * @param timeSectionType
	 * @param point
	 * @return
	 */
	public static long getOverflowTime(TimeSectionType timeSectionType,Date point){
		AssertUtil.assertMethodRequire(point, "point");
		AssertUtil.assertMethodRequire(timeSectionType, "timeSectionType");
		return  point.getTime() - DateUtil.getCleanedDateUnit(TimeSectionType.day, point).getTime();
	}
	
	/**
	 * 得到一个时间单元的起点时间
	 * @param oldDate
	 * @return
	 */
	public static Date getCleanedDateUnit(TimeSectionType timeSectionType,Date oldDate,int offset){
		Date cleanedDate = getCleanedDateUnit(timeSectionType, oldDate);
		return addTimeUnit(timeSectionType, cleanedDate, offset);
	}


	/**
	 * 得到一个时间单元的起点时间
	 * @param oldDate
	 * @return
	 */
	public static Date getCleanedDateUnit(TimeSectionType timeSectionType,Date oldDate){
		AssertUtil.assertMethodRequire(timeSectionType, "timeSectionType");
		AssertUtil.assertMethodRequire(oldDate, "oldDate");
		Date cleanedDate = oldDate;
		cleanedDate = DateUtils.setMilliseconds(cleanedDate, 0);//毫秒
		cleanedDate = DateUtils.setSeconds(cleanedDate, 0);//秒
		cleanedDate = DateUtils.setMinutes(cleanedDate, 0);//分
		
		
		switch (timeSectionType) {
		case all :
			cleanedDate = new Date(0);
			break;
		case week:
			cleanedDate = setWeek(cleanedDate, 0);//时
			break;
		case year:
			//TODO 如果日期是的1530360000000L对象的就会出问题，还有很多这样的日期。见  DateUtilTest.setMonthError 可能是时区导致的
			cleanedDate = DateUtils.setMonths(cleanedDate, 0);//月
		case season:
			int seasonMonth = seasonIndex(cleanedDate)*3 ;
			//当从一个大月转到一个小月的时候可能出错，所以先把日期改到第一天
			cleanedDate = DateUtils.setDays(cleanedDate, 1);//天
			cleanedDate = DateUtils.setMonths(cleanedDate, seasonMonth);
		case month:
			cleanedDate = DateUtils.setDays(cleanedDate, 1);//天
		case day:
			cleanedDate = DateUtils.setHours(cleanedDate, 0);//时
		case hour:
			break;
		default:
			throw new BadReqException(Status.BadReq.illParam,"日期%s无效必须在%s中选", timeSectionType,"[hour,day,week,month,year]");
		}
		return cleanedDate;
	}
	
	//Mon Mar 10 00:00:00 CST 1980    一个星期一
	private final static long oneWeekFirst = 321465600000L;  
	
	
	/**
	 * @param date
	 * @param weekIndex 0代表周一
	 * @return
	 */
	public static Date setWeek(Date date, int weekIndex) {
		AssertUtil.assertNoBadReq(weekIndex, Require.between,"日期必须介于[%s,%s)", 0, 7);
		long day = 1000 * 60 * 60 * 24;
		long time = date.getTime() ;
		//找到与星期一的差，然后把当前日期去除余数。
		long clear =( time - oneWeekFirst)%(day*7);
		time -= clear;
		time += day * weekIndex;
		date = new Date(time);
		return date;
	}
	
	public static Date addTimeUnit(TimeSectionType timeSectionType,Date oldDate,int amount){
		 
		Date newDate = null;
		switch (timeSectionType) {
		case hour:
			newDate = DateUtils.addHours(oldDate,amount);
			break;
		case day:
			newDate = DateUtils.addDays(oldDate,amount);
			break;
		case week:
			newDate = DateUtils.addWeeks(oldDate,amount);
			break;
		case month:
			newDate = DateUtils.addMonths(oldDate,amount);
			break;
		case season:
			newDate = DateUtils.addMonths(oldDate,amount*3);
			break;
		case year:
			newDate = DateUtils.addYears(oldDate,amount);
			break;
		default:
			throw new BadReqException(Status.BadReq.illParam,"日期%s无效必须在%s中选", timeSectionType,"[hour,day,week,month,season,year]");
		}
		return newDate;
	}
	
	/**
	 * 得到当前的时间点，去掉，传入的秒种的小数部分，
	 * 如果转的是5的话，结果只是   5 10 15 20 25等5的倍数。
	 * @param seconds
	 * @return
	 */
	public static Date currentFullPoint(int seconds){
		return dateFullPoint(new Date(), seconds);
	}
	
	public static Date dateFullPoint(Date date,int seconds){
		AssertUtil.assertMethodRequire(date	,"date");
		long currentTime = date.getTime();
		return new Date(currentTime - currentTime%(seconds*1000));
	}
	
	/**
	 * 返回当前的时间段
	 * @param timeSectionType
	 * @return
	 */
	public static TimeSection currentTimeSection(TimeSectionType timeSectionType){
		return createTimeSection(new Date(), timeSectionType);
	}
	
	public static TimeSection createTimeSection(Date date,TimeSectionType timeSectionType){
		Date sectionPoint = getCleanedDateUnit(timeSectionType, date);
		return new TimeSection(timeSectionType, sectionPoint);
	}
	
	/**
	 * 显示一个简单的时间
	 * @param date
	 * @return
	 */
	public static String toSimpleDateTime(Date date){
		return DateFormatUtils.format(date, "MM-dd HH:mm");
	}
	
	public static String toMilSecondsString(Date date){
		return DateFormatUtils.format(date, "yyyy-MM-dd HH:mm:ss.SSS");
	}
	
	public static String toSecondsString(Date date){
		return DateFormatUtils.format(date, "yyyy-MM-dd HH:mm:ss");
	}
	
	public static String toSecondsNumeric(Date date){
		return DateFormatUtils.format(date, "yyyyMMddHHmmss");
	}
	
	/**
	 * yyyyMMddHHmmssSSS
	 * @param date
	 * @return
	 */
	public static String toMilSecondsNumeric(Date date){
		return DateFormatUtils.format(date, "yyyyMMddHHmmssSSS");
	}
	

	public static String toDateString(DateStringFormat dsf){
		return toDateString(dsf, new Date());
	}
	
	public static String toDateString(DateStringFormat dsf, Date date){
		dsf.getDatePattern();
		String dateString = DateFormatUtils.format(date, dsf.getDatePattern());
		String string = String.format(dsf.getStringPattern(), dateString);
		return string;
	}
	

	/**
	 *  时间段转成英文简写
	 * @param date
	 * @param type
	 * @return
	 */
	public static String toDateSectionEnglishShort(Date date,TimeSectionType type){
		TimeSection scetion = createTimeSection(date, type);
		return toDateSectionEnglishShort(scetion);
	}
		
	
	/**
	 * 时间段转成英文简写
	 * @param timeSection
	 * @return
	 */
	public static String toDateSectionEnglishShort(TimeSection timeSection){
		AssertUtil.assertMethodRequire(timeSection,"timeSection");
		timeSection.validate();
		Map<TimeSectionType, String> sectionFormat = new HashMap<>();
		sectionFormat.put(TimeSectionType.year, "yyyy");
		sectionFormat.put(TimeSectionType.month, "yyyy-MM");
		//sectionFormat.put(TimeSectionType.week, "yyyy");
		sectionFormat.put(TimeSectionType.day, "MM-dd");
		sectionFormat.put(TimeSectionType.hour, "MM-dd HH");
		String format = sectionFormat.get(timeSection.getTimeUnit());
		AssertUtil.assertSupport(format!=null, "不支持%s类型的时间简写", timeSection.getTimeUnit());
		return DateFormatUtils.format(timeSection.getSectionPoint(), format);
	}	
	

	/**
	 * 转成时间段简写
	 * @param list
	 * @param showDate
	 * @param timeUnit
	 * @return
	 */
	public static <T> List<DateShortString> toDateSectionEnglishShort(Collection<T> list,Function<T, Date> showDate,TimeSectionType timeUnit){
		return list.stream().map(showDate).map(date->new DateShortString(date,DateUtil.toDateSectionEnglishShort(date, timeUnit)))
		.collect(Collectors.toList());
		
	}
	
	/**
	 * 转化到中文的时间之日期简写
	 * @param list
	 * @param showDate
	 * @return
	 */
	public static <T> List<DateShortString> toDateSectionChinaDateShort(Collection<T> list,Function<T, Date> showDate){
		return list.stream().map(showDate).map(date->new DateShortString(date,toDateSectionChinaDateShort(date)))
		.collect(Collectors.toList());
	}
	
	/**
	 * 转化到中文的时间之日期简写
	 * @param date
	 * @return
	 */
	public static String toDateSectionChinaDateShort(Date date){
		return DateFormatUtils.format(date, "yyyy年MM月dd日");
	}
	
	/**
	 * 目前每周以星期1为起点
	 * @param date
	 * @param minType
	 * @return
	 */
	public static String simpleWeekDateRetainKeyPoint(Date date,TimeSectionType minType){
		AssertUtil.assertMethodRequire(minType, "minType");
		AssertUtil.assertMethodRequire(date, "date");
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		calendar.setFirstDayOfWeek(Calendar.MONDAY);
		int index = calendar.get(Calendar.WEEK_OF_YEAR);
		//为了保证第一周在年前的情况下不出现错误，往后推了1周去算年份。
		Date weekDate = addTimeUnit(TimeSectionType.week, date, 1);
		String year = index == 1? DateFormatUtils.format(weekDate, "yyyy年") :"";
		return String.format("%s第%s周",year, index);
	}
	
	/**
	 * 简化时间，但是保留关键时间点
	 * 如果要看结果去跑测试用例simpleDateRetainKeyPoint
	 * @param date
	 * @param minType
	 * @return
	 */
	public static String simpleDateRetainKeyPoint(Date date,TimeSectionType minType){
		AssertUtil.assertMethodRequire(minType, "minType");
		AssertUtil.assertMethodRequire(date, "date");
		if(minType.equals(TimeSectionType.week)){
			return simpleWeekDateRetainKeyPoint(date, minType);
		}else if(minType.equals(TimeSectionType.season)){
			return "TODO";//TODO 这个要优化一下改一下文字
		}
		
		
		
		Map<TimeSectionType, Map<String, String>> timeTypeRange = new LinkedHashMap<TimeSectionType, Map<String,String>>();
		//TODO 可以使用时间scopeitem优化
		BiFunction<String,String,Map<String, String>> ss = (k,v) ->{
			Map<String, String> m = new HashMap<>();
			m.put(k,v);
			return m;
		};
		timeTypeRange.put(TimeSectionType.hour	, ss.apply("HH时", "00时"));
		timeTypeRange.put(TimeSectionType.day	, ss.apply("dd日", "01日"));
		timeTypeRange.put(TimeSectionType.month	, ss.apply("MM月", "01月"));
		timeTypeRange.put(TimeSectionType.year	, ss.apply("yyyy年", "3000年"));
		AssertUtil.assertSupport(timeTypeRange.containsKey(minType), "minType:[%s]必须在%s内",minType,timeTypeRange.keySet());
		
		Iterator<Entry<TimeSectionType, Map<String, String>>> rit = timeTypeRange.entrySet().iterator();
		while(rit.hasNext()){
		Entry<TimeSectionType, Map<String, String>> timeType  = rit.next();
			if(timeType.getKey().equals(minType)){
				break;
			}else{
				rit.remove();
			}
		}
		Function<TimeSectionType,Map<String, String>> typeToPair = type-> timeTypeRange.get(type);
		Function<Map<String, String>, String> pairToPattern = pair ->pair.entrySet().stream().findFirst().get().getKey();
		Function<Map<String, String>, String> pairToPoint = pair -> pair.entrySet().stream().findFirst().get().getValue(); 
		Function<String, String> patternToResult = p -> DateFormatUtils.format(date, p);

		String result = "";
		boolean requireMore = true;
		String preResult = typeToPair.andThen(pairToPattern).andThen(patternToResult).apply(minType);
		String prePoint = typeToPair.andThen(pairToPoint).apply(minType);
		Iterator<Entry<TimeSectionType, Map<String, String>>> it = timeTypeRange.entrySet().iterator() ;
		while (it.hasNext() && requireMore) {
			Entry<TimeSectionType, Map<String, String>> entry = it.next();
			preResult = typeToPair.andThen(pairToPattern).andThen(patternToResult).apply(entry.getKey());
			prePoint = typeToPair.andThen(pairToPoint).apply(entry.getKey());
			result = preResult + result;
			requireMore = prePoint.equals(preResult);
		}
		return result;
	}
	
	
	/**
	 * 返回季节的索引 0 1 2 3分别代表是Q1 Q2 Q3 Q4
	 * @param date
	 * @return
	 */
	public static int seasonIndex(Date date){
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		int monthIndex = calendar.get(Calendar.MONTH);
		int seasonMonth =monthIndex/3 ;
		return seasonMonth;
	}
	
	/**
	 * "Q1春季","Q2夏季","Q3秋季","Q4冬季"
	 * @param date
	 * @return
	 */
	public static String indexSeasonNameQsort(Date date){
		String[] seasonNames = new String[]{"Q1春季","Q2夏季","Q3秋季","Q4冬季"};
		int seasonIndex = DateUtil. seasonIndex(date);
		return seasonNames[seasonIndex];
	}

	public static String indexDayOfWeekSimpleName(Date date){
		String englishName = DateFormatUtils.format(date,"E");
		return englishName;
		/*int index = indexWeekOfYear(date);
		String[] weekNames = new String[]{"星期一","星期二","星期三","星期四","星期五","星期六","星期日"};
		return weekNames[index];*/
	}
	/*
	AssertUtil.assertNotFatalBug(indexs.get(weekIndexName)!=null, "找不到[%s]对应的值在indexs中 %s", weekIndexName,indexs);
	indexs.put("0", 0);
	indexs.put("1", 1);
	indexs.put("2", 2);
	indexs.put("3", 3);
	indexs.put("4", 3);
	indexs.put("5", 4);
	indexs.put("6", 5);
	indexs.put("7", 5);*/
	
	/**
	 * 
	 * @param date
	 * @return
	 */
	public static int indexWeekOfYear(Date date){
		String weekIndexName = DateFormatUtils.format(date, "w");
		return Integer.parseInt(weekIndexName);
	}
	
	
	public static int indexDayOfMonth(Date date){
		String dayIndexName = DateFormatUtils.format(date, "d");
		return Integer.parseInt(dayIndexName);
	}
	
	
	/**
	 * 看一个月中第几周
	 * @param date
	 * @return
	 */
	public static int indexWeekOfMonth(Date date){
		date = new Date(date.getTime());
		if(indexDayOfMonth(date)<15){
			Date nextWeek = DateUtil.addTimeUnit(TimeSectionType.week, date, 1);
			int nextWeekDayIndex = indexDayOfMonth(setWeek(nextWeek,0));
			return nextWeekDayIndex / 7 ;
		}else{
			Date nextWeek = DateUtil.addTimeUnit(TimeSectionType.week, date, -1);
			int nextWeekDayIndex = indexDayOfMonth(setWeek(nextWeek,0));
			return nextWeekDayIndex / 7 + 2;
		}
	}
	
	
	public static  List<RelativeDate> relativeNow1 (Stream<Date> actualDates) {
		return RelativeDate.countRelatives(RelativeDateAliasScheme.nowScheme1, new Date(), actualDates);
	}
	
	public static  List<RelativeDate> relativeNow1(List<Date> actualDates) {
		return RelativeDate.countRelatives(RelativeDateAliasScheme.nowScheme1, new Date(), actualDates.stream());
	}
	
	public static <D>  List<RelativeDate> relativeNow1(List<D> actualDates,Function<D, Date> dateGetter) {
		return RelativeDate.countRelatives(RelativeDateAliasScheme.nowScheme1, new Date(),actualDates.stream().map(dateGetter));
	}
	
	
	
	//TODO 提供相对时间的简要说明，比如  几秒以前，几分钟以前，一周以前，二周以前， 几月以前。
	
}
