package com.mini.framework.util.function;

import com.mini.framework.core.exception.ServerException;
import com.mini.framework.core.status.Status;
import com.mini.framework.core.status.StatusCode;
import com.mini.framework.util.asserts.AssertUtil;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Stream;

/**
 * @author jayheo
 */
public class FinderStream<T> {

    protected final Predicate<T> condition;

    protected Map<Supplier<T>,Optional<Consumer<Integer>>> finderLink = new LinkedHashMap<>();

    protected FinderStream(Predicate<T> condition){
        AssertUtil.assertMethodRequire(condition,"condition");
        this.condition = condition;
    }

    /**
     * 查询为true的
     * */
    public static FinderStream<Boolean>findTrue(){
        return of(Boolean.TRUE::equals);
    }

    /**
     * 查询为true的
     * */
    public static FinderStream<Boolean>findFalse(){
        return of(Boolean.FALSE::equals);
    }

    /**
     * 查询存在的bean
     * */
    public static <T>FinderStream<T>findBean(Predicate<T> condition){
        return of(Objects::nonNull);
    }

    /**
     * 生成查找流
     * */
    public static <T>FinderStream<T>of(Predicate<T> condition){
        return new FinderStream<>(condition);
    }

    /**
     * 查询是空的
     * */
    public static <T>FinderStream<T>requireIsNull(T bean){
        return of(Objects::isNull);
    }

    /**
     * 查询不是空的
     * */
    public static <T>FinderStream<T> requireNotNull(T bean){
        return of(Objects::nonNull);
    }

    /**
     * 下一个条件
     * */
    public FinderStream<T> nextOptional(Optional<T> optional){
        return next(()->optional.orElse(null),Optional.empty());
    }

    /**
     * 下一个条件
     * */
    public FinderStream<T> nextOptional(Optional<T> optional,Consumer<Integer> ifMiss){
        AssertUtil.assertMethodRequire(ifMiss,"ifMiss");
        return nextOptional(()->optional,ifMiss);
    }

    /**
     * 下一个条件
     * */
    public FinderStream<T> nextOptional(Supplier<Optional<T>> optionalSupplier){
        return next(()-> optionalSupplier.get().orElse(null),Optional.empty());
    }

    /**
     * 下一个条件
     * */
    public FinderStream<T> next(Supplier<T> supplier){
        return next(supplier,Optional.empty());
    }

    /**
     * 下一个条件,如果不匹配要做一点事情
     * */
    protected FinderStream<T> next(Supplier<T> supplier,Optional<Consumer<Integer>> ifMissOptional){
        AssertUtil.assertMethodRequire(supplier,"supplier");
        AssertUtil.assertMethodRequire(ifMissOptional,"ifMissOptional");
        AssertUtil.assertNotFatalBug(!finderLink.containsKey(supplier), "supplier出现重复,索引:[%s]",finderLink.size());
        finderLink.put(supplier,ifMissOptional);
        return this;
    }


    /**
     * 下一个条件,如果不匹配要做一点事情
     * */
    public FinderStream<T> nextOptional(Supplier<Optional<T>> optionalSupplier,Consumer<Integer> ifMiss){
        AssertUtil.assertMethodRequire(ifMiss,"ifMiss");
        return next(()->optionalSupplier.get().orElse(null),Optional.of(ifMiss));
    }


    /**
     * 下一个条件,如果不匹配要做一点事情
     * */
    public FinderStream<T> next(T bean,Consumer<Integer> ifMiss){
        return next(()->bean,Optional.of(ifMiss));
    }


    /**
     * 下一个条件,如果不匹配要做一点事情
     * */
    public FinderStream<T> next(Supplier<T> supplier,Consumer<Integer> ifMiss){
        return next(supplier,Optional.of(ifMiss));
    }


    /**
     * 断言可以找到
     * */
    public T foundBean(StatusCode errorStatus, String message, Object... params){
        Optional<T> resultOptional = find();
        AssertUtil.simpleAssertByStatus(resultOptional.isPresent(),errorStatus,message,params);
        //  return resultOptional.get(); 这里为什么不使用get() 仔细想想
        return resultOptional
                .orElseThrow(()->new ServerException(Status.Server.fatalBug,"框架出现bug,前端断言出错了"));
    }


    /**
     * 查找对应的值
     * */
    public Optional<T> find(){
        return findMatchConditionAsSupplier().map(Supplier::get);
    }

    /**
     * 查找对应的值
     * */
    public T findOrElse(T bean){
        return find().orElse(bean);
    }


    /**
     * 找到匹配的转成提供者
     * */
    public Optional<Supplier<T>> findMatchConditionAsSupplier(){
        ArrayList<Map.Entry<Supplier<T>, Optional<Consumer<Integer>>>> finderEntries = new ArrayList<>(finderLink.entrySet());
        for (int index = 0; index <finderEntries.size() ; index++) {
            int findIndex = index;
            Map.Entry<Supplier<T>, Optional<Consumer<Integer>>> finderEntry = finderEntries.get(index);
            Supplier<T> finder = finderEntry.getKey();
            Optional<Consumer<Integer>> onAbsentOptional = finderEntry.getValue();
            T found = finder.get();
            if(condition.test(found)){
                return Optional.of(()->found);
            }else{
                onAbsentOptional.ifPresent(onAbsent->onAbsent.accept(findIndex));
            }
        }
        return Optional.empty();
    }

}
