package com.mini.framework.util.function;

import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;

/**
 * 对Predicate的扩展
 * 
 * @author jayheo
 *
 */
public interface PredicateSpread<T> extends Predicate<T> {

	/**
	 * 检查对象是不是相等的判断器，类型必须一样。
	 * @param targetRef
	 * @return
	 */
	static <T> Predicate<T> isEqual( T targetRef) {
		return  Predicate.isEqual((Object)targetRef);
	}

	/**
	 * 检查对象是不是相等的判断器
	 * 类型可以不一样，
	 * 可以解构对比
	 * @param fieldRef 从元素中取出要比的对应。
	 * @param targetRef 目标对像
	 * @return
	 */
	static <T, F> Predicate<T> isEqual(Function<T, F> fieldRef, F targetRef) {
		Predicate<F> predicate = Predicate.isEqual(targetRef);
		return object -> predicate.test(fieldRef.apply(object));
	}
	
	/**
	 * 返回一个一定是正确的测试器
	 * @return
	 */
	static <T> PredicateSpread<T> right(){
		return t->true;
	}
	
	static <T> PredicateSpread<T> incorrect(){
		return t->false;
	}
	
	
	/**
	 * 转成 PredicateSpread
	 * @param predicate
	 * @return
	 */
	static <T> PredicateSpread<T> of(Predicate<T> predicate){
		return predicate::test;
	}


	/**
	 * 转成 PredicateSpread 变反转
	 * @param predicate
	 * @param <T>
	 * @return
	 */
	static <T> PredicateSpread<T> ofNegate(Predicate<T> predicate){
		return predicate.negate()::test;
	}

	
	static <T> PredicateSpread<T> of(Optional<Boolean> right){
		return of(right.orElse(true));
	}
	
	static <T> PredicateSpread<T> of(boolean right){
		if(right){
			return PredicateSpread.right();
		}else{
			return PredicateSpread.incorrect();
		}
	}

	
	/**
	 * 参考下一个对象对比的结果
	 * @param nextGetter 拿到下一个对象
	 * @param nextTest 下一个对象对比的比较器
	 * @return
	 */
	static <T, F> PredicateSpread<T> nextTest(Function<T, F> nextGetter,Predicate<F> nextTest){
		return t->nextTest.test( nextGetter.apply(t));
	}
	
	/**
	 * 匹配测试
	 * @param matchFn  匹配的方法
	 * @param goal 匹配的数据
	 * @return
	 */
	static <T,G> PredicateSpread<T> matchTest(BiFunction<T,G,Boolean> matchFn,G goal){
		return t-> matchFn.apply(t,goal);
	}
	
	
	/**
	 * 应用第二个参数
	 * @param secondParam
	 * @param applySecondParamFn
	 * @return
	 */
	static<T,P> PredicateSpread<T> applySecondParam(P secondParam,BiFunction<T, P, Boolean> applySecondParamFn){
		return t-> applySecondParamFn.apply(t,secondParam);
	}
}
