package com.mini.framework.util.http.full;

import com.mini.framework.core.exception.HandleIOException;
import com.mini.framework.core.exception.ServerException;
import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.params.MapParams;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.http.HttpEntity;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.protocol.HttpContext;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author jayheo
 */
public class FullHttpRequest {

    private HttpMethod method;

    private String url;

    public FullHttpRequest(HttpMethod method, String url) {
        this.method = method;
        this.url = url;
    }

    private HttpEntity asEntity(String body){
        return new StringEntity(body,"utf-8");
    }

    private HttpEntity asEntity(MapParams queries){
        List<NameValuePair> list = queries.toStringMap().entrySet().stream()
                .map(entry->new BasicNameValuePair(entry.getKey(),entry.getValue()))
        .collect(Collectors.toList());
        try {
            return new UrlEncodedFormEntity(list,"utf-8");
        } catch (UnsupportedEncodingException e) {
            throw new ServerException(e,"生成url encode参数错误 params:%s",queries.toStringMap());
        }
    }


    private HttpRequestBase asRequest(Map<String,String> queries, Optional<String> bodyOptional) {
        return asRequest(MapParams.build().params(queries),bodyOptional);
    }
    private HttpRequestBase asRequest(MapParams queries, Optional<String> bodyOptional) {
        String urlWithParams = queries.toUrlParams(url);

        Function<HttpEntityEnclosingRequestBase,HttpRequestBase> useBodyOptional = request->{
            bodyOptional.map(this::asEntity).ifPresent(request::setEntity);
            return request;
        };

        Function<HttpEntityEnclosingRequestBase,HttpRequestBase> bodyAttachQueries = request->{
            request.setEntity(asEntity(queries));
            return request;
        };

        switch (method) {
            case DELETE:
                return setConfig(new HttpDelete(urlWithParams));
            case GET:
                return setConfig(new HttpGet(urlWithParams));

            case PUT:
                HttpPut put = new HttpPut(urlWithParams);
                useBodyOptional.apply(put);
                return setConfig(put);
            case POST:
                HttpPost post = new HttpPost(urlWithParams);
                useBodyOptional.apply(post);
                return setConfig(post);
            default:
                throw new ServerException(Status.Server.fatalBug,"不知道的method:%s", method);
        }
    }

    private static HttpRequestBase setConfig(HttpRequestBase request){
        request.setConfig(createDefaultRequestConfig());
        return request;
    }

    private static RequestConfig createDefaultRequestConfig() {
        RequestConfig.Builder builder = RequestConfig.custom();
        //连接建立时间，三次握手完成时间
        builder.setConnectTimeout(6000);
        //httpclient使用连接池来管理连接，这个时间就是从连接池获取连接的超时时间，可以想象下数据库连接池
        builder.setConnectionRequestTimeout(60000);
        //数据传输过程中数据包之间间隔的最大时间
        builder.setSocketTimeout(12000);
        builder.setMaxRedirects(5);
        return builder.build();
    }


    public FullHttpResponse execute(Map<String, String> headers, Map<String, String> queries) {
        return execute(headers, queries, Optional.empty());
    }
    public FullHttpResponse execute(Map<String, String> headers, Map<String, String> queries,Optional<String> bodyOptional) {
        AssertUtil.assertMethodRequire(headers,"headers");
        AssertUtil.assertMethodRequire(queries,"queries");
        CloseableHttpClient closeableHttpClient = HttpClientBuilder.create().build();
        HttpRequestBase request = asRequest(queries,bodyOptional);
        headers.forEach(request::setHeader);
        HttpContext context = new BasicHttpContext();
        queries.forEach(context::setAttribute);
        try {
            CloseableHttpResponse response = closeableHttpClient.execute(request,context);
            int responseStatus = response.getStatusLine().getStatusCode();
            HashMap<String,String> responseHeaders = new HashMap<>(20);
            Stream.of(response.getAllHeaders()).forEach(header->responseHeaders.put(header.getName(),header.getValue()));
            InputStream content = response.getEntity().getContent();
            String responseBody = IOUtils.toString(content, "utf-8");
            return new FullHttpResponse(this,responseStatus,responseHeaders,responseBody);
        } catch (IOException e) {
            throw new HandleIOException(e, "在发http:[%s]请求时出错",this);
        }
    }

    public FullHttpRequest setMethod(HttpMethod method) {
        this.method = method;
        return this;
    }

    public String getUrl() {
        return url;
    }

    public FullHttpRequest setUrl(String url) {
        this.url = url;
        return this;
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }
}
