package com.mini.framework.util.log.trace;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.lang.math.RandomUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.mini.framework.util.log.MiniLogLevel;

/**
 * @author jayheo
 * 方法和特征映射缓存器。
 */
public class MethodFeatureCache {
	private Logger logger = LogManager.getLogger(MethodFeatureCache.class);
	private MapStorager mapStorager;
	private static Map<String, String> featureMapMethod;
	private MethodFeatureResolver methodFeatureResolver = new SimpleMethodFeatureResolver();
	
	/**
	 * 需要传入一个可用的存储器<br>
	 * @param mapStorager
	 */
	public MethodFeatureCache(MapStorager mapStorager) {
		super();
		this.mapStorager = mapStorager;
	}
	/**
	 * 根据特征得到跟踪id 6位<br>
	 * @param feature 特征
	 * @return
	 */
	public String loadFeature(String feature){
		if(featureMapMethod==null){//如果还没被实例化就返回null
			return null;
		}
		String traceId = featureMapMethod.get(feature);
		if(traceId == null){//双判断防止性能问题
			synchronized(featureMapMethod){
				traceId = featureMapMethod.get(feature);
				if(traceId == null){
					traceId = register(feature);
				}
			}
		}
		return traceId;
	}
	/**
	 * 根据特方法征得到跟踪id 6位<br>
	 * @param method 方法对象
	 * @param methodFeatureResolver 方法解析器
	 * @return
	 */
	public String loadFeature(Method method,MethodFeatureResolver methodFeatureResolver){
		String feature = methodFeatureResolver.resolveFeature(method);
		return loadFeature(feature);
	}
	
	/**
	 * 根据方法拿到特征 key
	 * @param method
	 * @return
	 */
	public String loadFeature(Method method){
		return this.loadFeature(method, methodFeatureResolver);
	}
	
	/**
	 * 注册一个特征
	 * @param feature 特征
	 * @return
	 */
	private String register(String feature) {
		boolean contain = false;
		String traceId = null;
		do{
			int size = featureMapMethod.size();
			traceId = String.valueOf(300000 + size);
			contain = featureMapMethod.containsValue(traceId);
			if(contain){
				continue;
			}else{
				featureMapMethod.put(feature, traceId);
				slowCacheFeatureMapMethod();
				break;
			}
		}while(contain);
		return traceId;
	}
	
	private boolean toCacheFeatureMapMethod;
	
	/**
	 * 缓慢的缓存map,不宜操作得太频繁<br>
	 * 异步操作同步到库<br>
	 */
	private void slowCacheFeatureMapMethod(){
		final long timeMark = System.currentTimeMillis();
		
		toCacheFeatureMapMethod = true;
		Thread thread = new Thread() {  
            @Override  
            public void run() {
            	LinkTraceFace.pointSystemQuartz();
                int delay = 2000 + RandomUtils.nextInt(8000);
                try {
					Thread.sleep(delay);
				} catch (InterruptedException e) {
					logger.error("延迟出错",e);
					if(timeMark + 300 < System.currentTimeMillis() ){
						//如果在300ms内被重新唤醒那么忽略下面的事情。以免死循环
						return;
					}
				}
            	if(toCacheFeatureMapMethod){
            		logger.log(MiniLogLevel.getFrameWorkLog(), "缓存featureMapMethod到数据库size:" + featureMapMethod.size());
            		Map<String, String> dbMapStorager = mapStorager.get();
            		Map<String, String> temp = new ConcurrentHashMap<>(featureMapMethod.size());
            		temp.putAll(dbMapStorager);//先找到db的
            		temp.putAll(featureMapMethod);//再用现有的覆盖数据库上的
            		featureMapMethod = temp;
            		mapStorager.set(featureMapMethod);
            		toCacheFeatureMapMethod = false;
            	}
            	LinkTraceFace.clearTrace();
            }  
        };
        thread.start();
	}

	/**
	 * 初始化数据库的数据<br>
	 */
	public void init(){
		featureMapMethod  = new ConcurrentHashMap<String, String>();
		Map<String, String> dbMap = this.mapStorager.get();
		logger.log(MiniLogLevel.getFrameWorkLog(),"从mapStorager得到原有的method map:" + dbMap);
		if(dbMap!=null){
			featureMapMethod.putAll(dbMap);
		}
		
		if(methodFeatureResolver==null){
			methodFeatureResolver = new SimpleMethodFeatureResolver();
		}
	}
	
	/**
	 * 把链接的特征显示出来<br>
	 * @return
	 */
	public Map<String, String> showLinkFeatures(){
		HashMap<String, String> linkFeature = new HashMap<>();
		linkFeature.putAll(featureMapMethod);
		return linkFeature;
	}
}
