package com.mini.framework.util.relation;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;

/**
 * 我们拒绝自己造轮子。
 * 实在找不到实现方法时我们才考虑造一下轮子。
 * 这里做一下集合特殊处理
 * */
public class ShipUtil {
	/**
	 * 例如去除集合中的所有副本,只要集合中A是B的副本，那么 A就会被移除
	 * @param coll 需要处理的集合。
	 * */
	public static  <T> void removeDeputy(Collection<T> coll,Relateder<? super T> relateder){
		removeOnRelation(coll, relateder, Relateder.DEFAULT_DEPUTY_RELATION);
	}
	
	
	/**
	 * 根据一个key去除重复的
	 * 一般是排在后面的会被去掉
	 * @param coll
	 * @param keyGetter 用于对比的key 如果是null会只留下一下
	 */
	public static  <T,K> void distinct(Collection<T> coll,Function<T, K> keyGetter){
		removeAlike(coll,new Flagable<T>() {

			@Override
			public String showFlag(T t) {
				return String.valueOf(keyGetter.apply(t));
			}
		}, 1);
	}
	
	
	/**
	 * 全量比较集合中的每对两两关系,这个关系会在调用的时候传入，如果A-B得关系等于传入的值，那么A会被移除
	 * eg : @see {@link CollUtil#removeDeputy(Collection, Relateder)}
	 * */
	public static  <T> void removeOnRelation(Collection<T> coll,Relateder<? super T> relateder,int relation){
		boolean change = true;
		while (change && coll.size() > 0){
			change = true;
			Iterator<T> it1 = coll.iterator();
			T toRemove = null;
			boolean conFlag = true;
			while (it1.hasNext()&& conFlag) {
				T element1 = it1.next();
				Iterator<T> it2 = coll.iterator();
				while (it2.hasNext() && conFlag) {
					T element2 = it2.next();
					if(relateder.showRelation(element1,element2)== relation){
						toRemove = element1;
						conFlag = false;
						break;
					}
				}
			}
			change = toRemove != null;
			if(change){
				coll.remove(toRemove);
			}
		}
	}
	
	
	/** 
	* @Title: distinctString 
	* @Description: 把String集合去除重复
	* @param @param coll    设定文件 
	* @return void    返回类型 
	* @author jayheo
	* @throws 
	*/
	public static void distinctString(Collection<String> coll){
		Flagable<String> flagable = new Flagable<String>() {
			
			@Override
			public String showFlag(String t) {
				return t;
			}
		};
		removeAlike(coll, flagable , 1);
	}
	
	
	/** 
	* @Title: removeAlike 
	* @Description: 去掉相似的。顺序找一下去，记录相似的的数据直到达到timeLimit后剩下的就从集合中去除，
	* 这里要求集合是有序的。否则结果是不确定的
	* @param @param coll
	* @param @param flagable
	* @param @param timeLimit    允许相似的的次数
	* @return void    返回类型 
	* @author jayheo
	* @throws 
	*/
	public static  <T> void removeAlike(Collection<T> coll,Flagable<? super T> flagable,int timeLimit){
		Map<String, Integer> timeMap = new HashMap<String, Integer>();
		Iterator<T> it = coll.iterator();
		while (it.hasNext()) {
			T t = it.next();
			String flag = flagable.showFlag(t);
			Integer haveTime = timeMap.get(flag);
			if( haveTime == null){
				timeMap.put(flag, 1);
				continue;
			}else if( haveTime < timeLimit){
				timeMap.put(flag, haveTime + 1);
				continue;
			}else{
				it.remove();
			}
		}
	}
	
	/** 
	 * @Title: tranCascadeTree 
	 * @Description: 把集合转变成为级联树
	 * @param @param cascadables   集合
	 * @return void    返回类型 
	 * @author jayheo
	 * @throws 
	 */
	public static <T> void tranCascadeTree(Collection<? extends Cascadable<T>> cascadables){
		List<Cascadable<T>> superTemps = new ArrayList<>(cascadables);
		for (Cascadable<T> superTemp : superTemps) {
			Iterator<? extends Cascadable<T>> it = cascadables.iterator();
			while (it.hasNext()) {
				Cascadable<T> cascadable = it.next();
				if(cascadable.showSuperNodeKey()!=null && cascadable.showSuperNodeKey().equals(superTemp.showNodeKey())){
					it.remove();
					superTemp.addSonNode((T)cascadable);
				}
			}
		}
	}
	
	/**
	 * 把树状拉平成集合
	 * @param cascadable
	 * @return
	 */
	public static <T> List<T> tranTreeToList(Cascadable<T> cascadable){
		List<T> result = cascadable.showSonNodes();
		result.add((T)cascadable);
		return result;
		
	}
	
	/**
	 * 把树状拉平成集合
	 * @param cascadables
	 */
	public static <T> void tranTreeToList(List<? extends Cascadable<T>> cascadables){
		for (int i = 0; i < cascadables.size(); i++) {
			Cascadable<T> cascadable = cascadables.get(i);
			Iterator<T> sonNodeIt = cascadable.showSonNodes().iterator();
			while (sonNodeIt.hasNext()) {
				T sonNode =  sonNodeIt.next();
				((List<T>)cascadables).add(sonNode);
				sonNodeIt.remove();
			}
		}
	}
	
	
	
	
	/** 
	* @Title: tranCascadeTree 
	* @Description: 把集合转变成为级联树
	* @param @param cascadables   集合
	* @return void    返回类型 
	* @author jayheo
	* @throws 
	*/
	public static <T> void tranCascadeTree(Collection<T> targColl,Cascader<T> cascader){
		List<T> superTemps = new ArrayList<>(targColl);
		for (T superTemp : superTemps) {
			Iterator<T> it = targColl.iterator();
			while (it.hasNext()) {
				T cascadable = it.next();
				if(cascader.showSuperNodeKey(cascadable)!=null && cascader.showSuperNodeKey(cascadable).equals(cascader.showNodeKey(superTemp))){
					it.remove();
					cascader.addSubNodes(superTemp,cascadable);
				}
			}
		}
	}
	
	
	
	/**
	 * 根据nodeKey查询联中的节点
	 * @param cascadables
	 * @param nodeKey
	 * @return
	 */
	public static <T> T findByNodeKey(Collection<? extends Cascadable<T>> cascadables,String nodeKey){
		AssertUtil.assertAllNotNull(Status.BadReq.def, "级联[%s]和nodeKey[%s]都不能为空", cascadables,nodeKey);
		T found = null;
		Iterator<? extends Cascadable<T>> it = cascadables.iterator();
		while (it.hasNext()) {
			Cascadable<T> cascadable = it.next();
			found = findByNodeKey(cascadable, nodeKey);
			if(found!=null){
				break;
			}
		}
		return found;
	}
	
	/**
	 * 根据nodeKey查询联中的节点
	 * @param cascadables
	 * @param nodeKey
	 * @return
	 */
	public static <T> T findByNodeKey(Cascadable<T> cascadable,String nodeKey){
		AssertUtil.assertAllNotNull(Status.BadReq.def, "级联[%s]和nodeKey[%s]都不能为空", cascadable,nodeKey);
		if(nodeKey.equals(cascadable.showNodeKey())){
			return (T)cascadable;
		}
		return (T)findByNodeKey((Collection<? extends Cascadable<T>>) cascadable.showSonNodes(),nodeKey);
	}

	
	public static <T, I extends Comparable<I>, F> void indexSort(List<T> list, Function<T, F> field, Map<F, I> mapper) {
		indexSort(list, (t) -> mapper.get(field.apply(t)));
	}

	/**
	 * 根据索引排序
	 * 
	 * @param list
	 * @param index
	 *            得到可以比较的索引
	 */
	public static <T, I extends Comparable<I>> void indexSort(List<T> list, Function<T, I> index) {
		Comparator<? super T> comp = new Comparator<T>() {
			@Override
			public int compare(T o1, T o2) {
				return index.apply(o1).compareTo(index.apply(o2));
			}
		};
		list.sort(comp);
	}

	/**
	 * 对于实现了indexable元素直接根据排序
	 * @param list
	 */
	public static <T extends Indexable<I>, I extends Comparable<I>> void indexSort(List<T> list) {
		indexSort(list, (t)->t.index());
	}
}
