package com.mini.framework.util.report.statistics.ranking;

import com.mini.framework.core.exception.ServerException;
import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.date.DateRange;
import com.mini.framework.util.report.statistics.exception.DuplicateUniqueKeyPersistentException;
import com.mini.framework.util.report.statistics.protocol.RankingDimension;
import com.mini.framework.util.report.statistics.protocol.StatisticsSerializeParams;
import com.mini.framework.util.report.statistics.protocol.TimeRegionRange;
import com.mini.framework.util.report.statistics.protocol.TimeRegionUnit;
import com.mini.framework.util.report.statistics.protocol.params.QueryParamsOverview;
import com.mini.framework.util.report.statistics.protocol.process.ProcessContext;
import com.mini.framework.util.report.statistics.ranking.bean.NativeDateScoreRankingScheme;
import com.mini.framework.util.report.statistics.ranking.bean.RankingElementSummationAmount;
import com.mini.framework.util.report.statistics.ranking.bean.StatisticsRankingTable;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;


import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 某个一个纬度的排行榜的特征。<BR>
 * 任何纬度都可能做排行榜。<br>
 * 这些排行榜有相同的地方，也有不同的地方，
 * 不同的地方就是他的特征，需要不同的维度是实现。
 * #标准名词说明#dimension:业务维度例如 订单量/推广员人数/总人数等;
 * #标准名词说明#dimension:排行榜;
 */
public interface RankingDimensionFeature<P extends StatisticsSerializeParams,E extends RankingElementSummationAmount<E>> extends RankingStatistics<P,E>  {


    //TODO NativeDateScoreRankingScheme 的 StatisticsRankingTable 分开存储有一个麻烦。 比较难保证两者一致。
    //目前只需要主要解决  NativeDateScoreRankingScheme 有但 StatisticsRankingTable 没有的情况。

    Logger logger = LogManager.getLogger(RankingDimensionFeature.class);

    @Override
    default String showStatisticsFeatureKey() {
        return showStatisticsDimension().showDimensionStringKey();
    }

    /**
     * 保存一个排行榜
     * @param rankingKey 排序的key
     * @param ranking 排行榜
     */
    void saveRankingTableToCache(String rankingKey, StatisticsRankingTable<E> ranking);


    /**
     * 迭加排行榜。<br>
     * 向原来有的排行榜中叠加新的排行绑数据。
     * 相当于把增量数据加到原来的数据中
     * 排行榜的相同的元素出叠加分数，不同的元素就添加集合
     * @param rankingKey 排序的key
     * @param incrementRanking 要增加的数据，排行榜增量 
     */
    StatisticsRankingTable<E> accumulateRankingTableToCache(String rankingKey, StatisticsRankingTable<E> incrementRanking, boolean coverOrigin);

    /**
     * 查询当前维度下面某个自然时间的排序方案
     * @param paramsUniqueKeyMd5 参数的唯一值 的md5
     * @param timeRegionUnit 时间单元/原子
     * @param offsetDate 某个时间段的起始时间
     * @return 自然时间排行方案如果没有结果就返回 empty <br>
     * @apiNote
     * 注意，如果存和取的时候不要漏掉了纬度，本类是针对当前纬度，所以参数表中不会有维度字段。<br>
     * 为什么针对当前纬度就不传维度字段? 因为，在当前维度下，维度是不变，如果对于不变的东西使用一个可变参数，设计不合理。<br>
     * 如果在某些引用的地方又需要纬度字段，可以参考:
     * @see #showStatisticsDimension()
     */
    Optional<NativeDateScoreRankingScheme> queryNatureTimeRankingScheme(String paramsUniqueKeyMd5, TimeRegionUnit timeRegionUnit, Date offsetDate);


    /**
     * 量查询当前维度下面某个自然时间的排序方案
     * @param paramsUniqueKeyMd5 参数的唯一值 的md5
     * @param regionUnionKey 区域的联合key
     * @return 如果没有这个数据就返回empty 如果连接错误等应该抛出异常。
     */
    Optional<NativeDateScoreRankingScheme> queryNatureTimeRankingScheme(String paramsUniqueKeyMd5, String regionUnionKey);


    /**
     * 批量查询当前维度下面某个自然时间的排序方案
     * @param paramsUniqueKeyMd5 参数的唯一值 的 md5
     * @param regionUnionKeys 区域的联合key 的集合
     * @return 返回regionUnionKey与scheme的映射。
     */
    List<NativeDateScoreRankingScheme> queryBatchNatureTimeRankingScheme(String paramsUniqueKeyMd5, Set<String> regionUnionKeys);


    /**
     * 初始化并储存一个自然时间的排序方案<br>
     * 1把方案存下来。<br>
     * 2把排行榜存下来。<br>
     * TODO 思考一下，这个方法的实现如果有异常要怎么处理。例如数据已初始化，还去初始化出错，导致的问题。
     * @see NativeDateScoreRankingScheme#createByBasicField
     * @param params 参数
     * @param timeRegionUnit 时间单元/原子
     * @param offsetDate 某个时间段的起始时间
     * @return 自然时间排行方案 永远不会返回空
     */
    NativeDateScoreRankingScheme initThenSaveNatureRankingScheme(P params, TimeRegionUnit timeRegionUnit, Date offsetDate);


    /**
     * 统计某个时间段的完整排行榜，应该把所有的信息都记入进来。
     * @param dateRange 时间date范围
     * @return 统计某个时间区间的排行榜 框架要求不允许返回空指针
     */
    StatisticsRankingTable<E> statisticsDateRangeRankingFromImpl(DateRange dateRange, P params, int limit);


    /**
     * 查询缓存中的排行中前多少位
     * @param headOptional 前多少名如果为empty表示不生效
     * @return 某个排行榜 如果没有就返回empty
     */
    Optional<StatisticsRankingTable<E>> queryCacheRankingTable(String rankingKey , Optional<Integer> headOptional);

    /**
     * 批量查询缓存中的排行中前多少位
     * @param keyHeadOptionalMapper
     * @return
     */
    Map<String,StatisticsRankingTable<E>> queryBatchCacheRankingTable(Map<String, Optional<Integer>> keyHeadOptionalMapper);


    /**
     * 最大换存的
     * @return
     */
    int maxCacheRankingElementSize(TimeRegionUnit regionUnit);

    /**
     * 批量查询缓存中的排行中前多少位
     * @param keys
     * @param headOptional
     * @return
     */
    default Map<String,StatisticsRankingTable<E>> queryBatchCacheRankingTable(Set<String> keys, Optional<Integer> headOptional){
        Map<String, Optional<Integer>> keyHeadOptionalMapper = keys.stream().collect(Collectors.toMap(Function.identity(),key->headOptional));
        return queryBatchCacheRankingTable(keyHeadOptionalMapper);
    }

    /**
     * 修改方案的触达时间。
     * @param paramsUniqueKeyMd5 参数的唯一值
     * @param timeRegionUnit 时间单元/原子
     * @param offsetDate 某个时间段的起始时间
     * @param touchLimit 触达的时间上限
     * @param touchUpperLimitOptional 是不是触达的范围上限
     */
    void updateSchemeTouchPoint(String paramsUniqueKeyMd5, TimeRegionUnit timeRegionUnit, Date offsetDate, Date touchLimit,Optional<Boolean> touchUpperLimitOptional);


    /**
     * 创建一个空的数据出来 要注意能过经得起数据验证
     * @param countKey
     * @return 返回一个空的排行榜中一条记录。注意不要出现空指针 一直没有值的一般用0代替
     */
    E createEmptyElement(String countKey);


    /**
     * 显示维度的key
     * @return
     */
    RankingDimension showStatisticsDimension();


    /**
     * 能接受的最大的统计源数据的时间范围<BR>
     * 在作统计的时间会按时间单元拆分统计，
     * 为了不让统计的范围太大导致一些问题，
     * 这里可以设置时间拆分的最大粒度。
     * @return
     */
    TimeRegionUnit acceptStatisticsOriginMaxUnit();

    /**
     * 能接受的最小的缓存聚合数据的时间范围<BR>
     * 在作统计的时间会按时间单元拆分统计，这里设置，最小的聚合单元。
     * @return
     */
    TimeRegionUnit acceptCacheStatisticsMinDateUnit();


    /**
     * 最大的缓存统计增量的毫秒数。<br>
     * 每次上查询和上次查询有一定的时间范围的差异<br>
     * 如果时间范围的差异不大的话，就可以不增更新原来的缓存。
     * @return
     */
    long maxCacheStatisticsIncrementMilliseconds();

    /**
     * 部分工具可能要用到实体，所以特意工具型的对象，这个不会存入数据库。
     * @return 返回一个空的排行榜中一条记录
     */
    default E utilFullyEmptyElement(){
        return createEmptyElement("");
    }








    /**
     * 允许忽略的有缺陷的时间单元<br>
     * 例如 这里返回 hour 那么 12月13号5点15分16秒。<br>
     * 会被当作 12月13号5点 处理。而忽略 最后的15分16秒。这个忽略可以提高比较大的性能。
     * @return
     */
    Optional<TimeRegionUnit> acceptSkipStatisticsDefectDateUnit();


    //-------------------------------------以上是需要被实现的方法-----------------------------------------


    /**
     * 计算出可能使用的历史限制时间
     * @param historyLimitDate 上层调用输入的历史限制时间
     * @return 真正用于统计的历史限制时间。
     */
    default Date countUsefulHistoryLimitDate(Date historyLimitDate){
        return acceptSkipStatisticsDefectDateUnit()
                .map(unit->unit.nearLazyFenceRangeDate(historyLimitDate))
                .orElse(historyLimitDate);
    }


    /**
     * 是不是要忽略时间范围的增量
     * @param milliseconds
     * @return
     */
    default boolean ifSkipCacheStatisticsIncrement(long milliseconds){
        return milliseconds<maxCacheStatisticsIncrementMilliseconds();
    }

    /**
     * 在使用做一次前置验证。有问题早发现
     */
    default void previewValidate(){
        Function<String, ServerException> supplier
                = message-> new ServerException(Status.Server.programConfigJava,"程序使用有问题需要检查utilFullyEmptyElement()出错,%s",message);
        utilFullyEmptyElement().validateOriginComplete(supplier);
        TimeRegionUnit maxUnit = acceptStatisticsOriginMaxUnit();
        List<TimeRegionUnit> supportMaxUnits = Stream.of(TimeRegionUnit.year, TimeRegionUnit.season,TimeRegionUnit.month, TimeRegionUnit.week, TimeRegionUnit.day).collect(Collectors.toList());
        if(!supportMaxUnits.contains(maxUnit)){
            throw supplier.apply(String.format("最大支持的统计点:[%s]应该在此范围内:%s",maxUnit,supportMaxUnits));
        }
    }



    /**
     * 修改方案的触达时间。
     * @param params 查询参数，即标准参数以外的其它参数，也叫子参数
     * @param regionRange 多个自然时间组成的范围
     * @param touchLimit 触达的时间上限
     * @param touchUpperLimitOptional 是不是触达的范围上限
     */
    default void updateSchemeTouchPoint(P params, TimeRegionRange regionRange, Date touchLimit, Optional<Boolean> touchUpperLimitOptional){
        updateSchemeTouchPoint(params.uniqueSerializeMd5(),regionRange.getUnit(),regionRange.getOffset(),touchLimit,touchUpperLimitOptional);
    }


    /**
     * 查询或者创建一个方案。 <br>
     * 这里查了如果没有的话，就创建一个。
     * @param params 查询参数，即标准参数以外的其它参数，也叫子参数
     * @param regionRange 多个自然时间组成的范围
     * @return 自然时间区间的排行榜方案 不能为空
     */
    default NativeDateScoreRankingScheme queryOrInitNatureTimeRankingScheme(P params, TimeRegionRange regionRange){
        return queryNatureTimeRankingScheme(params,regionRange)
                .orElseGet(()-> initThenSaveNatureRankingScheme(params,regionRange));
    }

    /**
     * 查询某个自然时间的排序方案
     * @param params 查询参数，即标准参数以外的其它参数，也叫子参数
     * @param regionRange 多个自然时间组成的范围
     * @return 自然时间区间的排行榜方案 如果没有查到返回 empty
     */
    default Optional<NativeDateScoreRankingScheme> queryNatureTimeRankingScheme(P params, TimeRegionRange regionRange){
        return queryNatureTimeRankingScheme(params.uniqueSerializeMd5(),regionRange.getUnit(),regionRange.getOffset());
    }

    default List<NativeDateScoreRankingScheme> queryBatchNatureTimeRankingScheme(P params, List<TimeRegionRange> regionRanges){
        return queryBatchNatureTimeRankingScheme(params.uniqueSerializeMd5(),regionRanges.stream().map(TimeRegionRange::unitOffsetUnionString).collect(Collectors.toSet()));
    }


    /**
     * 创建并储存一个自然时间的排序方案
     * @param params 查询参数，即标准参数以外的其它参数，也叫子参数
     * @param regionRange 多个自然时间组成的范围
     * @return 自然时间区间的排行榜方案 不能为空
     */
    default NativeDateScoreRankingScheme initThenSaveNatureRankingScheme(P params, TimeRegionRange regionRange){

        //TODO 需要参考统计求和的处理 executePersistentAddRecord
        NativeDateScoreRankingScheme scheme = null;
        try{
            scheme = initThenSaveNatureRankingScheme(params, regionRange.getUnit(), regionRange.getOffset());
            AssertUtil.assertNormal(scheme!=null,()->new ServerException("initThenSaveNatureRankingScheme 方法必须有返回值"));
            assert scheme != null;
        }catch (DuplicateUniqueKeyPersistentException e){
            // 这个异常可以忽略。
            logger.debug("记录一个可以忽略的异常",e);
            logger.warn(String.format("出现一个唯一约束异常 field:[%s],key:[%s],%s  当前异常可以忽略", e.getUniqueKeyField(),e.getUniqueKey(),e.getMessage()));
            scheme = queryNatureTimeRankingScheme(params, regionRange)
                    .orElseThrow(()->new ServerException("发现联合唯一约束key:[%s]冲突又查不到数据params:[%s],region:%s",e.getUniqueKey(),params.uniqueSerialize(),regionRange));
        }
        saveRankingTableToCache(scheme.getRankingKey(), StatisticsRankingTable.createEmpty());
        return scheme;
    }
    

    //-----------------------------------------------


    /**
     * 统计某个时间范围的排行榜。
     * @param context 查询过程的上下文
     * @param params 查询参数，即标准参数以外的其它参数，也叫子参数
     * @param dateRange 时间的开始与结束时间，走左闭又开原则。
     * @param headOptional
     * @param queryLimitDate
     * @return
     */
    default  StatisticsRankingTable<E> statisticsDateRangeRanking(
            ProcessContext context, P params
            , DateRange dateRange
            ,Optional<Integer> headOptional
            ,Date queryLimitDate){
        previewValidate();

        Date realStatisticsLimitDate = countUsefulHistoryLimitDate(queryLimitDate);

        //体现上 realStatisticsLimitDate 大于这个时间直接忽略。
        DateRange fullyStatisticsDateRange = dateRange.limitUpperDate(realStatisticsLimitDate);

        // 无法切分的时间碎片时间段。
        List<DateRange> regionChipDateRanges = new ArrayList<>();
        
        
        int retainSize = maxCacheRankingElementSize(acceptCacheStatisticsMinDateUnit());

        //1 得到要计算的自然时间片段。
        List<TimeRegionRange> requireQueryRegions = TimeRegionRange.splitRangeLessPointHandleChip(fullyStatisticsDateRange
                , acceptCacheStatisticsMinDateUnit(), acceptStatisticsOriginMaxUnit()
                ,regionChipDateRanges::add);

        //得到完全cache过的数据的区域key
        List<NativeDateScoreRankingScheme> fullyCacheRegionUnionKeys = queryBatchNatureTimeRankingScheme(params, requireQueryRegions).stream()
                //TODO 检查最大保留值是不是一样的。
                .filter(NativeDateScoreRankingScheme::ifTouchFullyScope)
                .collect(Collectors.toList());

        // 走批量算法得到各个已完整缓存的时间片段对应的结果。
        StatisticsRankingTable<E> fullyCacheRanking = queryBatchCacheRankingTable(
                fullyCacheRegionUnionKeys.stream().map(NativeDateScoreRankingScheme::getRankingKey).collect(Collectors.toSet())
                , Optional.empty())
                .values().stream()
                .filter(Objects::nonNull)
                .reduce(StatisticsRankingTable.reduceAccumulator(this::createEmptyElement))
                .orElse(StatisticsRankingTable.createEmpty());


        List<Supplier<StatisticsRankingTable<E>>> futureRequireCountRegionRankings = requireQueryRegions.stream()
                //TODO 条件语句比较复杂要分开一下
                .filter(region -> !fullyCacheRegionUnionKeys.stream().map(NativeDateScoreRankingScheme::getRegionUnionKey).collect(Collectors.toSet()).contains(region.unitOffsetUnionString()))
                //打乱顺序，在多个线程请求时候可以复用，其它线程的结果。
                .sorted(TimeRegionRange.randomSorter())
                .map(region -> context.getProcess().pushQuery(() -> statisticsNativeDateOneScopeRanking(context, params, region, Optional.empty(), realStatisticsLimitDate), QueryParamsOverview.create(params,region)))
                .collect(Collectors.toList());


        List<Supplier<StatisticsRankingTable<E>>> futureRegionChipDateRangeRankings = regionChipDateRanges.stream()
                .map(regionChip -> context.getProcess().pushQuery(() -> statisticsDateRangeRankingFromImpl(regionChip, params, retainSize), QueryParamsOverview.create(params,regionChip)))
                .collect(Collectors.toList());

        
        
        //从需要重新统计的时间区域中找出没有得到对应结果的数据走临时查询。
        StatisticsRankingTable<E> requireCountRegionRanking = futureRequireCountRegionRankings.stream()
                .map(Supplier::get)
                .reduce(StatisticsRankingTable.reduceAccumulator(this::createEmptyElement))
                .orElse(StatisticsRankingTable.createEmpty());
        

        //碎片时间中的排行榜。
        StatisticsRankingTable<E> regionChipDateRangeRanking = futureRegionChipDateRangeRankings.stream().map(Supplier::get)
                .reduce(StatisticsRankingTable.reduceAccumulator(this::createEmptyElement))
                .orElse(StatisticsRankingTable.createEmpty());


        //把三类排行榜加上来。
        StatisticsRankingTable<E> fullyScopeRanking = Stream.of(fullyCacheRanking, requireCountRegionRanking, regionChipDateRangeRanking)
                .reduce(StatisticsRankingTable.reduceAccumulator(this::createEmptyElement))
                .orElse(StatisticsRankingTable.createEmpty());
        
        fullyScopeRanking.markResultMeta(realStatisticsLimitDate);
        fullyScopeRanking.fillQueryProcess(context.getProcess());
        return fullyScopeRanking.retainLimitHead(headOptional);
    }


    /**
     * 某个自然时间的范围的排行榜
     * @param params 查询参数，即标准参数以外的其它参数，也叫子参数
     * @param regionRange 多个自然时间组成的范围
     * @param topSize 前多少名
     * @param queryLimitDate 历史时间的截至点，这个不能大于 当前时间。否则可能有问题。
     * @return 统计自然时间区间的排行榜  不能为空
     */
    @Override
    default StatisticsRankingTable<E> statisticsNativeDateRanking(
            ProcessContext context, P params, TimeRegionRange regionRange
            , int topSize
            , Date queryLimitDate) {
            int limitHead = maxCacheRankingElementSize(regionRange.getUnit());
            if(topSize>limitHead){
              throw new ServerException(Status.Server.programConfigJava,"不允许使用topSize:[%s]上限为:[%s]",topSize,limitHead);
            };
        return  statisticsDateRangeRanking(context,params, regionRange.showDateRange(), Optional.of(topSize), queryLimitDate);
    }

    /**
     * 某个自然时间的范围的排行榜
     * @param context 查询过程的上下文
     * @param params 查询参数，即标准参数以外的其它参数，也叫子参数
     * @param regionRange 多个自然时间组成的范围
     * @param headOptional 前多少名如果为empty表示不生效
     * @param queryLimitDate 历史时间的截至点，这个不能大于 当前时间。否则可能有问题。
     * @return 统计自然时间区间的排行榜  不能为空
     */
    default StatisticsRankingTable<E> statisticsNativeDateOneScopeRanking(
            ProcessContext context,P params, TimeRegionRange regionRange
            ,Optional<Integer> headOptional
            ,Date queryLimitDate) {
        previewValidate();

        Date realStatisticsLimitDate = countUsefulHistoryLimitDate(queryLimitDate);
        
        // 查询到排行数据
        NativeDateScoreRankingScheme rankingScheme = queryOrInitNatureTimeRankingScheme(params,regionRange);
        Optional<DateRange> requireTouchRangeOptional = rankingScheme.countRequireTouchRange(realStatisticsLimitDate);

        // 如果 requireTouchRangeOptional 有值，那么要执行，查询操作。
        StatisticsRankingTable<E> ranking = requireTouchRangeOptional.map(requireTouchRange -> {
            // 查询所有的统计数据。
            //TODO 这里应该注意，如果时间范围过大，还是有可能数据量太多，所以应该考虑一下切分一下查询范围。
            StatisticsRankingTable<E> requireTouchRanking = statisticsDateRangeRankingFromImpl(requireTouchRange, params, maxCacheRankingElementSize(regionRange.getUnit()));
            requireTouchRanking.stream().forEach(element -> element.validateComplete(message -> new ServerException(Status.Server.programConfigJava, "数据有错误%s,数据是:%s", message, element)));
            requireTouchRanking.sorterDesc();
            // 叠加次数太多可能不太好。  所以这里判断要不是要叠加
            boolean skipUpdateScheme = ifSkipCacheStatisticsIncrement(requireTouchRange.showTimeScopeMilliseconds());
            StatisticsRankingTable<E> fullyRanking = accumulateRankingTableToCache(rankingScheme.getRankingKey(), requireTouchRanking, !skipUpdateScheme);
            if (!skipUpdateScheme) {
                // scheme中的 touchUpperLimit 并写入数据库。
                updateSchemeTouchPoint(params, regionRange, requireTouchRange.getMaxDate(), Optional.of(rankingScheme.ifTouchRangeUpperLimit(realStatisticsLimitDate)));
            }
            return fullyRanking;
        }).orElseGet(() -> queryCacheRankingTable(rankingScheme.getRankingKey(), headOptional).orElse(StatisticsRankingTable.createEmpty()));
        ranking.markResultMeta(realStatisticsLimitDate);
        ranking.fillQueryProcess(context.getProcess());
        return ranking;
    }


    /**
     * 自定义某个时间范围的排行榜
     * @param params 查询参数，即标准参数以外的其它参数，也叫子参数
     * @param regionUnit 时间单元/原子
     * @param scopeRange 时间的范围
     * @param topSize 前多少名
     * @param queryLimitDate 历史时间的截至点，这个不能大于 当前时间。否则可能有问题。
     * @return 统计自然时间区间的排行榜  不能为空
     */
    @Override
    default StatisticsRankingTable<E> statisticsNativeDateArrayRanking(
            ProcessContext context,P params, TimeRegionUnit regionUnit, DateRange scopeRange
            , int topSize
            , Date queryLimitDate) {
        previewValidate();
        Date realStatisticsLimitDate = countUsefulHistoryLimitDate(queryLimitDate);
        Date rangeMinDate = scopeRange.getMinDate();
        //坚持左闭又左原则，不再加1，如果前端为了操作方便可以controller提供加1的方法。
        Date rangeMaxDate = scopeRange.getMaxDate();
        //Date rangeMaxDate = regionUnit.addTimeUnit(scopeRange.getMaxDate(),1);
        StatisticsRankingTable<E> result = statisticsDateRangeRanking(context,params, DateRange.create(rangeMinDate, rangeMaxDate), Optional.of(topSize), queryLimitDate);
        result.markResultMeta(realStatisticsLimitDate);
        result.fillQueryProcess(context.getProcess());
        return result;
    }


    @Override
    default StatisticsRankingTable<E> statisticsCustomDateRangeRanking(ProcessContext context,P params, DateRange dateRange, int topSize, Date queryLimitDate){
        previewValidate();
        StatisticsRankingTable<E> result = statisticsDateRangeRanking(context,params, dateRange, Optional.of(topSize), queryLimitDate);
        return result;
    }
}
