package com.mini.framework.util.sql.view.database;

import com.mini.framework.core.exception.ServerException;
import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.params.MapParams;
import com.mini.framework.util.string.RegexUtil;

import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Stream;

/**
 * @author jayheo
 */
public enum PaasDatabaseName {

    /**
     * 交易追踪的总商家
     */
    tdtPaas("交易追踪的总商家", DatabaseSchemeType.tdt,false),
    /**
     * 商城的总商家
     */
    mallPaas("商城的总商家", DatabaseSchemeType.mall,false),
    /**
     * 学院的总商家
     */
    schoolPaas("学院的总商家", DatabaseSchemeType.school,false),

    /**
     * 媒体秀的总商家
     */
    mshowPaas("媒体秀的总商家", DatabaseSchemeType.mshow,false),
    /**
     * 交易追踪的一个商家
     */
    tdtSite("交易追踪的一个商家", DatabaseSchemeType.tdt,true),
    /**
     * 商城的一个商家
     */
    mallSite("商城的一个商家", DatabaseSchemeType.mall,true),
    /**
     * 学院的一个商家
     */
    schoolSite("学院的一个商家", DatabaseSchemeType.school,true),

    /**
     * 媒体秀的一个商家
     */
    mshowSite("媒体秀的一个商家", DatabaseSchemeType.mshow,true),
    ;


    private String title;

    private DatabaseSchemeType type;


    private Boolean requireSiteKey;


    PaasDatabaseName(String title, DatabaseSchemeType type, Boolean requireSiteKey) {
        this.title = title;
        this.type = type;
        this.requireSiteKey = requireSiteKey;
    }

    public String getTitle() {
        return title;
    }

    public DatabaseSchemeType getType() {
        return type;
    }

    public Boolean getRequireSiteKey() {
        return requireSiteKey;
    }

    public boolean ifRequireSiteKey(){
        return getRequireSiteKey();
    }

    public boolean ifNotRequireSiteKey(){
        return !ifRequireSiteKey();
    }


    /**
     * 查询数据库的名字
     * @param siteKeyOptional
     * @param paasNameFinder
     * @param siteNameFinder
     * @param <S>
     * @return
     */
    public <S> Optional<String> queryDatabaseName(Optional<S> siteKeyOptional, Function<DatabaseSchemeType,Optional<String>> paasNameFinder, BiFunction<DatabaseSchemeType,S,Optional<String>> siteNameFinder){
        AssertUtil.assertMethodRequire(siteKeyOptional,"siteKeyOptional");
        if(requireSiteKey){
            S siteKey = siteKeyOptional.orElseThrow(() -> new ServerException(Status.Server.programConfigJava, "枚举:[%s]必须指明商家key", this));

            return siteNameFinder.apply(type, siteKey);
        }else{
            return paasNameFinder.apply(type);
        }
    }


    /**
     * 替换占位符号
     * @param origin
     * @param siteKeyOptional
     * @param paasNameFinder
     * @param siteNameFinder
     * @param defaultValueFunction
     * @param <S>
     * @return
     */
    public static <S> String replacePlaceHolder(String origin, Optional<S> siteKeyOptional, Function<DatabaseSchemeType,Optional<String>> paasNameFinder, BiFunction<DatabaseSchemeType,S,Optional<String>> siteNameFinder, Function<String, String> defaultValueFunction){
        AssertUtil.assertMethodRequire(siteKeyOptional,"siteKeyOptional");
        MapParams params = MapParams.build();
        Stream.of(values()).filter(nameType->nameType.ifNotRequireSiteKey() || siteKeyOptional.isPresent())
                .forEach(nameType->{
                    Optional<String> databaseNameOptional = nameType.queryDatabaseName(siteKeyOptional, paasNameFinder, siteNameFinder);
                    databaseNameOptional.ifPresent(databaseName->params.param(nameType.name(),databaseName));
                });

        return RegexUtil.applyPlaceholder(origin,params, key->{throw new ServerException(Status.Server.fatalBug,"占位key:[%s]找不到商家:[%s]库名",key,siteKeyOptional.map(o->(Object) o).orElse("none"));});
    }

}
