package com.mini.framework.util.string;

import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.function.UnaryOperator;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.google.gson.JsonParser;
import com.google.gson.internal.LinkedTreeMap;
import com.google.gson.reflect.TypeToken;
import com.mini.framework.core.exception.HandleStringJsonException;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.string.gson.DateMilliSecondDeserializer;
import com.mini.framework.util.string.gson.DateMilliSecondSerializer;
import com.mini.framework.util.string.gson.DateSecondDeserializer;
import com.mini.framework.util.string.gson.DateSecondSerializer;
import com.mini.framework.util.string.gson.deserializer.GsonUnknownTypeable;
import com.mini.framework.util.string.gson.deserializer.UnknownTypeInBeanDeserializer;
import com.mini.framework.util.string.gson.deserializer.UnknownTypeInGsonMapDeserializer;
import org.apache.commons.lang3.StringUtils;

/**
 * gson操作
 * @author jayheo
 * https://www.jianshu.com/p/e740196225a4
 *
 */
public class GsonUtil {
		
	public static UnaryOperator<GsonBuilder> registerMilliSecondDate(){
		return builder-> builder.setDateFormat(DateFormat.LONG)
				.registerTypeAdapter(Date.class, new DateMilliSecondDeserializer())
				.registerTypeAdapter(Date.class, new DateMilliSecondSerializer());
	}
	
	public static UnaryOperator<GsonBuilder> registerSecondDate(){
		return builder-> builder.setDateFormat(DateFormat.LONG)
				.registerTypeAdapter(Date.class, new DateSecondDeserializer())
				.registerTypeAdapter(Date.class, new DateSecondSerializer());
	}
	
	/**
	 * 处理没有定义数据类型的map
	 * @return
	 */
	public static UnaryOperator<GsonBuilder> handleUnknownTypeInGsonMap(){
		return builder->builder.registerTypeHierarchyAdapter(LinkedTreeMap.class, new UnknownTypeInGsonMapDeserializer());
	}
	
	public static <T extends GsonUnknownTypeable<T>> UnaryOperator<GsonBuilder> handleUnknownTypeInBean(Class<T> clazz){
		return builder->builder.registerTypeHierarchyAdapter(clazz, new UnknownTypeInBeanDeserializer<>(clazz));
	}
	
	public static GsonBuilder createBaseBuilder(){
		return createBaseBuilder(UnaryOperator.identity());
	}
	public static GsonBuilder createBaseBuilder(UnaryOperator<GsonBuilder> operator){
		return operator.apply(new GsonBuilder().disableHtmlEscaping());
	}

	public static Gson buildMilliSecondDateGson(UnaryOperator<GsonBuilder> operator) {
		GsonBuilder builder = createBaseBuilder();
		builder = operator.andThen(registerMilliSecondDate()).apply(builder);
		return builder.create();
	}
	
	
	public static Gson buildMilliSecondDateGson() {
		return buildMilliSecondDateGson(UnaryOperator.identity());
	}
	
	public static Gson buildSecondDateGson(){
		GsonBuilder builder = new GsonBuilder().setDateFormat(DateFormat.LONG)
				.registerTypeAdapter(Date.class, new DateSecondDeserializer())
				.registerTypeAdapter(Date.class, new DateSecondSerializer());
		return builder.disableHtmlEscaping().create();
	}


	/**
	 * 将json转换成对象
	 * Gson().fromJson 没有处理好空指针问题。
	 * @param json json字符串
	 * @param clazz 转换的对象
	 * @return 转换后的对象
	 * @param <E> 转换的对象
	 */
	public static <E>  E fromJson(final String json,Class<E> clazz){
		return fromJson(json, clazz, new Gson());
	}

	/**
	 * 将json转换成对象
	 * Gson().fromJson 没有处理好空指针问题。
	 * @param json json字符串
	 * @param clazz 转换的对象
	 * @param gson gson转换器
	 * @return 转换后的对象
	 * @param <E> 转换的对象
	 */
	public static <E>  E fromJson(final String json,Class<E> clazz,Gson gson){
		AssertUtil.assertMethodRequire(json,"json");
		AssertUtil.assertNotFatal(json.length()>0,"json 的长度为0是无意义的");
		try{
			return gson.fromJson(json, clazz);
		}catch (IllegalStateException | JsonParseException e) {
			String jsonError = StringUtils.abbreviate(json, 200);
			throw new HandleStringJsonException(e,"json转换失败:[%s]",jsonError);
		}
	}

	
	/**
	 * 解析一个list的json
	 * @param elementClass
	 * @param listJson
	 * @param gson
	 * @return
	 */
	public  static <E> List<E> fromListJson( Class<E> clazz, String listJson,Gson gson){
		 //Json的解析类对象
	    JsonParser parser = new JsonParser();
	    //将JSON的String 转成一个JsonArray对象
	    JsonArray jsonArray = parser.parse(listJson).getAsJsonArray();
	    ArrayList<E> elements = new ArrayList<>();
	    //加强for循环遍历JsonArray
	    for (JsonElement user : jsonArray) {
	        //使用GSON，直接转成Bean对象
			E element = gson.fromJson(user, clazz);
	        elements.add(element);
	    }
		return elements;
	}
	
	
	/**
	 * 换成字符串型的map
	 * @param stringMapJson
	 * @param gson
	 * @return
	 */
	public static Map<String, String> fromStringMap(String stringMapJson,Gson gson){
		Map<String,String> objectMap = gson.fromJson(stringMapJson, new TypeToken<Map<String, String>>(){}.getType());
		return objectMap;
		/*Map<String, String> stringMap = new HashMap<String, String>();
		for (Entry<?, ?> entry : objectMap.entrySet()) {
			stringMap.put(entry.getKey()==null?null: entry.getKey().toString(), entry.getValue()==null?null: entry.getValue().toString());
		}
		return stringMap;*/
	}
	

	/**
	 * {1:9,4:10}
	 * {"1":9,"4":10}
	 * 换成整型的map
	 * @param stringMapJson
	 * @param gson
	 * @return
	 */
	public static Map<Integer, Integer> fromIntegerMap(String stringMapJson,Gson gson){
		Map<Integer,Integer> objectMap = gson.fromJson(stringMapJson, new TypeToken<Map<Integer, Integer>>(){}.getType());
		return objectMap;
	}
}
