package com.mini.framework.util.string;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;

import com.mini.framework.core.exception.BadReqException;
import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.string.bean.SplitItemPoint;

/**
 * 正则小工具
 * @author jayheo
 *
 */
public class RegexUtil {
	/**
	 * 通过正则取到匹配的字符串
	 * @param regex
	 * @param input
	 * @return
	 */
	public static String getMatch(String regex,String input){
		AssertUtil.assertMethodRequire(regex, "regex");
		if(input==null)return null;
		Pattern pattern = Pattern.compile(regex);
		Matcher match = pattern.matcher(input);
		if(match.find()){
			return match.group();
		}else{
			return null;
		}
	}
	
	/**
	 * 得到input可以匹配regex的所有的点<br>
	 * 例如:getMatchPoints('\d+','fdkkj45lk4j4ir4oifj43oiti5goi3no4')->[45, 4, 4, 4, 43, 5, 3, 4]
	 * @param regex 匹配规则
	 * @param input 被匹配的对象
	 * @return
	 */
	public static List<String> getMatchPoints(String regex ,String input){
		AssertUtil.assertMethodRequire(regex, "regex");
		String match = getMatch(String.format("(%s)", regex), input,1);
		if(match!=null){
			input = StringUtils.substringAfter(input, match);
			List<String> subMatchs = getMatchPoints(regex, input);
			subMatchs.add(0, match);
			return subMatchs;
		}else{
			return new ArrayList<>();
		}
	}
	

	public static String regexCharsAsNormal(String match){
		String regexChars= "${}()?*"; 
		return regexCharsAsNormal(match,regexChars);
	}
	
	/**
	 * 把正则转义字符换成正常的字符
	 * 比如    {  ->  \\{
	 * @param match
	 * @param regexChars
	 * @return
	 */
	public static String regexCharsAsNormal(String match,String regexChars){
		for (int i = 0; i < regexChars.length(); i++) {
			String regexChar = String.valueOf(regexChars.charAt(i));
			match = match.replace(regexChar, "\\" + regexChar);
		}
		return match;
	}
	
	
	/**
	 * 命名已错误，需要换一个新的方法。
	 * 把字符串用一个切点切开，然后每个数据经过某种处理，再组装起来。
	 * @param pointRegex
	 * @param input
	 * @param handler
	 * @see #splitByPointEachHandle(String, String, Function)
	 * @return
	 */
	@Deprecated
	public static String splitByPointEachHande(String pointRegex,String input, Function<String, String> handler){
		return splitByPointEachHandle(pointRegex, input, handler);
	}


	/**
	 * 把字符串用一个切点切开，然后每个数据经过某种处理，再组装起来
	 * @param pointRegex
	 * @param input
	 * @param handler
	 * @return
	 */
	public static String splitByPointEachHandle(String pointRegex,String input, Function<String, String> handler){
		AssertUtil.assertMethodRequire(pointRegex,"pointRegex");
		AssertUtil.assertMethodRequire(input,"input");
		AssertUtil.assertMethodRequire(handler,"handler");
		SplitItemPoint items = getSplitItemPoint(pointRegex, input);
		items.foreach(handler);
		return items.assem();
	}
	
	
	/**
	 * 根据 切点正则 得到 n+1段和n切点
	 * @param pointRegex 切点正则
	 * @param input
	 * @return
	 */
	public static SplitItemPoint getSplitItemPoint(String pointRegex,String input){
		
		List<String> points = getMatchPoints(pointRegex, input);
		List<String> items = new ArrayList<String>();
		for (String point : points) {
			int offset = input.indexOf(point);
			String item = input.substring(0,offset);
			items.add(item);
			input = StringUtils.substringAfter(input, point);
		}
		items.add(input);
		return new SplitItemPoint(items, points);
	}
	
	/**
	 * 得到匹配的占位符<br>
	 * 例如:getMatchItems('(\d+).*(\d+).*(\d+)','fdkkj45lk4j4ir4oifj43oiti5goi3no4')->Optional[[45, 3, 4]]
	 * @param regex
	 * @param input
	 * @return
	 */
	public static Optional<List<String>> getMatchItems(String regex,String input){
		Pattern pattern = Pattern.compile(regex);
		Matcher match = pattern.matcher(input);
		List<String> items = null;
		if(match.find()){
			int count = match.groupCount();
			items = new ArrayList<String>();
			for (int i = 0;  i < count; i++) {
				items.add(match.group(i + 1));
			}
			return Optional.of(items);
		}
		return Optional.empty();
	}
	
	
	/**
	 * 通过正则取到匹配的字符串
	 * @param regex
	 * @param input
	 * @param patternIndex
	 * @return
	 */
	public static String getMatch(String regex,String input,int patternIndex){
		String[] results = getMatch(regex, input, new int[]{patternIndex});
		if(results==null){
			return null;
		}else{//这里出现数组越界说明，方法有问题。
			return results[0];
		}
		
	}
	
	/**
	 * 这个方法能保证结果的一致性，要么没有结果，要么就是一致的，
	 * @param regex
	 * @param input
	 * @param patternIndexs
	 * @return
	 */
	public static String[] getMatch(String regex,String input,int ... patternIndexs){
		AssertUtil.assertMethodRequire(regex, "regex");
		AssertUtil.assertMethodRequire(input, "input");
		Pattern pattern = Pattern.compile(regex);
		Matcher match = pattern.matcher(input);
		if(match.find()){
			String[] results = new String[patternIndexs.length];
			for (int i = 0; i < results.length; i++) {
				AssertUtil.simpleAssertByStatus(match.groupCount()>=patternIndexs[i], Status.Busy.fatalBug
						, "正则[%s]中没有索引为[%s]的匹配组,需要马上修改代码", regex,patternIndexs[i]);
				AssertUtil.simpleAssertByStatus(0<=patternIndexs[i], Status.Busy.fatalBug,"查询的组索引不能小于0");
				results[i] = match.group(patternIndexs[i]);
			}
			return results;
		}else{
			return null;
		}
	}
	
	
	/**
	 *
	 * <pre>
	 https://study-xzh.aquke.com/a/dff?a=d---------------study-xzh.aquke.com
	 https://study-xzh.aquke.com/dfd/d.html---------------study-xzh.aquke.com
	 https://study-xzh.aquke.com:8080/dfd/d.html---------------study-xzh.aquke.com
	 https://study-xzh.aquke.com/dfd/d.js---------------study-xzh.aquke.com
	 https://study-xzh.aquke.com/---------------study-xzh.aquke.com
	 http://study-xzh.aquke.com/---------------study-xzh.aquke.com
	 //study-xzh.aquke.com/---------------null
	 //study-xzh.aquke.com---------------null
	 study-xzh.aquke.com---------------null
	 https://study-xzh.aquke.com---------------study-xzh.aquke.com
	 https://study-xzh.aquke.com??https://study2-xzh.aquke.com???https://study3-xzh.aquke.com---------------study-xzh.aquke.com??https
	 https://www.baidu.com/s?wd=%E5%BE%AE%---------------www.baidu.com
	 <a href="http://www.baidu.com/s?wd=%E5%BE%AE%---------------www.baidu.com">...</a>
	 * </pre>
	 * 从url中取到domain
	 * @param url
	 * @return
	 */
	public static String getUrlDomain(String url){
		String domain = getUrlDomainWithPort(url);
		//把冒号后面的去掉
		return StringUtils.substringBefore(domain, ":");
	}


	/**
	 * 从url中取到domain 包含端口。
	 * @param url
	 * @return
	 */
	public static String getUrlDomainWithPort(String url){
		return RegexUtil.getMatch("https?://+([^/]*)", url, 1);
	}


	
	/**
	 * 从url中取到host
	 * @param url
	 * @return
	 */
	public static String getUrlHost(String url){
		String host = RegexUtil.getMatch("(http[s]{0,1}://+[^/]*)", url, 1);
		return host;
	}
	
	
	/**
	 * 取到url的命名空间，从域名后算起
	 * @param url
	 * @return
	 */
	public static String getUrlNamespace(String url){
		String namespace = RegexUtil.getMatch("http[s]{0,1}://[^/]*(.*)", url, 1);
		return namespace;
	}
	
	
	
	/**
	 * 从url中取到domain
	 * @param url
	 * @return
	 */
	public static String getUrlParams(String url,String name){
		String value = RegexUtil.getMatch(String.format("[\\?&]%s=([^&]+)",name), url, 1);
		return value;
	}
	
	
	/**
	 * 从xml节点中值
	 * @param xml
	 * @param nodeName
	 * @return
	 */
	public static String xmlRegexNodeValue(String nodeName,String xml){
		String regex =String.format("<%s>.*\\[(.*)\\].*\\].*</%s>",nodeName,nodeName);
		String value = RegexUtil.getMatch(regex, xml, 1);
		return value;
	}
	
	
	/**
	 * @param long32or16 32用true 16位用false
	 * @param charLower   小写用true 大写用false
	 * @return
	 */
	public static boolean md5Match(String md5,boolean long32or16,boolean charLower){
		String regex = String.format("^[0-9%s]{%s}$", charLower?"a-f":"A-F",long32or16?32:16);
		return  getMatch(regex, md5)!=null;
	}
	
	

	public static String applyPlaceholder(String input ,Map<String, Object> params){
		return applyPlaceholder(input, params," ");
	}
	

	public static Optional<int[]> getFromToIndex(String searchChar,String input){
		int fromIndex = StringUtils.indexOf(input, searchChar);
		if(fromIndex>=0){
			return Optional.of(new int[]{fromIndex,fromIndex + searchChar.length() });
		}else{
			return Optional.empty();
		}
	}

	/**
	 * <pre>
	 * 如果提供参数  {abc:345,eft:144}
	 * 替换 ${abc}   ${abc!1234} -> 345
	 * 替换 ${abcd!1234} -> 1234
	 * 替换 ${abcd  } -> 报错
	 * 如果匹配不到!前的变量会使用!后的值 ，如果还匹配不到会使用  defaultValue
	 *  </pre>
	 * @param input
	 * @param params
	 * @param defaultValue
	 * @return
	 */
	public static String applyPlaceholder(String input ,Map<String, Object> params,String defaultValue){
		return applyPlaceholder(input, params, (name)->defaultValue);
	}
	
	/**
	 * <pre>
	 * 如果提供参数  {abc:345,eft:144}
	 * 替换 ${abc}   ${abc!1234} -> 345
	 * 替换 ${abcd!1234} -> 1234
	 * 替换 ${abcd  } -> 报错
	 * 如果匹配不到!前的变量会使用!后的值 ，如果还匹配不到会使用  defaultValue
	 *  </pre>
	 * @param input
	 * @param params
	 * @param defaultValue
	 * @return
	 */
	public static String applyPlaceholder(String input ,Map<String, Object> params,Function<String, String> defaultValueFunction){
		AssertUtil.assertMethodRequire(defaultValueFunction, "defaultValueFunction");
		AssertUtil.assertMethodRequire(params, "params");
		AssertUtil.assertMethodRequire(input, "input");
		input = " " + input;//不知道什么原因如果是${abc}以他为起点就报错，所以前面加一个空格最后再去掉。
		String regex = "\\$\\{([a-zA-Z0-9]+)(\\!?[^\\!^\\}]*)\\}";//(![a-zA-Z0-9]+
		Pattern pattern = Pattern.compile(regex);
		boolean tryMatch = false;
		do{
			tryMatch = false;
			Matcher match = pattern.matcher(input);
			if(match.find()){
				if(match.find(1)){// TODO 如果是${abc}以他为起点就找不到这个但不知道原因。
					String name = match.group(1);
					String def = null;
					if(match.find(2)){
						String defMatch = match.group(2);
						if(defMatch.length()>0){
							def = defMatch.substring(1);
						}
					}
					
					Object val = Optional.ofNullable(params.get(name)).orElse(def);
					val = Optional.ofNullable(val).orElseGet(()->defaultValueFunction.apply(name));
					AssertUtil.assertNoBadReq(val!=null, Status.BadReq.illParam, "defaultValueFunction不允许返回空");
					//Object val = VarUtil.firstNotNull(params.get(name),def,defaultValueFunction.apply(name));
					if(val instanceof Supplier<?>){//允许使用supplier
						val = ((Supplier<?>) val).get();
					}
					input = match.replaceFirst(String.valueOf(val));
					tryMatch = true;
				}
			}
		}while(tryMatch);
		
		
		String exRegex = "\\$\\{.*\\}";//${.*}
		String errorPlaceHolder = getMatch(exRegex, input);
		if(errorPlaceHolder!=null){
			throw new BadReqException(Status.BadReq.illPlaceHolder, "无法解析的占位符:[%s]", errorPlaceHolder).detail("placeHolder", errorPlaceHolder);
		}
		return input.substring(1);//不知道什么原因如果是${abc}以他为起点就报错，所以前面加一个空格最后再去掉。
	}
}
