package com.mini.framework.util.surplus.reduce;

import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.log.Event5WBuilder;
import com.mini.framework.util.log.MiniLogLevel;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.util.Optional;

/**
 * 持久层递减用的执行器，
 * 如果是mysql一般建议使用sql
 * @param <K> 用户描述数据的数据库id，一般是Integer或者Long等
 * <pre>
 *      update from table
 *           set surplus = surplus - increment
 *           where id = key
 *               and surplus >= increment
 * </pre>
 * @author jayheo
 */
@FunctionalInterface
public interface PersistenceReduceExecutor<K> {


    Logger logger = LogManager.getLogger(PersistenceReduceExecutor.class);
    /**
     * 执行减少某个值。
     * 如果是mysql一般建议使用sql
     * <pre>
     *      update from table
     *           set surplus = surplus - increment
     *           where id = key
     *               and surplus >= increment
     * </pre>
     * */
    boolean executeReduce(K key, int increment);


    /**
     * 执行减少某个值。
     * 允许部分缺乏，比如说的要扣  100个，但是没有100个可以扣。那么最小可以接受扣10个。
     * 这里会使用二分法尝试性接近 最小值
     * */
    default Optional<Integer> executeReduceAllowLack(K key, int targetIncrement, int minIncrement){
        AssertUtil.assertNotFatalProgramConfig(targetIncrement>0, Status.Server.programConfigJava,"increment:[%s]必须大于 0",targetIncrement);
        AssertUtil.assertNotFatalProgramConfig(minIncrement>0, Status.Server.programConfigJava,"minIncrement:[%s]必须大于 0",minIncrement);
        AssertUtil.assertNotFatalProgramConfig(targetIncrement>=minIncrement, Status.Server.programConfigJava,"increment:[%s]不能小于 minIncrement:[%s]",targetIncrement,minIncrement);

        Event5WBuilder event = Event5WBuilder.event(1, "executeReduceAllowLack")
                .who("持久层key:[%s]", key).what("执行减操作")
                .how("目标减少数 targetIncrement:[%s],最少可接受减少数 minIncrement:[%s]", targetIncrement, minIncrement);;
        logger.debug(event);

        boolean executeSuccess = executeReduce(key,targetIncrement);
        logger.debug(event.when("执行结果:[%s]",executeSuccess));
        if(executeSuccess){
            return Optional.of(targetIncrement);
        }else if(targetIncrement > minIncrement) {
            int nextTargetIncrement = (targetIncrement + minIncrement )/2;
            AssertUtil.assertNotFatalProgramConfig(nextTargetIncrement!=targetIncrement, Status.Server.programConfigJava
                    ,"二分后增量值:[%s]与当前增量值:[%s]还是一样,最小增量为:[%s]",nextTargetIncrement,targetIncrement,minIncrement);
            event.why("执行失败后发还没有到最少接受量即:targetIncrement:[%s] > minIncrement:[%s]",targetIncrement , minIncrement)
                    .result("修改目标增量再执行即 nextTargetIncrement:[%s] = (targetIncrement:[%s] + minIncrement:[%s] )/2",nextTargetIncrement,targetIncrement ,minIncrement );
            logger.log(MiniLogLevel.getKeyBizLog(),event);
            return executeReduceAllowLack(key,nextTargetIncrement,minIncrement);
        }else{
            return Optional.empty();
        }
    }
}
