package com.mini.framework.util.surplus.reduce;

import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.function.PredicateSpread;
import com.mini.framework.util.log.Event5WBuilder;
import com.mini.framework.util.log.MiniLogLevel;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * 剩余量预扣代理工具
 * 在真正要扣库存前弄代理一下，用来减少并发压力。
 * @param <K> 用户描述数据的数据库id，一般是Integer或者Long等
 * @author jayheo
 */
public class SurplusReduceCacheAgent<K> {

    
    private final static Logger logger = LogManager.getLogger(SurplusReduceCacheAgent.class);

    private final SurplusReduceCacheQueue surplusReduceCacheQueue;

    private final Integer reduceCacheSize;


    /**
     * 剩余量状态延迟多少毫秒
     * */
    private final Long surplusStatusDelay;


    public Optional<ReduceQueueItem> reduceSurplus(K key, PersistenceReduceExecutor<K> persistenceIncrementExecutor){
        return reduceSurplus(key,1,persistenceIncrementExecutor)
                .orElse(new ArrayList<>()).stream().findFirst();
    }

    /**
     * 扣库存
     * 这里返回的应该都是可用的
     * */
    public Optional<List<ReduceQueueItem>> reduceSurplus(K key,int size, PersistenceReduceExecutor<K> persistenceIncrementExecutor) {
        AssertUtil.assertNotFatalBug(size > 0, "扣除库存数:[%s]必须是正数", size);
        Event5WBuilder event = Event5WBuilder.event(2, "cacheAgentReduceSurplus").who("key:[%s]", key).what("尝试执行扣除尘:[%s]个剩余量", size);
        logger.debug(event);

        List<ReduceQueueItem> foundItems = new ArrayList<>();

        boolean cacheLackItem = false;
        int popStepIndex = 1;

        while (!cacheLackItem && foundItems.size() < size) {
            Optional<ReduceQueueItem> cacheItemOptional = surplusReduceCacheQueue.popItem(key);
            logger.debug(event.where("第?次/总?次:[%s/%s]", popStepIndex, size).when("从cacheQueue中拿到了结果是:%s", cacheItemOptional));
            if (cacheItemOptional.isPresent()) {
                ReduceQueueItem cacheItem = cacheItemOptional.get();
                if (cacheItem.ifNotLack()) {
                    foundItems.add(cacheItem);
                } else if (cacheItem.ifLackAndEffective()) {
                    foundItems.add(cacheItem);
                } else {
                    logger.debug(event.result("当前找到记录是过期了的直接忽略，继续循环，foundItems:%s", foundItems));
                }
            } else {
                cacheLackItem = true;
            }
            popStepIndex++;
        }

        AssertUtil.assertNotFatalProgramConfig(cacheLackItem != (foundItems.size() == size), Status.Server.programConfigJava
                , "上面while中应该保证requireFromPersistence:[%s] != (foundItems.size():[%s]==size:[%s])", cacheLackItem, foundItems.size(), size);

        List<ReduceQueueItem> realUsefulItems = foundItems.stream().filter(ReduceQueueItem::ifNotLack).collect(Collectors.toList());
        logger.debug(event.why("应该从缓存找到:[%s]个实际找到:[%s]个", size, foundItems.size()));
        if (cacheLackItem) {
            //TODO 这里要加一个分布锁
            logger.log(MiniLogLevel.getFrameWorkLog(), event.result("把有用的退回到缓存，重新去数据库中取,希望取:[%s]个能接受取:[%s]个", reduceCacheSize, 1));
            surplusReduceCacheQueue.batchPushItem(key, realUsefulItems);
            Optional<Integer> itemSizeOptional = persistenceIncrementExecutor.executeReduceAllowLack(key, reduceCacheSize, 1);
            int gotItemSize = itemSizeOptional.orElse(0);
            Date now = new Date();
            List<ReduceQueueItem> allItemArray = ReduceQueueItem.createEffectiveAndLackItemArray(now, gotItemSize
                    , reduceCacheSize - gotItemSize, new Date(), surplusStatusDelay);
            logger.debug(event.result("实际从持久层取到:[%s]个拼装部分缺乏的后推入缓存%s", gotItemSize, allItemArray));
            surplusReduceCacheQueue.batchPushItem(key, allItemArray);
            return reduceSurplus(key, size, persistenceIncrementExecutor);
        } else if (foundItems.stream().allMatch(ReduceQueueItem::ifNotLack)) {
            //这里返回的应该都是可用的。
            return Optional.of(foundItems);
        } else {
            surplusReduceCacheQueue.batchPushItem(key, realUsefulItems);
            return Optional.empty();
        }

    }



    /**
     *
     * @param surplusReduceCacheQueue
     * @param reduceCacheSize
     * @param surplusStatusDelay 剩余量状态保持时间，比如设置成1000，那么发现没有库存时，1000ms内，程序都理解成没有库存，哪怕实际上有库存了。
     * */
    public static <K> SurplusReduceCacheAgent<K> createInstance(SurplusReduceCacheQueue<K> surplusReduceCacheQueue, Integer reduceCacheSize, Long surplusStatusDelay) {
        return new SurplusReduceCacheAgent(surplusReduceCacheQueue,reduceCacheSize,surplusStatusDelay);
    }


    private SurplusReduceCacheAgent(SurplusReduceCacheQueue<K> surplusReduceCacheQueue, Integer reduceCacheSize, Long surplusStatusDelay) {
            AssertUtil.assertMethodRequire(surplusReduceCacheQueue,"surplusReduceCacheQueue");
            AssertUtil.assertMethodRequire(surplusStatusDelay,"surplusStatusDelay");
            AssertUtil.assertMethodRequire(reduceCacheSize,"reduceCacheSize");
            this.surplusReduceCacheQueue = surplusReduceCacheQueue;
            this.reduceCacheSize = reduceCacheSize;
            this.surplusStatusDelay = surplusStatusDelay;
        }
    }

