package com.mini.framework.util.annotation;

import com.mini.framework.core.code.standard.style01.flag.define.CodeStandardDefineFindList;
import com.mini.framework.core.code.standard.style01.flag.define.CodeStandardDefineFindOne;
import com.mini.framework.core.exception.HandleReflectException;
import com.mini.framework.util.reflect.ReflectUtil;
import com.mini.framework.util.relation.ShipUtil;

import java.lang.annotation.Annotation;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 注解工具类。
 * @author jayheo
 */
public class AnnotationUtil {


    @CodeStandardDefineFindOne
    public static <A extends Annotation> Optional<A> findFieldAnnotation(Field field, Class<A> annotationClass){
        return findElementAnnotation((AnnotatedElement) field,annotationClass);
    }

    @CodeStandardDefineFindOne
    public static <A extends Annotation> Optional<A> findMethodAnnotation(Method method, Class<A> annotationClass){
        return findElementAnnotation((AnnotatedElement) method,annotationClass);
    }

    @CodeStandardDefineFindOne
    public static <A extends Annotation> Optional<A> findClassAnnotation(Class<?> clazz, Class<A> annotationClass){
        return findElementAnnotation((AnnotatedElement) clazz,annotationClass);
    }


    /**
     * 从某个元素中找到某些注解。
     * @param annotatedElement
     * @param annotationClassList
     * @return
     */
    @CodeStandardDefineFindList
    public static List<Annotation> findElementAnnotations(AnnotatedElement annotatedElement, List<Class<? extends Annotation>> annotationClassList){
        return annotationClassList.stream()
                .map(annotationClass->findElementAnnotation(annotatedElement,annotationClass))
                .filter(Optional::isPresent)
                .map(Optional::get)
                .distinct()
                .collect(Collectors.toList());
    }


    /**
     * 从某个元素中找到注解
     * @param annotatedElement
     * @param annotationClass
     * @param <A>
     * @return
     */
    @CodeStandardDefineFindOne
    public static <A extends Annotation> Optional<A> findElementAnnotation(AnnotatedElement annotatedElement, Class<A> annotationClass){

        if(!ReflectUtil.matchOneClassLoader(annotatedElement,annotationClass)){
            String annotationClassName = annotationClass.getName();
            try {
                annotationClass = (Class<A>) annotationClass.getClassLoader().loadClass(annotationClassName);
            } catch (ClassNotFoundException e) {
                throw new HandleReflectException(e,"annotationClass对应的类加载器无法加载类:[%s]",annotationClassName);
            }
        }
        return Optional.ofNullable(annotatedElement.getAnnotation(annotationClass));
    }


    /**
     * 递归有顺序的查询对应的注解实例，如果有重复，将是后面覆盖前面的
     * nnotatedElement
     * @param annotationClass
     * @param <A>
     * @return
     */
    public static <A extends Annotation> void fillElementDeepSortedAnnotations(ArrayList<A> foundAnnotations, Set<String> loopedClassNames, AnnotatedElement annotatedElement, Class<A> annotationClass){
        Predicate<Annotation> annotationMatchTargetClass = annotation->annotation.annotationType().getName().equals(annotationClass.getName());
        Stream.of(annotatedElement.getAnnotations())
            //.map(Annotation::annotationType)
            .forEach(nextAnnotation->{
                if(annotationMatchTargetClass.test(nextAnnotation)) {
                    foundAnnotations.add((A) nextAnnotation);
                }
                if(!loopedClassNames.contains( nextAnnotation.annotationType().getName())){
                    loopedClassNames.add(nextAnnotation.annotationType().getName());
                    fillElementDeepSortedAnnotations(foundAnnotations,loopedClassNames,nextAnnotation.annotationType(), annotationClass);
                }

            });
    }


    @CodeStandardDefineFindList
    public static <A extends Annotation> ArrayList<A> findElementDeepSortedAnnotations(AnnotatedElement annotatedElement, Class<A> annotationClass){
        ArrayList<A> result = new ArrayList<>();
        LinkedHashSet<String> loopedClassNames = new LinkedHashSet<>();
        fillElementDeepSortedAnnotations(result,loopedClassNames,annotatedElement,annotationClass);
        return result;
    }


    @CodeStandardDefineFindList
    public static <A extends Annotation> ArrayList<A> findElementDeepSortedAnnotationsDistinct(AnnotatedElement annotatedElement, Class<A> annotationClass){
        ArrayList<A> fullList = findElementDeepSortedAnnotations(annotatedElement, annotationClass);
        ShipUtil.distinct(fullList, Objects::hashCode);
        return fullList;
    }



}
