package com.mini.framework.util.aspect;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.mini.framework.util.asserts.AssertUtil;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.reflect.MethodSignature;

import com.mini.framework.core.exception.HandleException;
import com.mini.framework.core.status.Status;


/**
 * 切面编程操作bean的工具
 * @author jayheo
 *
 */
public class AspectBeanUtil {
	/**
	 * 取到类名方法名地参数信息
	 * @param joinPoint
	 * @return
	 */
	public static String typeMethodArgsInfo(JoinPoint joinPoint) {
		Method targetMethod = getMethod(joinPoint);
		String methodName = targetMethod.getName();
		String className = targetMethod.getDeclaringClass().getName();
		Object[] args = joinPoint.getArgs();
		String argsMsg = StringUtils.join(args, ",");
		String info = String.format("%s.%s(%s)", className, methodName, argsMsg);
		return info;
	}

	public static <T extends Annotation> T getAnnotation(JoinPoint joinPoint, Class<T> annotationClass) {
		Signature signature = joinPoint.getSignature();
		MethodSignature methodSignature = (MethodSignature) signature;
		Method targetMethod = methodSignature.getMethod();
		return targetMethod.getAnnotation(annotationClass);
	}
	
	
	/**
	 * 从参数中取到某个类型的值，慎重使用<br>
	 * @param joinPoint
	 * @param paramSuper
	 * @return
	 */
	public static <T> T getParam(JoinPoint joinPoint, Class<T> paramSuper) throws HandleException{
		Object[] args = joinPoint.getArgs();
		Method method = getMethod(joinPoint);
		//从参数中取到某个类型的值
		return queryParamByClass(joinPoint,paramSuper)
				.orElseThrow(()->new HandleException(Status.Handle.def, "参数列表:%s(%s)中必须包含%s", method,StringUtils.join(args, ","),paramSuper));
		
	}



	/**
	 * 通过注解查询参数。
	 * */
	public static <A extends Annotation> Optional<?> queryParamByAnnotation(JoinPoint joinPoint, Class<A> flag){
		AssertUtil.assertMethodRequire(joinPoint,"joinPoint");
		Parameter[] params = getMethod(joinPoint).getParameters();
		Object[] args = joinPoint.getArgs();
		//如果这里出现错误说明，程序的写法就有错误，需要立即修改。
		AssertUtil.assertNotFatalProgramConfig(params.length==args.length, Status.Server.programConfigJava,"params:[%s].length==args:[%s].length必须为true",params,args);
		List<Integer>haveFlagIndexes = new ArrayList<>();

		for (int index = 0; index < params.length; index++) {
			if(params[index].getAnnotation(flag)!=null){
				haveFlagIndexes.add(index);
			}
		}
		return haveFlagIndexes.stream().map(index->args[index]).filter(Objects::nonNull).findFirst();
	}

	/**
	 * 从参数中取到某个类型的值，慎重使用<br>
	 * @param joinPoint
	 * @param paramSuper
	 * @return
	 */
	public static <T> Optional<T> queryParamByClass(JoinPoint joinPoint, Class<T> paramSuper){
		AssertUtil.assertMethodRequire(paramSuper,"paramSuper");
		AssertUtil.assertMethodRequire(joinPoint,"joinPoint");
		Object[] args = joinPoint.getArgs();
		for (Object arg : args) {
			//判断是不是父类
			if(paramSuper.isAssignableFrom(arg.getClass())){
				return Optional.of((T)arg);
			}
		}
		return Optional.empty();
	}



	public static Method getMethod(JoinPoint joinPoint){
		Signature signature = joinPoint.getSignature();
		MethodSignature methodSignature = (MethodSignature)signature;
		Method method = methodSignature.getMethod();
		return method;
	}
	
	
	public static <T> List<T> listParam(JoinPoint joinPoint, Class<T> paramSuper) {
		Object[] args = joinPoint.getArgs();
		List<T> list = new ArrayList<T>();
		for (Object arg : args) {
			//判断是不是父类
			if(paramSuper.isAssignableFrom(arg.getClass())){
				list.add((T) arg);
			}
		}
		return list;
	}

}
