package com.mini.framework.util.date;

import com.mini.framework.util.asserts.AssertUtil;
import org.apache.commons.lang3.time.DateFormatUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 日期区间 ，一般走左闭右开的原则。即包含最小值不包含最大值。
 * @author jayheo
 *
 */
public class DateRange {
	/**
	 * 最小的 时间
	 */
	private Date minDate;

	/**
	 * 最大的 时间
	 */
	private Date maxDate;


	public static DateRange create(Date minDate, Date maxDate) {
		DateRange instance = new DateRange();
		instance.maxDate = maxDate;
		instance.minDate = minDate;
		return instance;
	}

	/**
	 * 限制上限时间。
	 * @param upperLimitDate
	 * @return
	 */
	public DateRange limitUpperDate(Date upperLimitDate){
		Date maxDate = this.maxDate.before(upperLimitDate)?this.maxDate:upperLimitDate;
		maxDate = maxDate.before(minDate)?this.minDate:maxDate;
		return create(  minDate,maxDate);
	}

	/**
	 * 显示时间包含的毫秒数
	 * @return
	 */
	public long showTimeScopeMilliseconds(){
		return Math.max(0,maxDate.getTime() - minDate.getTime());
	}


	public long showEffectTimes(){
		return maxDate.getTime() - minDate.getTime();
	}


	/**
	 * 两个时间范围的交集
	 * @param other
	 * @return 交集的时间范围
	 */
	public Optional<DateRange> intersect(DateRange other){
		BiFunction<Date,Date,Date> max = (a,b)->a.after(b)?a:b;
		BiFunction<Date,Date,Date> min = (a,b)->a.before(b)?a:b;
		Date newMin = max.apply(minDate, other.minDate);
		Date newMax = min.apply(maxDate, other.maxDate);
		return Optional.of(DateRange.create(newMin, newMax))
				.filter(range->range.showEffectTimes()>=0);
	}


	/**
	 * 与另外一个时间范围的差 即  A - B
	 * @param other 另外一个时间范围
	 * @return 差集的时间范围
	 */
	public List<DateRange> diffFromOther(DateRange other){
		List<DateRange> list = new ArrayList<>();

		BiFunction<Date,Date,Date> findMax = (a,b)->a.after(b)?a:b;
		BiFunction<Date,Date,Date> findMin = (a,b)->a.before(b)?a:b;
		if(other.haveTimeScope()){
			if(minDate.before(other.minDate)){
				list.add(DateRange.create(minDate,findMin.apply(other.minDate,maxDate)));
			}
			if(maxDate.after(other.maxDate)){
				list.add(DateRange.create(findMax.apply(other.maxDate,minDate),maxDate));
			}
		}else{
			list.add(this);
		}
		return list;
	}






	// <editor-fold defaultstate="collapsed" desc="均匀拆分时间">




	/**
	 * 拆分时间段使用使用每一分钟。
	 * @param rightClosed  是否右闭合
	 * @return 处理后的结果
	 */
	public List<Date> chuckMapUseFixedMinute(boolean rightClosed){
		return chuckMapUseFixedMinute(rightClosed,Function.identity());
	}


	/**
	 * 拆分时间段使用使用每一分钟。
	 * @param resultMapper 每一分钟的处理器
	 * @param rightClosed 是否右闭合
	 * @return 处理后的结果
	 * @param <R> 返回的类型
	 */
	public <R> List<R> chuckMapUseFixedMinute(int minutes,boolean rightClosed,Function<Date,R> resultMapper){
		return chuckMapUseFixedTime((long) minutes *60*1000,rightClosed,resultMapper);
	}
	/**
	 * 拆分时间段使用使用每一分钟。
	 * @param resultMapper 每一分钟的处理器
	 * @param rightClosed 是否右闭合
	 * @return 处理后的结果
	 * @param <R> 返回的类型
	 */
	public <R> List<R> chuckMapUseFixedMinute(boolean rightClosed,Function<Date,R> resultMapper){
		return chuckMapUseFixedMinute(1,rightClosed,resultMapper);
	}


	/**
	 * 拆分时间段使用使用每一分钟。
	 * @param rightClosed  是否右闭合
	 * @return 处理后的结果
	 */
	public List<Date> chuckMapUseFixedMinute10(boolean rightClosed){
		return chuckMapUseFixedMinute10(rightClosed,Function.identity());

	}

	/**
	 * 拆分时间段使用使用每一分钟。
	 * @param resultMapper 每一分钟的处理器
	 * @param rightClosed 是否右闭合
	 * @return 处理后的结果
	 * @param <R> 返回的类型
	 */
	public <R> List<R> chuckMapUseFixedMinute10(boolean rightClosed,Function<Date,R> resultMapper){
		return chuckMapUseFixedMinute(10,rightClosed,resultMapper);
	}


	/**
	 * 拆分时间段使用使用每一小时。
	 * @param rightClosed 是否右闭合
	 * @param resultMapper 每一小时的处理器
	 * @return 处理后的结果
	 * @param <R> 返回的类型
	 */
	public <R> List<R> chuckMapUseFixedHour(boolean rightClosed,Function<Date,R> resultMapper){
		return chuckMapUseFixedMinute(60,rightClosed,resultMapper);
	}

	/**
	 * 拆分时间段使用使用每一小时。
	 * @param rightClosed 是否右闭合
	 * @return 处理后的结果
	 */
	public List<Date> chuckMapUseFixedHour(boolean rightClosed){
		return chuckMapUseFixedHour(rightClosed,Function.identity());
	}



	/**
	 * 拆分时间段使用使用一个固定的时间
	 * @param fixedTime 固定的时间
	 * @param rightClosed 是否右闭合
	 * @param resultMapper 每一分钟的处理器
	 * @return 处理后的结果
	 * @param <R> 返回的类型
	 */
	public <R> List<R> chuckMapUseFixedTime(long fixedTime,boolean rightClosed ,Function<Date,R> resultMapper){
		return chuckMapUseFixedTime(fixedTime,rightClosed).stream().map(resultMapper).collect(Collectors.toList());
	}


	/**
	 * 拆分时间段使用使用一个固定的时间
	 * @param fixedTime 固定的时间
	 * @param rightClosed 是否右闭合
	 * @return 处理后的结果
	 */
	public List<Date> chuckMapUseFixedTime(long fixedTime,boolean rightClosed ){
		AssertUtil.assertNotFatal(fixedTime>0,"fixedTime必须大于0");
		List<Date> list = new ArrayList<>();
		Date date = getDateStartUseFixedTime(minDate,fixedTime);
		while (!date.after(maxDate)) {
			if(matchRange(date,rightClosed)){
				list.add(date);
			}
			date = new Date(date.getTime()+fixedTime);
		}
		return list;
	}


	/**
	 * 得到时间的开始
	 * @param originDate 原始时间
	 * @param fixedTime 固定的时间单位是毫秒
	 * @return 开始时间
	 */
	private Date getDateStartUseFixedTime(Date originDate, long fixedTime){
		long count = originDate.getTime()/fixedTime;
		return new Date(count*fixedTime);
	}


	/**
	 * 以自然天为单位拆分时间段。
	 * @return 时间段
	 */
	public List<DateRange> chuckFlatByNatureDayDate(){
		Date minDayDate = DateUtil.getCleanedDateUnit(TimeSectionType.day, minDate);
		Date maxDayDate = DateUtil.getCleanedDateUnit(TimeSectionType.day, maxDate);
		Date minNextDayDate = DateUtil.addTimeUnit(TimeSectionType.day, minDayDate, 1);
		List<DateRange> ranges = new ArrayList<>();
		if(maxDayDate.equals(minDayDate)){
			ranges.add(this);
		} else if(maxDayDate.after(minDayDate)){
			ranges.add(DateRange.create(minDate,minNextDayDate));
			ranges.addAll(DateRange.create(minNextDayDate,maxDate).chuckFlatByNatureDayDate());
		}
		return ranges;
	}

	// </editor-fold>


	@Override
	public String toString() {
		String format =  "yyyy-MM-dd HH:mm:ss";
		return String.format("[%s->%s)",  DateFormatUtils.format(minDate, format), DateFormatUtils.format(maxDate, format));
	}

	/**
	 * 是不是匹配时间范围。
	 * @param target 目标时间
	 * @return 是否匹配
	 */
	public boolean matchRangeClosed(Date target){
		return matchRange(target,true) ;

	}

	/**
	 * 是不是匹配时间范围。
	 * @param target 目标时间
	 * @param rightClosed 是否右闭合
	 * @return 是否匹配
	 */
	public boolean matchRange(Date target,boolean rightClosed){
		AssertUtil.assertMethodRequire(target,"target");
		boolean notRightClosed = !target.before(minDate) && target.before(maxDate);
		return notRightClosed || (rightClosed && target.equals(maxDate));
	}



	/**
	 * 是否还有时间范围
	 * @return
	 */
	public boolean haveTimeScope(){
		return showTimeScopeMilliseconds() >0;
	}

	public DateRange() {
		super();
	}

	public Date getMaxDate() {
		return maxDate;
	}

	public void setMaxDate(Date maxDate) {
		this.maxDate = maxDate;
	}

	public Date getMinDate() {
		return minDate;
	}

	public void setMinDate(Date minDate) {
		this.minDate = minDate;
	}

}
