package com.mini.framework.util.report.statistics.ranking.bean;


import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonParser;
import com.mini.framework.core.exception.ServerException;
import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.report.statistics.protocol.OneStatisticsResult;
import com.mini.framework.util.report.statistics.protocol.StatisticsResultMeta;
import com.mini.framework.util.report.statistics.protocol.process.ForkQueryProcess;
import org.apache.commons.lang3.builder.ToStringBuilder;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;


/**
 * 排行榜中一个点(一条记录)的数据
 */
public abstract class RankingElementSummationAmount<A extends RankingElementSummationAmount<A>> implements OneStatisticsResult {

    private StatisticsResultMeta resultMeta;

    private ForkQueryProcess process;

    /**
     * 聚合计算使用的key 一般是用户id或者什么的。
     */
    private String countKey;

    /**
     * 用于排名的值
     * TODO 排名的值，应该由使用方提供计算方法，例如 如何根据 v1234 得到sorter
     */
    private Double sorterScore;


    @Override
    public void clearRedundancyFields() {
        //什么都不要做。
    }

    /**
     * 根据现的元素计算出总分数
     * 当这里出现异常的时候，建议检查一下，
     * @see #validateComplete(Function)
     * @return 返回综合排序分数
     */
    public abstract double countSorterScore() ;

    /**
     * 与另外一个数合并
     * @param other
     */
    protected abstract void mergeOtherHandleImplElement(A other);


    @Override
    public Optional<StatisticsResultMeta> showResultMeta() {
        return Optional.ofNullable(resultMeta);
    }

    @Override
    public void fillResultMeta(StatisticsResultMeta resultMeta) {
        this.resultMeta = resultMeta;
    }

    /**
     * 合并另外一个元素到表元素
     * @param other
     */
    public void mergeOther(A other){
        mergeOtherHandleImplElement(other);
        sorterScore = countSorterScore();
    }

    /**
     * 验证原始数据的完整性<br>
     * 可以参考下面的方法的写法。
     * @see #validateComplete(Function)
     * @param exceptionSupplier 如果不完成就抛出这个异常。
     * @param <E> 异常。
     * @throws E 调用方提供的异常
     */
    public abstract <E extends Throwable> void validateOriginComplete(Function<String,E> exceptionSupplier) throws E;


    /**
     * 把别的值累加到自己身上
     *
     * @param other
     */
    public void accumulateOther(A other) {
        Function<String, ServerException> supplier = message->new ServerException(Status.Server.programConfigJava,"做数据累加的时候出错:%s",message);
        validateOriginComplete(supplier);
        other.validateOriginComplete(supplier);
        this.mergeOtherHandleImplElement(other);
        resetSorter();
    }


    /**
     * 重新设置一下排序值
     */
    public void resetSorter(){
        sorterScore = countSorterScore();
    }


    /**
     * 完整性的验证
     * @param exceptionSupplier
     * @param <E>
     * @throws E
     */
    public <E extends Throwable> void validateComplete(Function<String,E> exceptionSupplier) throws E{
        validateOriginComplete(exceptionSupplier);
        resetSorter();
        if(sorterScore ==null){
            throw exceptionSupplier.apply("sorter 不能为空");
        }
        if(countKey==null){
            throw exceptionSupplier.apply("countKey 不能为空");
        }
    }




    public RankingElementSummationAmount() {
    }

    public RankingElementSummationAmount(String countKey, Double sorter) {
        this.countKey = countKey;
        this.sorterScore = sorter;
    }




    /**
     * 累加多个计数
     * @param keyAmounts
     * @return
     */
    public static <E extends RankingElementSummationAmount<E>> E accumulate(Map.Entry<String, List<E>> keyAmounts, Function<String ,E> emptyFunction) {
        return accumulate(keyAmounts.getKey(),keyAmounts.getValue(),emptyFunction);
    }


    /**
     * 累加多个计数
     *
     * @param countKey
     * @param amounts
     * @return
     */
    public static <E extends RankingElementSummationAmount<E>> E accumulate(String countKey, List<E> amounts, Function<String ,E> emptyFunction) {
        E master = emptyFunction.apply(countKey);
        amounts.stream().forEach(master::accumulateOther);
        return master;
    }


    /**
     * 返回地排行榜依据的降低序排序器
     * @param <E>
     * @return
     */
    public static <E extends RankingElementSummationAmount<E>> Comparator<E> rankingSorterDesc() {
        return Comparator.comparing(amount->-amount.getSorterScore());
    }

    /**
     * 返回地排行榜依据的升序排序器
     * @param <E>
     * @return
     */
    public static <E extends RankingElementSummationAmount<E>> Comparator<E> rankingSorter() {
        return Comparator.comparing(RankingElementSummationAmount::getSorterScore);
    }

    public static <E extends RankingElementSummationAmount<E>> E copyNewInstance(Function<String,E> instanceSupplier, E other){
        E instance = instanceSupplier.apply(other.getCountKey());
        instance.mergeOther(other);
        return instance;
    }


    public boolean matchKey(A amount) {
        return matchKey(amount.getCountKey());
    }

    public boolean matchKey(String countKey) {
        AssertUtil.assertNotFatalProgramConfig(this.countKey!=null, Status.Server.programConfigJava,"当前对象中countKey为空this:%s",this);
        return this.countKey.equals(countKey);
    }

    /**
     * 要考虑并发，要考虑扩展
     * @return
     */
    public static Gson createGsonBean(RankingElementSummationAmount<?> amount){
        return amount.createGsonBuilder().create();
    }

    protected abstract StatisticsRankingTable<A> deserializeJsonRankingTable(String paramsJson);


    public GsonBuilder createGsonBuilder(){
        return new GsonBuilder();
    }


    /**
     * 序列化 这个方法可以重写
     *
     * @param amounts
     * @return
     */
    public static <E extends RankingElementSummationAmount<E>> String serializeList(List<E> amounts,E bean) {
        return createGsonBean(bean).toJson(amounts);
    }


    /**
     * 反序列化，这个方法可以重写
     *
     * @param serializeString
     * @return
     */
    public static <E extends RankingElementSummationAmount<E>> List<E> deserializeList(String serializeString,E bean) {
        Gson gson = createGsonBean(bean);
        JsonArray array = new JsonParser().parse(serializeString).getAsJsonArray();
        List<E> list = new ArrayList<>();
        Class<E> clazz = (Class<E>) bean.getClass();
        array.forEach(jsonElement -> {
            E element = gson.fromJson(jsonElement, clazz);
            list.add(element);
        });
        return list;
    }


    /**
     * 把两个排行榜合并。
     *
     * @param masterList
     * @param slaverList
     */
    public static <E extends RankingElementSummationAmount<E>> void accumulateOtherList(
            List<E> masterList, List<E> slaverList) {
        slaverList.stream().forEach(slaver -> {
            Optional<E> masterOptional = masterList.stream().filter(master -> master.matchKey(slaver)).findFirst();
            masterOptional.ifPresent(master -> master.accumulateOther(slaver));
            if (!masterOptional.isPresent()) {
                masterList.add(slaver);
            }
        });
        masterList.sort(rankingSorterDesc());
    }


    /**
     * 把两个排行榜合并到成为一个新的排行榜
     * @param ranking1
     * @param ranking2
     * @param <E>
     * @return
     */
    public static <E extends RankingElementSummationAmount<E>> List<E> accumulateRankings(List<E> ranking1,List<E> ranking2){
        return accumulateAllRanking(Stream.of(ranking1,ranking2).collect(Collectors.toList()));
    }


    /**
     * 把多个排行榜合并到成为一个新的排行榜<BR>
     * 注意一下，这个方法，会改变  rankingList里的内容，如果这个事情有影响需要修改一下本方法。
     * @param rankingList
     * @param <E>
     * @return
     */
    public static <E extends RankingElementSummationAmount<E>> List<E> accumulateAllRanking(List<List<E>> rankingList){
        List<E> result = new ArrayList<>();
        rankingList.forEach(elements->accumulateOtherList(result,elements));
        result.sort(rankingSorterDesc());
        return result;
    }

    public StatisticsResultMeta getResultMeta() {
        return resultMeta;
    }

    public void setResultMeta(StatisticsResultMeta resultMeta) {
        this.resultMeta = resultMeta;
    }



    @Override
    public Optional<ForkQueryProcess> showQueryProcess() {
        return Optional.ofNullable(process);
    }

    @Override
    public void fillQueryProcess(ForkQueryProcess process) {
        this.process = process;
    }

    public String getCountKey() {
        return countKey;
    }

    public void setCountKey(String countKey) {
        this.countKey = countKey;
    }


    public Double getSorterScore() {
        return sorterScore;
    }

    public void setSorterScore(Double sorterScore) {
        this.sorterScore = sorterScore;
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }
}