package com.mini.framework.util.report.statistics.summation.bean.compare;

import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.report.statistics.protocol.OneStatisticsResult;
import com.mini.framework.util.report.statistics.protocol.StatisticsResultMeta;
import com.mini.framework.util.report.statistics.protocol.process.ForkQueryProcess;
import com.mini.framework.util.report.statistics.summation.bean.SummationElementAmount;
import org.apache.commons.lang3.builder.ToStringBuilder;

import java.util.Optional;

/**
 * 两者求和的对比
 * @author jayheo
 */
public class BothSummationCompare implements OneStatisticsResult {

    private StatisticsResultMeta resultMeta;
    
    private ForkQueryProcess process;

    /**
     * count 参考的数
     * 大于或者等于0
     */
    private Long contrastCount;

    /**
     * count 主要的数
     * 大于或者等于0
     */
    private Long mainCount;

    /**
     * count 对比比例
     * 可能为空的非负数
     */
    private Optional<Double> countRatio;
    
    
    /**
     * sum 参考的数
     * 大于或者等于0
     */
    private Optional<Long> contrastSum;

    /**
     * sum 主要的数
     * 大于或者等于0
     */
    private Optional<Long> mainSum;

    /**
     * sum 对比比例
     * 可能为空的非负数
     */
    private Optional<Double> sumRatio;


    /**
     * 对比两个求和。
     * @param mainSummation
     * @param contrastSummation
     * @return
     */
    public static BothSummationCompare createSummationInstance(SummationElementAmount mainSummation, SummationElementAmount contrastSummation){
        return createCountSumInstance(mainSummation.getCount(),contrastSummation.getCount(),mainSummation.getSum(),contrastSummation.getSum());
    }


    /**
     * 创建一个 count 的对比
     * @param main
     * @param contrast
     * @return
     */
    public static BothSummationCompare createCountInstance(Long main, Long contrast){
        BothSummationCompare instance = new BothSummationCompare();
        instance.mainCount  = main;
        instance.contrastCount  = contrast;
        instance.countRatio = compareRatio(main, contrast);
        return instance;
    }


    /**
     * 创建一个 count sum 的对比
     * @param mainCount
     * @param contrastCount
     * @param mainSum
     * @param contrastSum
     * @return
     */
    public static BothSummationCompare createCountSumInstance(Long mainCount, Long contrastCount, Long mainSum, Long contrastSum){
        BothSummationCompare instance = createCountInstance(mainCount, contrastCount);
        instance.mainSum  =Optional.ofNullable(mainSum);
        instance.contrastSum  = Optional.ofNullable(contrastSum);

        if(instance.mainSum.isPresent() && instance.contrastSum.isPresent()){
            instance.sumRatio = compareRatio(mainSum, contrastSum);
        }else{
            instance.sumRatio = Optional.empty();
        }

        return instance;
    }


    /**
     * 对比两个Long数据。
     * @param main
     * @param contrast
     * @return
     */
    public static Optional<Double> compareRatio(Long main,Long contrast ){
        AssertUtil.assertMethodRequire(main,"main");
        AssertUtil.assertMethodRequire(contrast,"contrast");
        if(contrast>0){
            return Optional.of(1.0*main/contrast);
        }else{
            return Optional.empty();
        }
    }
    


    public Long getContrastCount() {
        return contrastCount;
    }

    public BothSummationCompare setContrastCount(Long contrastCount) {
        this.contrastCount = contrastCount;
        return this;
    }

    public Long getMainCount() {
        return mainCount;
    }

    public BothSummationCompare setMainCount(Long mainCount) {
        this.mainCount = mainCount;
        return this;
    }

    public Optional<Double> getCountRatio() {
        return countRatio;
    }

    public BothSummationCompare setCountRatio(Optional<Double> countRatio) {
        this.countRatio = countRatio;
        return this;
    }


    public Optional<Double> getSumRatio() {
        return sumRatio;
    }

    public BothSummationCompare setSumRatio(Optional<Double> sumRatio) {
        this.sumRatio = sumRatio;
        return this;
    }

    public ForkQueryProcess getProcess() {
        return process;
    }

    public void setProcess(ForkQueryProcess process) {
        this.process = process;
    }

    public StatisticsResultMeta getResultMeta() {
        return resultMeta;
    }

    public void setResultMeta(StatisticsResultMeta resultMeta) {
        this.resultMeta = resultMeta;
    }

    public Optional<Long> getContrastSum() {
        return contrastSum;
    }

    public Optional<Long> getMainSum() {
        return mainSum;
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

    @Override
    public Optional<StatisticsResultMeta> showResultMeta() {
        return Optional.ofNullable(resultMeta);
    }

    @Override
    public void fillResultMeta(StatisticsResultMeta resultMeta) {
        this.resultMeta = resultMeta;
    }


    @Override
    public void clearRedundancyFields() {
        //没有任务要清理的数据
    }

    @Override
    public Optional<ForkQueryProcess> showQueryProcess() {
        return Optional.ofNullable(process);
    }

    @Override
    public void fillQueryProcess(ForkQueryProcess process) {
        this.process = process;
    }
}
