package com.mini.framework.util.string.gson.format;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.gson.GsonBuilder;
import com.mini.framework.core.exception.ServerException;
import com.mini.framework.util.string.gson.format.bean.Json;
import com.mini.framework.util.string.gson.format.flag.ResponseExportType;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.mini.framework.core.exception.BadReqException;
import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.bean.ArrayUtil;
import com.mini.framework.util.export.ExcelColumnDeclares;
import com.mini.framework.util.export.ExcelExportSupport;
import com.mini.framework.util.function.PredicateSpread;
import com.mini.framework.util.string.GsonUtil;

/**
 * 通过gson方式把数据梳理一次的工具
 * @author jayheo
 *
 */
public class GsonDataFormatBean {

	private static Logger logger = LogManager.getLogger(GsonDataFormatBean.class);
	
	private Object origin;
	
	private JsonElement currentObject;
	
	private List<ExcelColumnDeclares> excelColumnDeclareList;


	private ExportResourceFromByte exportResourceFunction;
	
	//TODO  谢志豪  添加数据按某个规则分组聚合。比如 有500条数据，现在按日期中的时间分组，得到   20组数据。所有的组加起来正好 是500个。这里可以参考collects里的方法。
	//TODO 等上面的方法OK后，再提供多阶分组。
 	

	public static GsonDataFormatBean build(ExportResourceFromByte exportResourceFunction,List<ExcelColumnDeclares> excelColumnDeclaresList,Object object){
		GsonDataFormatBean gsonDataFormat = new GsonDataFormatBean();
		gsonDataFormat.setOrigin(object);
		gsonDataFormat.getCurrent();
		gsonDataFormat.excelColumnDeclareList = excelColumnDeclaresList;
		AssertUtil.assertNotFatalBug(exportResourceFunction!=null,"参数 exportResourceFunction 不能为空");
		gsonDataFormat.exportResourceFunction = exportResourceFunction;
		return gsonDataFormat;
	}


	/**
	 * 当前一个json来处理
	 * @param target
	 * @return
	 */
	private JsonElement useAsJson(Object target){
		if(target instanceof JsonElement){
			return (JsonElement) target;
		}else{
			return gsonForUse().toJsonTree(target);
		}
	}

	//----------------------------获取某个节点---------------------------

	private JsonElement getCurrent(){
		if(currentObject==null){
			//TODO null判断
			// 这里注意，如果出现这样的，会导致内存溢出	private Logger logger = LogManager.getLogger();
			currentObject = gsonForUse().toJsonTree(origin);
		}
		return currentObject;
	}
	
	private Gson gsonForUse(){
		return GsonUtil.buildMilliSecondDateGson(GsonBuilder::serializeNulls);
	}
	

	/**
	 * 根据命名空间取json元素中的json对象
	 * @param jsonElement
	 * @param names
	 * @return 返回自己本身
	 */
	private List<JsonObject> getObject(JsonElement jsonElement,String... names){
		AssertUtil.assertMethodRequire(jsonElement, "jsonElement");
		AssertUtil.assertSupport(names != null, "names长度要大于1");
		List<JsonObject> elements = new ArrayList<>();
		//如果没有这个json
		if(jsonElement.isJsonNull()){
			return elements;
			//如果是一个json对象
		}else if(jsonElement.isJsonObject()){
			if(names.length==0){
				elements.add(jsonElement.getAsJsonObject());
			}else{
				JsonElement subJson = jsonElement.getAsJsonObject().get(names[0]);
				if (subJson!=null){//如果没有这个数据就忽略
					elements.addAll(getObject(subJson,ArrayUtils.subarray(names, 1, names.length)));
				}
			}
		}else if(jsonElement.isJsonArray()){
			for (JsonElement subJsonElement : jsonElement.getAsJsonArray()) {
				elements.addAll(getObject(subJsonElement,names));
			}
		}else{
			logger.debug("发现一个没有处理的jsonElement, names:{},jsonElement:{}",jsonElement,Arrays.toString(names));
		}
		return elements;
	}
	
	
	/**
	 * 根据命名空间取对应的json对象，如果碰到数组直接拆分成多个
	 * @param namespace
	 * @return 返回自己本身
	 */
	private List<JsonObject> getObject(String namespace){
		String[] names = namespace.split("\\.");
		if(names[0].length()==0){
			names = new String[]{};
		}
		JsonElement element = getCurrent();
		return getObject(element, names);
	}


	//----------------------------做某些节点操作--------------------------------

	/**
	 * 添加根节点的属性。
	 * @param name
	 * @param property
	 */
	public void addRootProperty(String name,Object property){
		getCurrent().getAsJsonObject().add(name,useAsJson(property));
	}



	//----------------------------------做某些命令类操作------------------------------------------

	/**
	 * 保留某些属性
	 * @param jsonObject
	 * @param names  要保留的属性名字
	 */
	private void retain(JsonObject jsonObject,String... names){
		List<String> keys = jsonObject.entrySet().stream().map(Entry::getKey).collect(Collectors.toList());
		keys.removeAll(Arrays.asList(names));
		for (String key : keys) {
			jsonObject.remove(key);
		}
	}
	

	/**
	 * 删除某些属性
	 * @param namespace 指向目标对象的命名空间
	 * @param names  要删除的属性名字
	 * @return 返回自己本身
	 */
	public GsonDataFormatBean remove(String namespace,String... names){
		AssertUtil.assertNoBadReq(namespace!=null, Status.BadReq.noReq,"namespace不能为空");
		List<JsonObject> objects = getObject(namespace);
		for (String name : names) {
			for (JsonObject object : objects) {
				object.remove(name);
			}
		}
		return this;
	}
	
	

	/**
	 * 关联并加入数据
	 * @param mainNamespace 主体数据的命名空间
	 * @param mainKey 在主体数据中的key，这个key可以找到去匹配数据的关联值。
	 * @param unionNamespace 需要合并数据的命名空间
	 * @param unionKey 需要合并数据的中的key
	 * @param unionNames 需要保存属性的名字
	 * @return 返回自己本身
	 */
	public GsonDataFormatBean union(String mainNamespace,String mainKey,String unionNamespace,String unionKey,String... unionNames){
		List<JsonObject> mainObjects = getObject(mainNamespace);
		
		for (JsonObject mainObject : mainObjects ){
			union(mainObject, mainKey, unionNamespace, unionKey, unionNames);
		}
		
		return this;
	}
	
	/**
	 * 关联并加入数据
	 * @param mainObject 主体数据
	 * @param mainKey 在主体数据中的key，这个key可以找到去匹配数据的关联值。
	 * @param unionNamespace 需要合并数据的命名空间
	 * @param unionKey 需要合并数据的中的key
	 * @param unionNames 需要保存属性的名字
	 */
	private void union(JsonObject mainObject, String mainKey, String unionNamespace, String unionKey, String... unionNames) {
		List<JsonObject> unionDataList =  getObject(unionNamespace);
		JsonElement mainElement = mainObject.getAsJsonObject().get(mainKey);
		if(mainElement!=null && !mainElement.isJsonNull()){//如果是空就没有必要去找了
			JsonElement foundElement = ArrayUtil.findElement(mainElement, unionDataList, unionElement->unionElement.getAsJsonObject().get(unionKey));
			if(foundElement!=null && foundElement.isJsonObject()){
				JsonObject unionObject = foundElement.getAsJsonObject();
				for (String unionName : unionNames) {
					mainObject.add(unionName, unionObject.get(unionName));
				}
			}
		}
		
	}

	/**
	 * 保留某些属性，其它的都删除
	 * @param namespace 指向目标对象的命名空间
	 * @param names  要保留的属性名字
	 * @return 返回自己本身
	 */
	public GsonDataFormatBean retain(String namespace,String... names){
		AssertUtil.assertNoBadReq(namespace!=null, Status.BadReq.noReq,"namespace不能为空");
		List<JsonObject> objects = getObject(namespace);
		for (JsonObject object : objects) {
			retain(object, names);
		}
		return this; 
	}

	/**
	 * 合并
	 * @param mainNamespace 主体数据的命名空间
	 * @param mainKey 在主体数据中的key，这个key可以找到去匹配数据的关联值。
	 * @param mergeNamespace 需要合并数据的命名空间
	 * @param mergeKey 需要合并数据的中的key
	 * @param alias 别名
	 * @return 返回自己本身
	 */
	public GsonDataFormatBean merge(String mainNamespace,String mainKey,String mergeNamespace,String mergeKey,String alias){
		List<JsonObject> mainObjects = getObject(mainNamespace);
		for (JsonObject mainObject : mainObjects ){
			merge(mainObject, mainKey, mergeNamespace, mergeKey, alias);
		}
		return this;
	}
	
	
	/**
	 * @param main 主体数据
	 * @param mainKey 在主体数据中的key，这个key可以找到去匹配数据的关联值。
	 * @param mergeNamespace 需要合并数据的命名空间
	 * @param mergeKey 需要合并数据的中的key
	 * @param alias 别名
	 */
	private void merge(JsonObject main,String mainKey,String mergeNamespace,String mergeKey,String alias){
		List<JsonObject> mergeDataList =  getObject(mergeNamespace);
		JsonElement mainElement = main.getAsJsonObject().get(mainKey);
		if(mainElement!=null && !mainElement.isJsonNull()){//如果是空就没有必要去找了
			JsonElement foundElement = ArrayUtil.findElement(mainElement, mergeDataList, mergeElement->mergeElement.getAsJsonObject().get(mergeKey));
			if(foundElement!=null){
				main.getAsJsonObject().add(alias, foundElement);
			}
		}
	}
	
	/**
	 * 合并多个 
	 * @param mainNamespace 主体数据的命名空间
	 * @param mainKey 在主体数据中的key，这个key可以找到去匹配数据的关联值。
	 * @param mergeNamespace 需要合并数据的命名空间
	 * @param mergeKey 需要合并数据的中的key
	 * @param listAlias 别名
	 * @return 返回自己本身
	 */
	public GsonDataFormatBean mergeList(String mainNamespace,String mainKey,String mergeNamespace,String mergeKey,String listAlias){
		List<JsonObject> mainObjects = getObject(mainNamespace);
		for (JsonObject mainObject : mainObjects ){
			mergeList(mainObject, mainKey, mergeNamespace, mergeKey, listAlias);
		}
		return this;
	}
	
	
	/**
	 * @param main 主体数据
	 * @param mainKey 在主体数据中的key，这个key可以找到去匹配数据的关联值。
	 * @param mergeNamespace 需要合并数据的命名空间
	 * @param mergeKey 需要合并数据的中的key
	 * @param listAlias 别名
	 */
	private void mergeList(JsonObject main,String mainKey,String mergeNamespace,String mergeKey,String listAlias){
		List<JsonObject> mergeDataList =  getObject(mergeNamespace);
		JsonElement mainElement = main.getAsJsonObject().get(mainKey);
		JsonArray jsonArray = new JsonArray();
		//如果是空就没有必要去找了
		if(mainElement!=null && !mainElement.isJsonNull()){
			List<JsonObject> list = mergeDataList.stream().filter(mergeElement->mainElement.equals(mergeElement.getAsJsonObject().get(mergeKey))).collect(Collectors.toList());
			for (JsonObject jo : list) {
				jsonArray.add(jo);
			}
		}
		main.getAsJsonObject().add(listAlias, jsonArray);
	}
	

	/**
	 * 把某个map中一个值查出来放到mainObject中，以alias为key保存
	 * @param mainNamespace 主数据的命名空间
	 * @param mapperKey 用于查询的key
	 * @param mapDataNamespace 被查询的数据池
	 * @param alias 新值的名字
	 * @return 返回自己本身
	 */
	public GsonDataFormatBean mapper(String mainNamespace,String mapperKey,String mapDataNamespace,String alias){
		List<JsonObject> mainObjects = getObject(mainNamespace);
		for (JsonObject mainObject : mainObjects ){
			mapper(mainObject, mapperKey, mapDataNamespace,  alias);
		}
		return this;
	}
	
	
	/**
	 * 把某个map中一个值查出来放到mainObject中，以alias为key保存
	 * @param mainObject  主数据
	 * @param mapperKey 用于查询的key
	 * @param mapDataNamespace 被查询的数据池
	 * @param alias 新值的名字
	 */
	private void mapper(JsonObject mainObject,String mapperKey,String mapDataNamespace,String alias){
		List<JsonObject> mapDataList =  getObject(mapDataNamespace);
		AssertUtil.assertNoBadReq(mapDataList.size()<=1, Status.BadReq.illParam,"结果中path:[%s]对应的数据个数(%s)最多只能是1", mapDataNamespace,mapDataList.size());
		Optional<JsonElement> keyElementOptional = Optional.ofNullable(mainObject.get(mapperKey)).filter(PredicateSpread.of(JsonElement::isJsonNull).negate());
		if(!mapDataList.isEmpty()&&keyElementOptional.isPresent()){
			mainObject.add(alias, getElementAsStringKey(mapDataList.get(0), keyElementOptional.get()));
		}
	}
	
	/**
	 * 拿到元素，使用String型的key
	 * 查找key与map中的key可能都不是String这里统一使用String
	 * @param object
	 * @param key
	 * @return 返回自己本身
	 */
	private JsonElement getElementAsStringKey(JsonObject object,JsonElement key){
		String keyString = key.getAsString();
		Map<String, JsonElement> objectMap = object.entrySet().stream().collect(Collectors.toMap(e->e.getKey(), e->e.getValue()));
		return objectMap.get(keyString);
	}
	
	
	/**
	 * 设置别名
	 * @param namespace
	 * @param sourceName
	 * @param aliasName
	 * @return 返回自己本身
	 */
	public GsonDataFormatBean alias(String namespace,String sourceName,String aliasName){
		List<JsonObject> objects = getObject(namespace);
		for (JsonObject object : objects) {
			alias(object,sourceName,aliasName);
		}
		return this;
	}


	/**
	 * 修改名字
	 * @param namespace
	 * @param oldName
	 * @param newName
	 * @return 返回自己本身
	 */
	public GsonDataFormatBean rename(String namespace,String oldName,String newName){
		List<JsonObject> objects = getObject(namespace);
		for (JsonObject object : objects) {
			rename(object,oldName,newName);
		}
		return this;
	}


	/**
	 * json操作
	 * @param namespace
	 * @param targetKey
	 * @param resultAlias
	 * @return 返回自己本身
	 */
	public GsonDataFormatBean json( Json.Action action,String namespace, String targetKey, String resultAlias){
		List<JsonObject> objects = getObject(namespace);
		for (JsonObject object : objects) {
			json(action,object,targetKey,resultAlias);
		}
		return this;
	}
	
	/**
	 * 把同一个命名空间下几个字段拼接起来，赋值到alias中
	 * @param namespace 
	 * @param alias 需要被赋值的项
	 * @param nullAs 如果出现空值要怎么处理
	 * @param spilt 间隔的字符串
	 * @param items 需要拼接的项
	 * @return 返回自己本身
	 */
	public GsonDataFormatBean concat(String namespace,String alias,String nullAs,String spilt,String... items){
		List<JsonObject> objects = getObject(namespace);
		for (JsonObject object : objects) {
			concat(object, alias,nullAs, spilt, items);
		}
		return this;
	}
	
	
	/**
	 * 把object中几个字段拼接起来，赋值到alias中
	 * @param object
	 * @param alias 需要被赋值的项
	 * @param nullAs 如果出现空值要怎么处理
	 * @param spilt 间隔的字符串
	 * @param items 需要拼接的项
	 */
	private void concat(JsonObject object, String alias, String nullAs, String spilt, String... items) {
		String concat = Stream.of(items).map(item->{
			JsonElement itemJsonObject = object.get(item);
			String itemObject =itemJsonObject==null || itemJsonObject.isJsonNull()? nullAs:itemJsonObject.getAsString();
			return itemObject;
		}).collect(Collectors.joining(spilt));
		object.addProperty(alias, concat);
	}
	

	

	/**
	 * 设置别名
	 * @param object
	 * @param sourceName
	 * @param aliasName
	 */
	private void alias(JsonObject object,String sourceName,String aliasName){
		JsonElement element = object.get(sourceName);
		if(element!=null){
			object.add(aliasName, element);
		}
	}
	
	/**
	 * 修改别名
	 * @param object
	 * @param oldName
	 * @param newName
	 */
	private void rename(JsonObject object,String oldName,String newName){
		JsonElement element = object.remove(oldName);
		if(element!=null){
			object.add(newName, element);
		}
	}

	/**
	 * 修改别名
	 * @param object
	 * @param targetKey
	 * @param resultAlias
	 */
	private void json(Json.Action action,JsonObject object,  String targetKey, String resultAlias){
		JsonElement element = object.get(targetKey);
		if(element != null && !element.isJsonNull()){
			switch (action){
				case deserialize:
					JsonElement result = gsonForUse().fromJson(element.getAsString(), JsonElement.class);
					object.add(resultAlias,result);
					break;
				case serialize:
					object.addProperty(resultAlias,gsonForUse().toJson(element));
					break;
				default:
					throw new ServerException(Status.Server.programConfigJavaFramework,"发现没有定义的json action:%s",action);
			}
		}
	}

	

	/**
	 * 填充数据
	 * @param namespace
	 * @param name
	 * @param value
	 * @return 返回自己本身
	 */
	public GsonDataFormatBean fill(String namespace,String name,Object value){
		List<JsonObject> objects = getObject(namespace);
		for (JsonObject object : objects) {
			fill(object, name,value);
		}
		return this;
	}
	
	
	/**
	 * 填充数据
	 * @param object
	 * @param name
	 * @param value
	 */
	private void fill(JsonObject object, String name, Object value) {
		object.add(name, GsonUtil.buildMilliSecondDateGson().toJsonTree(value));
	}
	
	/**
	 * 导出
	 * @return 返回自己本身
	 */
	private GsonDataFormatBean exportExcel(String excelName){
		ExcelColumnDeclares matchDeclare = excelColumnDeclareList.stream()
				.filter(declare -> declare.name().equals(excelName))
				.findFirst()
				//如果你发现这里出现了，就检查一下注解 ExcelColumnDeclares.name 与 excelName 是不是一致 对应的命名应该是 export,excel,${excelName}
				.orElseThrow(()->new BadReqException(Status.BadReq.illParam,"导出excel时没有发现excel名字:[%s]",excelName));
		return exportExcel(matchDeclare);
	}


	/**
	 * 导出
	 * @param excelColumnDeclares 列配置
	 * @return 返回自己本身
	 */
	private GsonDataFormatBean exportExcel(ExcelColumnDeclares excelColumnDeclares){
		//TODO 暂时使用的是xls的类库，所以生成的文件必须是xls格式，后续如果有其它需要，就要升级库。
		byte[] bytes = ExcelExportSupport.build(currentObject).exportAsBytes(excelColumnDeclares.listRootPath(), Arrays.asList( excelColumnDeclares.value()));
		AssertUtil.assertNotFatal(exportResourceFunction !=null, "需要使用导出使用必须配置有属性:exportResourceFunction");
		//TODO 暂时使用 xls以后要根据 type来决定用什么后缀
		String typeSuffix = "xls";
		String resourcePath = exportResourceFunction.exportUrlResource(bytes,typeSuffix);
		currentObject = new JsonObject();
		currentObject.getAsJsonObject().addProperty("resourcePath", resourcePath);
		return this;
	}


	/**
	 * 应用来自于表达式的命令
	 * @param explain
	 * @return
	 */
	public GsonDataFormatBean applyCommandsFromExplain(String explain){
		return applyCommandsFromExplain(explain, cp->{});
	}
	
	/**
	 * 应用来自于表达式的命令
	 * @param explain
	 * @param beforeApply 应该前做一点操作
	 * @return 返回自己本身
	 */
	public GsonDataFormatBean applyCommandsFromExplain(String explain,Consumer<List<CommandParams>> beforeApply){
		List<CommandParams> functionCommands = CommandParams.resolverExplain(explain);
		//TODO formatAliases 最好要以对象的形式放到响应中去。而不是字符串。
		CommandParams fillAppliedDataFormat = CommandParams.createFromFunctionParams("fill", "", "appliedDataFormat", explain);
		functionCommands.add(fillAppliedDataFormat);
		beforeApply.accept(functionCommands);
		return applyFunctionCommands(functionCommands);
	}

	/**
	 * 应用函数命令
	 * @param functionCommands 函数命令
	 * @return 返回自己本身
	 */
	public GsonDataFormatBean applyFunctionCommands(List<CommandParams> functionCommands){
		for (CommandParams functionCommand : functionCommands) {
			String function = functionCommand.getName();
			String[] args = functionCommand.getArgs();
			String argsToString = Arrays.toString(args);
			switch (function) {
			case "union":
				AssertUtil.assertNoBadReq(args.length>4, Status.BadReq.illParam, "union的参数%s的个数必须大于4", argsToString);
				this.union(args[0], args[1],args[2],args[3],ArrayUtils.subarray(args, 4, args.length));
				break;
			case "remove":
				AssertUtil.assertNoBadReq(args.length>1, Status.BadReq.illParam, "remove的参数%s的个数必须大于1", argsToString);
				this.remove(args[0],ArrayUtils.subarray(args, 1, args.length));
				break;
			case "retain":
				AssertUtil.assertNoBadReq(args.length>1, Status.BadReq.illParam, "retain的参数%s的个数必须大于1", argsToString);
				this.retain(args[0],ArrayUtils.subarray(args, 1, args.length));
				break;
			case "merge":
				AssertUtil.assertNoBadReq(args.length==5, Status.BadReq.illParam, "merge的参数%s的个数必须等于5", argsToString);
				this.merge(args[0], args[1],args[2],args[3],args[4]);
				break;
			case "mergeList":
				AssertUtil.assertNoBadReq(args.length==5, Status.BadReq.illParam, "merge的参数%s的个数必须等于5", argsToString);
				this.mergeList(args[0], args[1],args[2],args[3],args[4]);
				break;
			case "mapper":
				AssertUtil.assertNoBadReq(args.length==4, Status.BadReq.illParam, "mapper的参数%s的个数必须等于4", argsToString);
				this.mapper(args[0], args[1],args[2],args[3]);
				break;
			case "alias":
				AssertUtil.assertNoBadReq(args.length==3, Status.BadReq.illParam, "alias的参数%s的个数必须等于3", argsToString);
				this.alias(args[0], args[1],args[2]);
				break;
			case "rename":
				AssertUtil.assertNoBadReq(args.length==3, Status.BadReq.illParam, "rename的参数%s的个数必须等于3", argsToString);
				this.rename(args[0], args[1],args[2]);
				break;
			case "json":
				AssertUtil.assertNoBadReq(args.length==4, Status.BadReq.illParam, "json的参数%s的个数必须等于4", argsToString);
				Json.Action action = Json.Action.from(args[0]);
				this.json(action,args[1],args[2],args[3]);
				break;
			case "fill":
				AssertUtil.assertNoBadReq(args.length==3, Status.BadReq.illParam, "fill的参数%s的个数必须等于3", argsToString);
				this.fill(args[0], args[1],args[2]);
				break;
			case "concat":
				AssertUtil.assertNoBadReq(args.length>4, Status.BadReq.illParam, "concat的参数%s的个数必须大于4", argsToString);
				this.concat(args[0], args[1],args[2],args[3],ArrayUtils.subarray(args, 4, args.length));
				break;
			case "export":
				AssertUtil.assertNoBadReq(args.length==1||args.length==2, Status.BadReq.illParam, "export的参数%s必须等于1或者2", argsToString);
				AssertUtil.assertNotFatal(excelColumnDeclareList!=null && !excelColumnDeclareList.isEmpty()
						, "必须配置excelColumnDeclares才能使用导出功能当前找到了:%s",excelColumnDeclareList);
				ExcelColumnDeclares firstExcelColumnDeclares = excelColumnDeclareList.stream()
						.findFirst()
						.orElseThrow(() -> new BadReqException(Status.BadReq.unsupport, ""));
				//TODO 应该做一点断言让错误更加明显
				ResponseExportType exportType = ResponseExportType.valueOf(args[0]);

				switch (exportType){
					case excel:
						//兼容以前没有指定名字的方式。
						if(args.length<=1){
							this.exportExcel(firstExcelColumnDeclares.name());
						}else if (args.length==2){
							this.exportExcel(args[1]);
						}else{
							throw new BadReqException(Status.BadReq.illParam,"导出表格export,excel的命令参数错误:%s",Arrays.asList(args));
						}
						break;
					default:
						throw new ServerException(Status.Server.programConfigJavaFramework,"没有处理导出类型:[%s]",exportType);
				}
				break;
			default:
				throw new BadReqException("暂时不支持[%s]方法", function);
			}
		}
		return this;
	}
	
	
	public String createString(){
		return toString();
	}

	@Override
	public String toString(){
		return currentObject.toString();
	}

	public Object getOrigin() {
		return origin;
	}

	public void setOrigin(Object origin) {
		this.origin = origin;
	}


	public static void setLogger(Logger logger) {
		GsonDataFormatBean.logger = logger;
	}

	public List<ExcelColumnDeclares> getExcelColumnDeclareList() {
		return excelColumnDeclareList;
	}

	public void setExcelColumnDeclareList(List<ExcelColumnDeclares> excelColumnDeclareList) {
		this.excelColumnDeclareList = excelColumnDeclareList;
	}
}
