package net.mingsoft.mdiy.parser;

import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;

import com.alibaba.druid.util.StringUtils;

import freemarker.cache.StringTemplateLoader;
import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import net.mingsoft.basic.util.SpringUtil;
import net.mingsoft.mdiy.biz.ITagBiz;
import net.mingsoft.mdiy.biz.ITagSqlBiz;
import net.mingsoft.mdiy.entity.TagEntity;
import net.mingsoft.mdiy.entity.TagSqlEntity;
import net.mingsoft.mdiy.parser.bean.TagBean;

/**
 * 
 * @ClassName: TagParser
 * @Description:TODO(铭飞标签解析)
 * @author: 铭飞开发团队
 * @date: 2018年8月23日 下午2:12:16
 * 
 * @Copyright: 2018 www.mingsoft.net Inc. All rights reserved.
 */
public class TagParser {

	/**
	 * 需解析内容
	 */
	private String content;

	/**
	 * 数据
	 */
	private Map data = new HashMap();

	private NamedParameterJdbcTemplate jdbc;

	/*
	 * log4j日志记录
	 */
	protected final Logger LOG = LoggerFactory.getLogger(this.getClass());

	private ITagBiz tagBiz;
	
	private ITagSqlBiz tagSqlBiz;

	/**
	 * 标签，方便查询tags，解决map无序存放问题
	 */
	private List<String> tagKeys = new ArrayList<String>();
	
	/**
	 * 返回分页数
	 */
	private int pageSize;

	/**
	 * 标签块代码
	 */
	private Map<String, TagBean> tags = new HashMap<String, TagBean>();
	
	public TagParser(String content) {
		this(content, null);
	}
	/**
	 * 解析构造方法
	 * @param content 读取的最原始模板内容
	 * @param map 全局的变量，每个标签对应的SQL都可以使用
	 */
	public TagParser(String content,Map map) {
		this.content = content;
		//读取对应的标签绑定的SQL
		this.tagBiz = SpringUtil.getBean(ITagBiz.class);
		this.tagSqlBiz = SpringUtil.getBean(ITagSqlBiz.class);
		this.jdbc = SpringUtil.getBean(NamedParameterJdbcTemplate.class);
		//将全局数据保存data管理
		if(map!=null) {
			this.data.putAll(map);
		}
		this.parser();
	}

	public String getContent() {
		return content;
	}

	/**
	 * 最基本的规则替换
	 * @return
	 */
	public TagParser parser() {
		this.parserSingle().parserData();
		this.parserDoublue().parserData();
		this.content = this.parserFreemarker(this.content);
		this.rendering();
		return this;
	}

	private TagParser parserData() {
		return this.parserData(this.data);
	}

	/**
	 * 解析 参考https://www.cnblogs.com/VergiLyn/p/6161081.html
	 * 
	 * @param param
	 *            sql语句where字段，key表示字段 value值
	 * @return
	 */
	@SuppressWarnings("unchecked")
	private TagParser parserData(Map root) {
		Map map = new HashMap<>();
		// 遍历tagKey,读取标签对应的sql,
		for (String tagName : this.tagKeys) {
			if(root.get(tagName)!=null) {
				continue;
			}
			TagBean tagBean = this.tags.get(tagName);
			// 获取对应的SQL语句
			TagEntity tag = new TagEntity();
			//替换成数据库对应的标签名称
			tag.setTagName(tagName.split("_")[0].replace("ms:", ""));
			tag = (TagEntity) tagBiz.getEntity(tag);
			if (tag == null) {
				continue;
			}
			
			
			if (tagBean != null) { //双标签数据解析
				Map tagParams = new HashMap();
				//添加需要查询的条件参数
				tagParams.putAll(this.data);
				tagParams.putAll(tagBean.getParams());
				Map refs = new HashMap();
				//refs判断是否是子标签还是父标签
				Object tagRefs = tagBean.getParams().get("refs");
				if (tagRefs != null) { //子标签
					//从root数据中取出对应父对象数据
					Object obj = root.get(tagRefs.toString().trim()); 
					Map mapData = (Map) obj;
					Iterator it = mapData.keySet().iterator();
					
					while (it.hasNext()) {
						List list = (List) mapData.get(it.next());
						
						//遍历父标签对应的数据列表，主要目的是让子标签获得父标签对象
						for (int i = 0; i < list.size(); i++) {
							//获取当前列表所有列
							Map row = (Map) list.get(i);
							//合并到当前标签数据，为了更好的传递给sql
							tagParams.putAll(row);
							//例如freemarker渲染sql语句
							
							List<TagSqlEntity> sqlList = tagSqlBiz.query(Integer.parseInt(tag.getId()));
							String sql = this.rendering(tagParams, sqlList.get(0).getTagSql());
							
							//执行sql
							Map<String, Object> whereParams = new HashMap<String, Object>();
							List _list = jdbc.queryForList(sql, whereParams);
							
							//将返回数据保存的到数据模型
							root.put(tagName + row.get("id"), _list);

							// 当前标签还存在被引用，或子类标签
							if (tagBean.getParams().get("ref") != null) {
								refs.put(tagBean.getParams().get("ref").toString() + row.get("id"), _list); 
								root.put(tagBean.getParams().get("ref"), refs); 
								TagBean child = tagBean.getChild();
								String ftl = "";
								if (child != null) {
									String temp = tagBean.getContent().replace(child.getContent(),child.getBeginTag().split(":")[1].trim() + "${item.id}");
									ftl = this.parserFreemarker(temp);
								} else {
									ftl = this.parserFreemarker(tagBean.getContent());
								}
								String cont = this.rendering(root, ftl.replace(tagName, tagName + row.get("id")));
								this.content = content.replace(tagName + row.get("id"), cont);
							} else {
								if (_list != null) {
									String ftl = this.parserFreemarker(tagBean.getContent());
									String cont = this.rendering(root, ftl.replace(tagName, tagName + row.get("id")));
									this.content = content.replace(tagName + row.get("id"), cont);
								}
							}
						}
					}

				} else { //父标签或普通对称标签
					
					//取出对称标签对应的sql并执行
					List<TagSqlEntity> sqlList = tagSqlBiz.query(Integer.parseInt(tag.getId()));
					//补全sql语句条件参数
					String sql = this.rendering(tagParams, sqlList.get(0).getTagSql());
					//LOG.debug(sql);
					Map<String, Object> whereParams = new HashMap<String, Object>();
					List list = jdbc.queryForList(sql, whereParams);
					root.put(tagName, list);
					
					//如果是存在被子类标签使用，需要更具ref定义的值，将数据存在在root数据结构，这里会存在当前标签数据保存了两份
					if (tagBean.getParams().get("ref") != null) {
						refs.put(tagBean.getParams().get("ref").toString(), list); 
						root.put(tagBean.getParams().get("ref"), refs); 
						//获取当前标签的子标签内容
						TagBean child = tagBean.getChild();
						//暂时替换成临时的标签名 例如:arclist_0
						String temp = tagBean.getContent().replace(child.getContent(),child.getBeginTag().split(":")[1].trim() + "${item.id}");
						//替换成ftl标签
						String ftl = this.parserFreemarker(temp);
						//ftl模版数据渲染
						String cont = this.rendering(root, ftl);
						//将标签对应的原始内容替换成渲染之后的数据
						this.content = this.content.replace(tagBean.getContent(), cont);
					}

				}
				
				if(tagBean.getParams().get("ispaging") != null) {
					this.data.remove("ispaging");
				}
			} else { // 单标签数据解析
				List<TagSqlEntity> sqlList = tagSqlBiz.query(Integer.parseInt(tag.getId()));
				String sql = this.rendering(this.data, sqlList.get(0).getTagSql());
				Map<String, Object> whereParams = new HashMap<String, Object>();
				List list = jdbc.queryForList(sql, whereParams);
				//栏目为封面且不存在文章的情况
				if(list.size()==0){
					//赋值一个默认文章
					Map article = new HashMap();
					list.add(article);
				}
				root.put(tagName, list.get(0));
			}

		}
		return this;
	}
	
	/**
	 * 解析当前模版分页数，如果没有设置size属性，默认返回10
	 * @param content
	 * @return 
	 */
	public static int getPageSize(String content) {
		Pattern r = Pattern.compile("\\{ms.*?}");
		Matcher m = r.matcher(content);
		int index = 0;
		
		TagBean pageTag = null;

		// 解决标签嵌套问题，将所有的开始标签添加序号
		while (m.find()) {
			TagBean tagBean = new TagBean();
			String line = m.group(0);
			String tag = line.split(" ")[0].replace("{", "").replace("}", "");
			String newTag = tag + "_" + index;
			content = content.replace(line, line.replace(tag, newTag));
			tagBean.setBeginTag(line.replace(tag, newTag));
			tagBean.setEndTag("{/" + newTag.split(":")[1] + "}");
			//解析标签的属性值，根据空格分隔
			String[] temp = line.replace("}", "").replace(newTag, "").split(" ");
			//遍历属性值
			for (String p : temp) {
				if (!StringUtils.isEmpty(p)) {
					String[] _p = p.split("=");
					//如果不=2，表示属性值编写的不完整
					if (_p.length == 2) {
						tagBean.getParams().put(_p[0], _p[1]);
						//判断是否是分页列表 
						if(_p[0].equalsIgnoreCase("ispaging") && _p[1].equals("true")) {
							pageTag = tagBean;
						}
					}
				}
			}
			tagBean.setContent(line.replace(tag, newTag));
			if(pageTag != null) {
				break;
			}
			
		}
		
		if(pageTag == null) {
			return 0;
		}
		if(pageTag.getParams().get("size")!=null) {
			return Integer.parseInt(pageTag.getParams().get("size")+"");
		} else {
			return 20;
		}
		
		
	}

	/**
	 * 解析双标签 例如： {ms:arclist flag=c flag=f size=5 titlelen=45 typeid=2734} ....
	 * {/ms:arclist} 第一次变化:注意是为了做嵌套标签 {ms:arclist-0 flag=c flag=f size=5
	 * titlelen=45 typeid=2734} ....[field.name/] {/ms:arclist-0} 最后一次变成 <#list
	 * arclist-0 as item> ....${item.name} </#list>
	 * 
	 * @param content
	 *            被解析的内容
	 * @return
	 */
	private TagParser parserDoublue() {
		// 从正则表达式实例中运行方法并查看其如何运行
		Pattern r = Pattern.compile("\\{ms.*?}");
		Matcher m = r.matcher(content);
		int index = 0;
		

		// 解决标签嵌套问题，将所有的开始标签添加序号
		while (m.find()) {
			TagBean tagBean = new TagBean();
			String line = m.group(0);
			//取出标签
			String tag = line.split(" ")[0].replace("{", "").replace("}", "");
			//标序号
			String newTag = tag + "_" + index;
			content = content.replace(line, line.replace(tag, newTag));
			tagBean.setBeginTag(line.replace(tag, newTag));
			tagBean.setEndTag("{/" + newTag.split(":")[1] + "}");
			String[] temp = line.replace("}", "").replace(newTag, "").split(" ");
			for (String p : temp) {
				if (!StringUtils.isEmpty(p)) {
					String[] _p = p.split("=");
					if (_p.length == 2) {
						tagBean.getParams().put(p.split("=")[0], p.split("=")[1]);
					}
				}
			}
			tagBean.setContent(line.replace(tag, newTag));

			tags.put(newTag.split(":")[1], tagBean);
			tagKeys.add(newTag.split(":")[1]);
			index++;
		}
		
		

		// 结束标签添加序号
		for (int i = tagKeys.size() - 1; i >= 0; i--) {
			String endTag = tagKeys.get(i).split("_")[0];
			TagBean tag = tags.get(tagKeys.get(i));
			if (tag == null) { // 跳过单标签
				continue;
			}
			String p = tag.getContent().replace("{", "\\{") + "([\\w\\W]*?)\\{/ms:" + endTag + "}";
			Pattern pt = Pattern.compile(p);
			Matcher mt = pt.matcher(content);
			while (mt.find()) {
				String temp = mt.group(0).replace("/ms:" + endTag + "}", "/" + tagKeys.get(i) + "}");
				TagBean tagBean = tags.get(tagKeys.get(i));
				tagBean.setContent(temp);
				content = content.replace(mt.group(0), temp);
			}
		}
		return this;
	}

	/**
	 * 快速替换成ftl标签
	 * 
	 * @return
	 */
	private String parserFreemarker(String content) {
		// 转换freemarker
		for (String str : tagKeys) {
	         if (str.indexOf("if") > -1) {
	            String _if = tags.get(str).getBeginTag().replace("{ms:" + str, "<#if ").replace("}", ">");
	            content = content.replace(tags.get(str).getBeginTag(), _if);
	            content = content.replace(tags.get(str).getEndTag(), "</#if>");
	            content = content.replace("{/ms:if}", "</#if>");
	            
	         } else {
	            if (tags.get(str) == null) { // 跳过单标签
	               continue;
	            }
	            content = content.replace(tags.get(str).getBeginTag(),
	                  "<#list " + str.replace("ms:", "") + " as item>");
	            content = content.replace(tags.get(str).getEndTag(), "</#list>");

	         }
	      }
	      content = content.replace("{ms:else}", "<#else>");
	      Pattern p = Pattern.compile("\\[.*?/]");
	      Matcher mt = p.matcher(content);
	      while (mt.find()) {
	         String field = mt.group(0);
	         if (field.indexOf("field.") > 0) {
	            field = mt.group(0).replace("[field.", "${item.").replace("/]", "!''}").replace("(!?\\d*)]", "$1!''}");
	         }
	         else if(field.indexOf("_root:") > 0) {
	          field = mt.group(0).replace("[_root:", "${").replace("/]", "}").replace("]", "}");
	         }
	         content = content.replace(mt.group(0), field);
	      }
	      return content;
	}

	/**
	 * 解析单标签 例如： 调用rendering方法到时候设置global对象的name值 {ms:global.name/} 变成
	 * ${global.name} {ms:global.logo/} 变成 ${global.logo}
	 * 
	 * @param content
	 *            被解析的内容，将标签转换成freemarker书写格式
	 * 
	 * @return
	 */
	private TagParser parserSingle() {
		Pattern pattern = Pattern.compile("\\{ms:+[\\S].[^\\{}]+?/}");
		Matcher matcher = pattern.matcher(content);
		while (matcher.find()) {
			String text = matcher.group(0);
			content = content.replace(text, matcher.group(0).replace("ms:", "").replace("{", "${").replace("/}", "!''}"));
			String key = text.split(":")[1].split("\\.")[0];
			//将标签保存序列，方便遍历使用
			tagKeys.add(key);
			this.tags.put(key, null);
		}
		return this;
	}

	/**
	 * 使用外部变量渲染数据，例如栏目生产时需要传递栏目编号
	 * @param map 传递参数，可以被sql、模版标签解析到
	 * @return 生成好到文件
	 */
	public String rendering(Map map) {
		if(map!=null) {
			this.data.putAll(map);
		}
		return this.rendering(this.data, this.content);
	}
	
	/**
	 * 数据渲染，通过freemarker模板方式映射数据
	 * @return
	 */
	public String rendering() {
		//content已经全部替换完成的内容
		return this.rendering(this.data, this.content);
	}

	/**
	 * 数据渲染，通过freemarker模板方式映射数据
	 * @param content
	 *            模板内容
	 * @param root
	 *            数据
	 * @return
	 */
	private  String rendering(Map root, String content) {
		Configuration cfg = new Configuration();
		StringTemplateLoader stringLoader = new StringTemplateLoader();
		stringLoader.putTemplate("template", content);
		cfg.setNumberFormat("#");
		cfg.setTemplateLoader(stringLoader);
		try {
			Template template = cfg.getTemplate("template", "utf-8");
			StringWriter writer = new StringWriter();
			try {
				template.process(root, writer);
				content = writer.toString();
			} catch (TemplateException e) {
				e.printStackTrace();
				LOG.error("错误",e);
			}
		} catch (IOException e) {
			e.printStackTrace();
			LOG.error("错误",e);
		}
		return content;
	}

	public int getPageSize() {
		return pageSize;
	}
	
}
