package com.mini.framework.third.weixin.server.model.template;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.gson.Gson;
import com.mini.framework.core.exception.BadReqException;
import com.mini.framework.core.exception.standard.CustomException;
import com.mini.framework.core.status.Status;
import com.mini.framework.core.status.StatusCode;

public class TemplateDocumentContent {

	private List<TemplateContentItem> contentItems;
	
	private String keywordFillExample;

	public List<TemplateContentItem> getContentItems() {
		return contentItems;
	}

	public void setContentItems(List<TemplateContentItem> contentItems) {
		this.contentItems = contentItems;
	}
	
	public String getKeywordFillExample() {
		return keywordFillExample;
	}

	public void setKeywordFillExample(String keywordFillExample) {
		this.keywordFillExample = keywordFillExample;
	}

	public Stream<TemplateContentItem> showVariableItemsStream(){
		return showVariableItems().stream();
	}
	public List<TemplateContentItem> showVariableItems(){
		return showContentItemsStream().filter(TemplateContentItem::ifVariable).collect(Collectors.toList());
	}
	public Stream<TemplateContentItem> showContentItemsStream(){
		return showContentItems().stream();
	}
	public List<TemplateContentItem> showContentItems(){
		return Optional.ofNullable(contentItems).orElse(new ArrayList<>());
	}
	
	public static TemplateDocumentContent fromTextDocument(String text){
		List<TemplateContentItem> items = new ArrayList<TemplateContentItem>();
		TemplateDocumentContent content = new TemplateDocumentContent();
		content.setContentItems(items);
		String variablePattern = "\\{\\{([a-zA-Z0-9]*)\\.DATA\\}\\}";
		Pattern pattern = Pattern.compile(variablePattern);
		String[] staticContents = pattern.split(text);
		Matcher matcher = pattern.matcher(text);
		List<String> matchs = new ArrayList<>();
		while(matcher.find()){
			String group = matcher.group(1);
			matchs.add(group);
		}
		for (int i = 0; i < matchs.size(); i++) {
			items.add(TemplateContentItem.createStaticItem(staticContents[i]));
			items.add(TemplateContentItem.createVariableItem(matchs.get(i),""));
		}
		if(matchs.size() + 1 == staticContents.length){
			items.add(TemplateContentItem.createStaticItem(staticContents[matchs.size()]));
		}
		return content ;
	}
	
	
	
	public String applyTemplateByParams(Map<String, Object> params){
		return contentItems.stream().map(item->item.toValue(params)).collect(Collectors.joining(""));
	}
	
	public String applyTemplateByParamAsValue(){
		return applyTemplateByParamFinder(Function.identity());
	}
	
	
	public String applyTemplateByParamFinder(Function<String,String> variableValue){
		Map<String, Object> params = new LinkedHashMap<String, Object>();
		contentItems.stream().filter(TemplateContentItem::ifVariable)
		.map(TemplateContentItem::getVarName)
		.forEach(vn->params.put(vn, variableValue.apply(vn)));
		return applyTemplateByParams(params );
	}
	
	public static TemplateDocumentContent fromJson(String json){
		return new Gson().fromJson(json, TemplateDocumentContent.class);
	}
	
	
	public String applyTemplateByFixedParam(String variableValue){
		return applyTemplateByParamFinder(vn->variableValue);
	}
	
	public TemplateDocumentContent createKeywordFillExample(Function<String, String> variableValue){
		this.keywordFillExample = applyTemplateByParamFinder(variableValue);
		return this;
	}
	
	public TemplateDocumentContent createKeywordFillExample(){
		return createKeywordFillExample(var->String.format("#{{%s}}", var));
	}
	
	/**
	 * 检查是不是匹配，如果不匹配返回不匹配的原因
	 * @param other
	 * @return
	 */
	public Optional<String> matchSameDocumentVersionFindCause(TemplateDocumentContent other){
		StatusCode status = Status.BadReq.illParam;
		try{
			validateSameDocumentVersion(other, (msg)->new BadReqException(status, msg));
			return Optional.empty();
		}catch(BadReqException e){
			if(e.matchStatus(status)){
				return Optional.of(e.getMessage());
			}else{
				throw e;
			}
		}
	}
	
	
	/**
	 * 是不是匹配同一个文档
	 * @return
	 */
	public <E extends CustomException> void validateSameDocumentVersion(TemplateDocumentContent other,Function<String,E>exceptionSupplier){
		if(this.contentItems==null && other.contentItems==null){
			return;
		}else if(this.contentItems==null || other.contentItems==null){
			throw exceptionSupplier.apply("contentItems不能一个为空一个不为空");
		}else if(this.contentItems.size()!=other.contentItems.size()){
			throw exceptionSupplier.apply(String.format("contentItems.size必须一样:[%s]!=[%s]",this.contentItems.size(),other.contentItems.size()));
		}else{
			for (int index = 0; index <this. contentItems.size(); index++) {
				TemplateContentItem thisItem = this.contentItems.get(index);
				TemplateContentItem otherItem = other.contentItems.get(index);
				int offset = index + 1;
				thisItem.validateSameDocumentVersion(otherItem,(msg)->exceptionSupplier.apply(String.format("第[%s]记录不匹配 %s",offset,msg)));
			}
		}
		
	}
	
	
}
