package com.mini.framework.third.weixin.wxpay.v3.model.weixin;

import com.google.gson.annotations.SerializedName;
import com.mini.framework.third.weixin.wxpay.util.certificate.WxpayMarketCertificateContainer;
import com.mini.framework.third.weixin.wxpay.v3.client.ClientConfigAttach;
import com.mini.framework.third.weixin.wxpay.v3.client.ParamBeanValidate;
import com.mini.framework.third.weixin.wxpay.v3.model.site.SiteCreatePayOrderRequest;
import com.mini.framework.util.asserts.AssertUtil;
import org.apache.commons.lang3.time.DateFormatUtils;

import java.util.Optional;

/**
 * @author jayheo
 */
public class WxpayCreatePayOrderRequest implements ClientConfigAttach , ParamBeanValidate {

    /**
     * 微信应用id<BR>
     * string[1,32]<br>
     * 必填：是<br>
     * 由微信生成的应用ID，全局唯一。请求基础下单接口时请注意APPID的应用属性，例如公众号场景下，需使用应用属性为公众号的APPID<BR>
     * 例如:wxd678efh567hg6787
     */
    @SerializedName("appid")
    private String appid;


    /**
     * 微信支付商户号<BR>
     * string[1,32]<BR>
     * 必填：是<br>
     * 直连商户的商户号，由微信支付生成并下发。<BR>
     * 例如:1230000109<BR>
     */
    @SerializedName("mchid")
    private String marketKey;


    /**
     * 商品描述<BR>
     * string[1,127]<BR>
     * 必填：是<br>
     * 商品描述<BR>
     * 例如:形象店-深圳腾大-QQ公仔<BR>
     */
    @SerializedName("description")
    private String description;


    /**
     * 商户订单号<BR>
     * string[6,32]<BR>
     * 必填：是<br>
     * 商户系统内部订单号，只能是数字、大小写字母_-*且在同一个商户号下唯一<BR>
     * 例如:1217752501201407033233368018<BR>
     */
    @SerializedName("out_trade_no")
    private String siteOrderNumber;


    /**
     * TODO 能不能使用 Date 呢。
     * 订单失效时间<BR>
     * 	string[1,64]<BR>
     * 必填：是<br>
     * 订单失效时间，遵循rfc3339标准格式，
     * 格式为YYYY-MM-DDTHH:mm:ss+TIMEZONE，
     * YYYY-MM-DD表示年月日，T出现在字符串中，表示time元素的开头，
     * HH:mm:ss表示时分秒，TIMEZONE表示时区（+08:00表示东八区时间，领先UTC 8小时，即北京时间）。
     * 例如：2015-05-20T13:29:35+08:00表示，北京时间2015年5月20日 13点29分35秒。
     * 订单失效时间是针对订单号而言的，由于在请求支付的时候有一个必传参数prepay_id只有两小时的有效期，
     * 所以在重入时间超过2小时的时候需要重新请求下单接口获取新的prepay_id。
     * 其他详见时间规则 time_expire只能第一次下单传值，不允许二次修改，二次修改系统将报错。
     * 如用户支付失败后，需再次支付，需更换原订单号重新下单。<BR>
     * 例如:2018-06-08T10:34:56+08:00<BR>
     */
    @SerializedName("time_expire")
    private String expireDate;


    /**
     * 服务端结果通知地址<BR>
     * string[1,256]<BR>
     * 必填：是<br>
     * 异步接收微信支付结果通知的回调地址，通知url必须为外网可访问的url，不能携带参数。
     * 公网域名必须为https，如果是走专线接入，使用专线NAT IP或者私有回调域名可使用http<BR>
     */
    @SerializedName("notify_url")
    private String resultPushUrl;


    /**
     * 订单金额<BR>
     * 必填：是<br>
     * 订单金额信息<BR>
     */
    @SerializedName("amount")
    private OrderMoney orderAmount;


    @SerializedName("scene_info")
    private TradeScene tradeScene;


    //-----------------------------------------------

    /**
     * 附加数据<BR>
     * string[6,32]<BR>
     * 必填：否<br>
     * 附加数据，在查询API和支付通知中原样返回，可作为自定义参数使用<BR>
     * 例如:自定义数据<BR>
     */
    @SerializedName("attach")
    private String attach;


    // 订单优惠标记 暂时忽略，需要的时候再用 @SerializedName("goods_tag")


    @SerializedName("payer")
    private ClientUser payerClient;



    /**
     * 填充人民币，单位是分钱。
     * @param amount
     * @return
     */
    public WxpayCreatePayOrderRequest fillChinaAmount(int amount){
        AssertUtil.assertNotFatalBug(amount>0,"支付金额必须大于0");
        orderAmount = OrderMoney.createChinaAmount(amount);
        return this;

    }

    /**
     * 填充一个默认的 场景
     * @return
     */
    public WxpayCreatePayOrderRequest fillDefaultScene(){
        tradeScene = TradeScene.createDefault();
        return this;
    }


    /**
     * 填充openid信息
     * html5不需要 openid信息
     * @param openid
     * @return
     */
    public WxpayCreatePayOrderRequest fillOpenid(String openid){
        payerClient = ClientUser.createByOpenid(openid);
        return this;
    }



    public static WxpayCreatePayOrderRequest createHtml5OrderFrom(SiteCreatePayOrderRequest request){
        AssertUtil.assertNotFatalBug(request.getOpenid()==null,"h5支付不需要openid");
        return createOrderFrom(request);
    }

    public static WxpayCreatePayOrderRequest createJsapiOrderFrom(SiteCreatePayOrderRequest request){
        AssertUtil.assertMethodRequire(request.getOpenid(),"request.getOpenid()");
        return createOrderFrom(request);
    }


    /**
     * 创建一个html5的订单
     * TODO 根据这个再弄一个jsapi的订单
     * @param request
     * @return
     */
    private static WxpayCreatePayOrderRequest createOrderFrom(SiteCreatePayOrderRequest request){
        WxpayCreatePayOrderRequest instance = new WxpayCreatePayOrderRequest();

        // 通过   applyClientConfig 方法会被设置
        instance.marketKey = null;

        instance.appid = request.getAppid();
        instance.expireDate = request.getExpireDate().toInstant().toString();

        instance.description = request.getOrderName();
        instance.siteOrderNumber = request.getOrderNumber();
        instance.resultPushUrl = request.getResultPushUrl();

        Optional.ofNullable(request.getOpenid()).ifPresent(instance::fillOpenid);
        instance.fillDefaultScene();
        instance.fillChinaAmount(request.getOrderAmount());

        return instance;
    }

    //TODO 还要增加珍个自检的动作。

    @Override
    public void applyClientConfig(WxpayMarketCertificateContainer certificateContainer) {
        this.marketKey = certificateContainer.getMarketKey();
    }

    @Override
    public void beforeRequestValidate() {
        //TODO 还要做一下其他的检查。
        AssertUtil.assertNotFatal(marketKey!=null,"marketKey 不能为空");
        AssertUtil.assertNotFatal(appid!=null,"appid 不能为空");
    }


    public static class ClientUser{

        /**
         * 用户标识<BR>
         * string[6,32]<BR>
         * 必填：是<br>
         * 用户在直连商户appid下的唯一标识。 下单前需获取到用户的Openid<BR>
         * 例如:oUpF8uMuAJO_M2pxb1Q9zNjWeS6o<BR>
         */
        @SerializedName("openid")
        private String openid;

        public static ClientUser createByOpenid(String openid){
            AssertUtil.assertMethodRequire(openid,"openid");
            ClientUser instance = new ClientUser();
            instance.openid = openid;
            return instance;
        }

        public String getOpenid() {
            return openid;
        }

        public void setOpenid(String openid) {
            this.openid = openid;
        }
    }

    /**
     * 交易的场景
     */
    public static class TradeScene {

        /**
         * 用户终端IP<BR>
         * 用户的客户端IP，支持IPv4和IPv6两种格式的IP地址。<BR>
         * 例如:14.23.150.211<BR>
         */
        @SerializedName("payer_client_ip")
        private String payerClientIp;

        public String getPayerClientIp() {
            return payerClientIp;
        }

        public void setPayerClientIp(String payerClientIp) {
            this.payerClientIp = payerClientIp;
        }

        public static TradeScene createDefault(){
            TradeScene instance = new TradeScene();
            //目前这个地址没有什么用，所以直接写死了。测试过写死也不会影响结果。
            instance.payerClientIp = "192.168.0.0";
            return instance;
        }
    }


    public static class OrderMoney{


        /**
         * 总金额<BR>
         * 必填：是<br>
         * 订单总金额，单位为分。<BR>
         * 例如:100<BR>
         */
        @SerializedName("total")
        private Integer amount;


        /**
         * 货币类型<BR>
         * string[1,16]<BR>
         * 必填：是<br>
         * 	CNY：人民币，境内商户号仅支持人民币。<BR>
         * 例如:CNY<BR>
         */
        @SerializedName("currency")
        private String currency;

        public static OrderMoney createChinaAmount(int amount){
            OrderMoney instance = new OrderMoney();
            instance.amount = amount;
            instance.currency = "CNY";
            return instance;
        }

        public Integer getAmount() {
            return amount;
        }

        public void setAmount(Integer amount) {
            this.amount = amount;
        }

        public String getCurrency() {
            return currency;
        }

        public void setCurrency(String currency) {
            this.currency = currency;
        }
    }


    public String getAppid() {
        return appid;
    }

    public void setAppid(String appid) {
        this.appid = appid;
    }

    public String getMarketKey() {
        return marketKey;
    }

    public void setMarketKey(String marketKey) {
        this.marketKey = marketKey;
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public String getSiteOrderNumber() {
        return siteOrderNumber;
    }

    public void setSiteOrderNumber(String siteOrderNumber) {
        this.siteOrderNumber = siteOrderNumber;
    }

    public String getExpireDate() {
        return expireDate;
    }

    public void setExpireDate(String expireDate) {
        this.expireDate = expireDate;
    }

    public String getResultPushUrl() {
        return resultPushUrl;
    }

    public void setResultPushUrl(String resultPushUrl) {
        this.resultPushUrl = resultPushUrl;
    }

    public OrderMoney getOrderAmount() {
        return orderAmount;
    }

    public void setOrderAmount(OrderMoney orderAmount) {
        this.orderAmount = orderAmount;
    }

    public String getAttach() {
        return attach;
    }

    public void setAttach(String attach) {
        this.attach = attach;
    }

    public TradeScene getTradeScene() {
        return tradeScene;
    }

    public void setTradeScene(TradeScene tradeScene) {
        this.tradeScene = tradeScene;
    }

    public ClientUser getPayerClient() {
        return payerClient;
    }

    public void setPayerClient(ClientUser payerClient) {
        this.payerClient = payerClient;
    }
}
