package com.mini.framework.third.weixin.mapp;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.FutureTask;

import com.mini.framework.third.weixin.mapp.model.bean.SubscribeMessageParam;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;

import com.mini.framework.core.exception.HandleException;
import com.mini.framework.third.weixin.mapp.model.AccessTokenResponse;
import com.mini.framework.third.weixin.mapp.model.MappCodePhotoRequest;
import com.mini.framework.third.weixin.mapp.model.SimpleAuthResponse;
import com.mini.framework.third.weixin.mapp.model.SimpleResponse;
import com.mini.framework.third.weixin.mapp.model.template.MessageTemplateLibraryResponse;
import com.mini.framework.third.weixin.mapp.model.template.MessageTemplateResponse;
import com.mini.framework.third.weixin.mapp.model.template.SiteMessageTemplatesResponse;

/**
 * @author jayheo
 *
 */
public interface WeixinMappSdk {

	String mappSdkPackage = "com.mini.framework.third.weixin.mapp";
	
	/**
	 * 登录凭证校验。通过 wx.login 接口获得临时登录凭证 code 后传到开发者服务器调用此接口完成登录流程。更多使用方法详见<BR>
	 * https://developers.weixin.qq.com/miniprogram/dev/api-backend/open-api/login/auth.code2Session.html#Object
	 * @param appid
	 * @param secret
	 * @param code
	 * @return
	 */
	@GetMapping("/sns/jscode2session?appid=%s&secret=%s&js_code=%s&grant_type=authorization_code")
	public SimpleAuthResponse queryAuthByCode(String appid,String secret,String code);
	
	
	
	/**
	 * <pre>
	https://developers.weixin.qq.com/miniprogram/dev/api/getWXACodeUnlimit.html
		如果调用成功，会直接返回图片二进制内容，如果请求失败，会返回 JSON 格式的数据。
		注意看一下，微信得到的图片是430*430
		https://aqukedev.oss-cn-shanghai.aliyuncs.com/file/7/201903/24/20190324143603714344020.png
	 * </pre>
	 * @param accessToken
	 * @param req
	 * @return
	 */
	@PostMapping("/wxa/getwxacodeunlimit?access_token=%s")
	public byte[] createMappQrcode(String accessToken,MappCodePhotoRequest req);
	
	
	/**
	 * 创建accesstoken
	 * @return
	 */
	@GetMapping("/cgi-bin/token?grant_type=client_credential&appid=%s&secret=%s")
	public AccessTokenResponse createAccessToken(String appid,String secret);
	
	
	
	///-----------------------------------小程序模板消息
	
	
	
	/**
	 * 获取小程序模板库标题列表
	 */
	@PostMapping("/cgi-bin/wxopen/template/library/list?access_token=%s")
	public MessageTemplateLibraryResponse listWeixinMessageTemplates(String accessToken,int offset,int limit);
	
	
	
	ExecutorService executor = Executors.newCachedThreadPool();

	/**
	 * 获取小程序模板库标题列表，附带详情
	 * @param accessToken
	 * @param offset
	 * @param limit
	 * @return
	 */
	default public MessageTemplateLibraryResponse listWeixinMessageTemplates(String accessToken,int offset,int limit,boolean detail){
		if(!detail){
			return listWeixinMessageTemplates(accessToken, offset, limit);
		}
		
		MessageTemplateLibraryResponse resp = listWeixinMessageTemplates(accessToken, offset, limit);
		List<MessageTemplateResponse> templates = resp.getTemplates();
		List<FutureTask<MessageTemplateResponse>> tasks = new ArrayList<>(); 
		
		for (int i = 0; i < templates.size(); i++) {
			String key = templates.get(i).getKey();
			FutureTask<MessageTemplateResponse> task = new FutureTask<>(()->{
				MessageTemplateResponse tempDesc = getWeixinMessageTemplate(accessToken, key);
				return tempDesc;
			});
			executor.submit(task);
			tasks.add(task);
		}
		templates.clear();
		
		for (FutureTask<MessageTemplateResponse> futureTask : tasks) {
			try {
				MessageTemplateResponse tempDesc = futureTask.get();
				templates.add(tempDesc);
			} catch (InterruptedException | ExecutionException e) {
				throw new HandleException(e,"在执行多任务时出错");
			}
		}
		return resp;
	}
	
	
	/**
		删除帐号下的某个模板
	 * @param accessToken
	 * @param templateKey  site下的模板 key
	 */
	@PostMapping("/cgi-bin/wxopen/template/del?access_token=%s")
	public void deleteSiteMessageTemplate(String accessToken,String siteTemplateId);

	
	
	/**
	 * 获取模板库某个模板标题下关键词库
	 * @param accessToken
	 * @param templateKey
	 * @return
	 */
	@PostMapping("/cgi-bin/wxopen/template/library/get?access_token=%s")
	public MessageTemplateResponse getWeixinMessageTemplate(String accessToken,String templateKey);
	
	
	
	
	/**
	 * 组合模板并添加至帐号下的个人模板库
	 * @param accessToken
	 * @param weixinTemplateKey 微信侧模板key
	 * @param keywordKeys 需要使用关键词
	 * @return 如果成功返回模板id，否则会报出异常
	 */
	@PostMapping("/cgi-bin/wxopen/template/add?access_token=%s")
	public String createSiteMessageTemplate(String accessToken,String weixinTemplateKey,int[] keywordKeys);
	
	
	

	/**
	 * 获取帐号下所有已存在的模板列表
	 * ,int offset,int limit 因为最多只会有25个模板可以没有必要在分页了
	 * @param accessToken
	 * @return
	 */
	@PostMapping("/cgi-bin/wxopen/template/list?access_token=%s")
	public SiteMessageTemplatesResponse getSiteTemplates(String accessToken);

	
	/**
	 * https://developers.weixin.qq.com/miniprogram/dev/api-backend/open-api/template-message/templateMessage.send.html
	发送模板消息
	 * @param accessToken
	 * @param openid 必填 接收者（用户）的 openid
	 * @param siteTemplateId 必填 所需下发的模板消息的id
	 * @param sendToken 发信息的授权token
	 * @param mappNamespace 选填 点击模板卡片后的跳转页面，仅限本小程序内的页面。支持带参数,（示例index?foo=bar）。该字段不填则模板无跳转。
	 * @param params 参数序列  参数序列是 keyword1234，所以直接用数组就可以了
	 * @param importantKeywordKeys 模板需要放大的关键词，不填则默认无放大
	 * @return
	 */
	@PostMapping("/cgi-bin/message/wxopen/template/send?access_token=%s")
	public SiteMessageTemplatesResponse sendTemplateMessage(String accessToken,String openid,String siteTemplateId,String sendToken,
			String mappNamespace,String[] params,String[] importantKeywordKeys);
	
	
	
	/**
	 * 
{
  "touser":"OPENID",
  "msgtype":"text",
  "text":
  {
    "content":"Hello World"
  }
}
	 * @param accessToken
	 * @param openid
	 * @param text
	 * @return
	 */
	@PostMapping("/cgi-bin/message/custom/send?access_token=%s")
	SimpleResponse sendCustomMessageText(String accessToken, String openid, String text);


	/**
{
 "touser":"OPENID",
 "msgtype":"miniprogrampage",
 "miniprogrampage": {
   "title":"title",
   "pagepath":"pagepath",
   "thumb_media_id":"thumb_media_id"
 }
}
	/**
	 * @param accessToken
	 * @param openid
	 * @param title 标题
	 * @param namespace  打开小程序的命名空间
	 * @param photoUrl 小程序上的图片地址
	 * @return
	 */
	@PostMapping("/cgi-bin/message/custom/send?access_token=%s")
	SimpleResponse sendCustomMessageMapp(String accessToken, String openid, String title, String namespace,
			String photoUrl);



	/**
{
  "touser": "OPENID",
  "msgtype": "link",
  "link": {
    "title": "Happy Day",
    "description": "Is Really A Happy Day",
    "url": "URL",
    "thumb_url": "THUMB_URL"
  }
}
	 * @param accessToken
	 * @param openid
	 * @param title 标题
	 * @param desc 描述
	 * @param linkUrl 跳转的链接
	 * @param icoUrl 小图标
	 * @return
	 */
	@PostMapping("/cgi-bin/message/custom/send?access_token=%s")
	SimpleResponse sendCustomMessagePhotoText(String accessToken, String openid, String title, String desc,
			String linkUrl, String icoUrl);


	/**
{
  "touser":"OPENID",
  "msgtype":"image",
  "image": {
    "media_id":"MEDIA_ID"
  }
}
	 * @param accessToken
	 * @param openid
	 * @param text
	 * @return
	 */
	@PostMapping("/cgi-bin/message/custom/send?access_token=%s")
	SimpleResponse sendCustomMessagePhoto(String accessToken, String openid, String photoUrl);


	/**
	 * 发送订阅消息
	 * @param accessToken 访问令牌
	 * @param openid 接收者（用户）的 openid
	 * @param templateId 订阅模板id
	 * @param pathOptional 跳转页面
	 * @param params 参数序列
	 * @return 响应
	 */
	@PostMapping("/cgi-bin/message/subscribe/send?access_token=%s")
	SimpleResponse sendSubscribeMessage(String accessToken, String openid, String templateId, Optional<String>pathOptional, List<SubscribeMessageParam> params);

	
	
}
