package com.mini.framework.third.weixin.mapp;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import com.mini.framework.third.weixin.mapp.model.bean.SubscribeMessageParam;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;
import com.mini.framework.third.weixin.mapp.model.AccessTokenResponse;
import com.mini.framework.third.weixin.mapp.model.MappCodePhotoRequest;
import com.mini.framework.third.weixin.mapp.model.MappCodePhotoResponse;
import com.mini.framework.third.weixin.mapp.model.SimpleAuthResponse;
import com.mini.framework.third.weixin.mapp.model.SimpleResponse;
import com.mini.framework.third.weixin.mapp.model.template.MessageTemplateCreateResponse;
import com.mini.framework.third.weixin.mapp.model.template.MessageTemplateLibraryResponse;
import com.mini.framework.third.weixin.mapp.model.template.MessageTemplateResponse;
import com.mini.framework.third.weixin.mapp.model.template.SiteMessageTemplatesResponse;
import com.mini.framework.third.weixin.server.util.tpmsg.WeixinJsonReqBuilder;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.http.SimpleHttpUtil;
import com.mini.framework.util.string.StringUtil;
import com.mini.framework.util.string.gson.builder.GsonData;

@Component
public class WeixinMappSdkSimpleImpl implements WeixinMappSdk {

	private Logger logger = LogManager.getLogger(WeixinMappSdkSimpleImpl.class);
	
	@Value("${weixin.open.api.url:https://api.weixin.qq.com}")
	private String server;
	
	@Override
	public SimpleAuthResponse queryAuthByCode(String appid,String secret,String code) {
		AssertUtil.assertMethodRequire(appid, "appid");
		AssertUtil.assertMethodRequire(secret, "secret");
		AssertUtil.assertMethodRequire(code, "code");
		String action = "/sns/jscode2session?appid=%s&secret=%s&js_code=%s&grant_type=authorization_code";
		String url = server  + String.format(action ,appid,secret,code);
		String respStr = SimpleHttpUtil.simpleGet(url);
		SimpleAuthResponse resp = new Gson().fromJson(respStr, SimpleAuthResponse.class);
		resp.handle("code:%s",code);
		return resp;
	}

	@Override
	public byte[] createMappQrcode(String accessToken,MappCodePhotoRequest req) {
		AssertUtil.assertMethodRequire(accessToken, "accessToken");
		String action = "/wxa/getwxacodeunlimit?access_token=%s";
		String url = server  + String.format(action ,accessToken);
		String body = req.toJson();
		byte[] bytes  = SimpleHttpUtil.simpleBytePost(url,body);
		try{
			MappCodePhotoResponse resp = new Gson().fromJson(StringUtil.toUtf8String(bytes), MappCodePhotoResponse.class);
			resp.handle("reqBody:%s",body);
		}catch(JsonSyntaxException e){
			//如果能解析说明，有错误，如果不能解析说明没有问题正好是想要的文件流
			logger.debug(e.getMessage() + " 出现这个异常是正常的情况忽略即可");
		}
		return bytes;
	}
	
	@Override
	public AccessTokenResponse createAccessToken(String appid,String secret) {
		AssertUtil.assertMethodRequire(secret, "secret");
		AssertUtil.assertMethodRequire(appid, "appid");
		String action = "/cgi-bin/token?grant_type=client_credential&appid=%s&secret=%s";
		String url = server + String.format(action ,appid,secret);
		String respStr = SimpleHttpUtil.simpleGet(url);
		AccessTokenResponse resp = new Gson().fromJson(respStr, AccessTokenResponse.class);
		resp.handle("url:%s",url);
		return resp;
	}

	@Override
	public MessageTemplateLibraryResponse listWeixinMessageTemplates(String accessToken, int offset, int limit) {
		AssertUtil.assertMethodRequire(accessToken, "accessToken");
		int maxLimit = 20;//微信最大支持查20个
		String action = "/cgi-bin/wxopen/template/library/list?access_token=%s";
		String url = server + String.format(action ,accessToken);
		MessageTemplateLibraryResponse respSum = null;
		boolean haveMore = true;
		do{
			String reqBody = String.format("{\"offset\":%s,\"count\":%s}", offset,Math.min(limit, maxLimit));
			String respStr = SimpleHttpUtil.simplePost(url,reqBody );
			MessageTemplateLibraryResponse resp = new Gson().fromJson(respStr , MessageTemplateLibraryResponse.class);
			resp.handle("reqBody:%s",reqBody);
			haveMore = resp.getTemplates().size()>0;
			if(respSum!=null){
				respSum.addTemplates(resp.getTemplates());
				respSum.setCount(resp.getCount());
			}else{
				respSum = resp;
			}
			limit-=maxLimit;
			offset+=maxLimit;
		}while(limit>0 && haveMore);
		return respSum;
	}

	@Override
	public void deleteSiteMessageTemplate(String accessToken, String siteTemplateKey) {
		// TODO Auto-generated method stub
		
	}

	@Override
	public MessageTemplateResponse getWeixinMessageTemplate(String accessToken, String templateKey) {
		AssertUtil.assertMethodRequire(accessToken, "accessToken");
		String action = "/cgi-bin/wxopen/template/library/get?access_token=%s";
		String url = server + String.format(action ,accessToken);
		String reqBody =  String.format("{\"id\":\"%s\"}",templateKey);
		String respStr = SimpleHttpUtil.simplePost(url,reqBody );
		MessageTemplateResponse resp = new Gson().fromJson(respStr , MessageTemplateResponse.class);
		resp.handle("reqBody:%s",reqBody);
		return resp;
	}

	
	//TODO com.mini.framework.core.exception.ThirdException: error:45026,msg:template num exceeds limit hint: [48juJa00781892] (420000) 这个要怎么办
	@Override
	public String createSiteMessageTemplate(String accessToken, String weixinTemplateKey, int[] keywordKeys) {
		AssertUtil.assertMethodRequire(accessToken, "accessToken");
		String action = "/cgi-bin/wxopen/template/add?access_token=%s";
		String url = server + String.format(action ,accessToken);
		String reqBody =  String.format("{\"id\":\"%s\",\"keyword_id_list\":%s}",weixinTemplateKey,Arrays.toString(keywordKeys));
		String respStr = SimpleHttpUtil.simplePost(url,reqBody );
		MessageTemplateCreateResponse resp = new Gson().fromJson(respStr , MessageTemplateCreateResponse.class);
		resp.handle("reqBody:%s",reqBody);
		return resp.getTemplateId();
	}

	@Override
	public SiteMessageTemplatesResponse getSiteTemplates(String accessToken) {
		// TODO Auto-generated method stub
		return null;
	}

	/*
	 * https://developers.weixin.qq.com/miniprogram/dev/api-backend/open-api/template-message/templateMessage.send.html
	 *  (non-Javadoc)
	 * @see com.mini.framework.third.weixin.mapp.WeixinMappSdk#sendTemplateMessage(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String[], java.lang.String[])
	 */
	@Override
	public SiteMessageTemplatesResponse sendTemplateMessage(String accessToken, String openid, String siteTemplateId,
			String sendToken, String mappNamespace,String[] params, String[] importantKeywordKeys) {
		WeixinJsonReqBuilder msgBuilder = WeixinJsonReqBuilder.create()
				.openid(openid)
				.tempId(siteTemplateId)
				.mappPathIfAbsent(mappNamespace)
				.mappToken(sendToken)
				.keyword(params);
		String reqBody = msgBuilder.build();
		AssertUtil.assertMethodRequire(accessToken, "accessToken");
		String action = "/cgi-bin/message/wxopen/template/send?access_token=%s";
		String url = server + String.format(action ,accessToken);
		String respStr = SimpleHttpUtil.simplePost(url, reqBody);
		SiteMessageTemplatesResponse resp = new Gson().fromJson(respStr, SiteMessageTemplatesResponse.class);
		resp.handle("reqBody:%s",reqBody);
		
		return resp;
	}

	/**
	 * 
{
  "touser":"OPENID",
  "msgtype":"text",
  "text":
  {
    "content":"Hello World"
  }
}
	 * @param accessToken
	 * @param openid
	 * @param text
	 * @return
	 */
	@Override
	public SimpleResponse sendCustomMessageText(String accessToken,String openid,String text) {
		String reqBody = GsonData.build().param("touser", openid).param("msgtype", "text")
				.object("text", textData->textData.param("content", text)).toJson();
		String action = "/cgi-bin/message/custom/send?access_token=%s";
		String url = server + String.format(action ,accessToken);
		String respStr = SimpleHttpUtil.simplePost(url, reqBody);
		SimpleResponse resp = new Gson().fromJson(respStr, SimpleResponse.class);
		resp.handle("reqBody:%s",reqBody);
		return resp;
	}
	
	
	/**
{
  "touser":"OPENID",
  "msgtype":"image",
  "image": {
    "media_id":"MEDIA_ID"
  }
}
	 * @param accessToken
	 * @param openid
	 * @param text
	 * @return
	 */
	@Override
	public SimpleResponse sendCustomMessagePhoto(String accessToken,String openid,String photoUrl) {
		String reqBody = GsonData.build().param("touser", openid).param("msgtype", "image")
				//TODO photoUrl要换成 媒体id
		.object("image", textData->textData.param("media_id", photoUrl)).toJson();
		String action = "/cgi-bin/message/custom/send?access_token=%s";
		String url = server + String.format(action ,accessToken);
		String respStr = SimpleHttpUtil.simplePost(url, reqBody);
		SimpleResponse resp = new Gson().fromJson(respStr, SimpleResponse.class);
		resp.handle("reqBody:%s",reqBody);
		return resp;
	}

	@Override
	public SimpleResponse sendSubscribeMessage(String accessToken, String openid, String templateId,Optional<String> pathOptional, List<SubscribeMessageParam> params) {
		GsonData requestData = GsonData.build().param("touser", openid).param("template_id", templateId);
		pathOptional.ifPresent(path->requestData.param("page", path));
		SubscribeMessageParam.fillBody(requestData,params);
		String requestBody = requestData.toJson();
		String action = "/cgi-bin/message/subscribe/send?access_token=%s";
		String url = server + String.format(action ,accessToken);
		String responseBody = SimpleHttpUtil.simplePost(url, requestBody);
		SimpleResponse response = new Gson().fromJson(responseBody, SimpleResponse.class);
		response.handle("reqBody:%s",requestBody);
		return response;

	}


	/**
{
  "touser": "OPENID",
  "msgtype": "link",
  "link": {
    "title": "Happy Day",
    "description": "Is Really A Happy Day",
    "url": "URL",
    "thumb_url": "THUMB_URL"
  }
}
	 * @param accessToken
	 * @param openid
	 * @param title 标题
	 * @param desc 描述
	 * @param linkUrl 跳转的链接
	 * @param icoUrl 小图标
	 * @return
	 */
	@Override
	public SimpleResponse sendCustomMessagePhotoText(String accessToken,String openid,String title,String desc,String linkUrl,String icoUrl) {
		String reqBody = GsonData.build().param("touser", openid).param("msgtype", "link")
		.object("link", textData->textData.param("title", title).param("description",desc).param("url", linkUrl).param("thumbUrl", icoUrl))
		.toJson();
		String action = "/cgi-bin/message/custom/send?access_token=%s";
		String url = server + String.format(action ,accessToken);
		String respStr = SimpleHttpUtil.simplePost(url, reqBody);
		SimpleResponse resp = new Gson().fromJson(respStr, SimpleResponse.class);
		resp.handle("reqBody:%s",reqBody);
		return resp;
	}
	
	/**
{
 "touser":"OPENID",
 "msgtype":"miniprogrampage",
 "miniprogrampage": {
   "title":"title",
   "pagepath":"pagepath",
   "thumb_media_id":"thumb_media_id"
 }
}
	/**
	 * @param accessToken
	 * @param openid
	 * @param title 标题
	 * @param namespace  打开小程序的命名空间
	 * @param photoUrl 小程序上的图片地址
	 * @return
	 */
	@Override
	public SimpleResponse sendCustomMessageMapp(String accessToken,String openid,String title,String namespace,String photoUrl) {
		String reqBody = GsonData.build().param("touser", openid).param("msgtype", "miniprogrampage")
		.object("miniprogrampage", textData->textData.param("title", title).param("pagepath",namespace)
				//TODO photoUrl要换成 媒体id
				.param("thumbMediaId", photoUrl))
		.toJson();
		String action = "/cgi-bin/message/custom/send?access_token=%s";
		String url = server + String.format(action ,accessToken);
		String respStr = SimpleHttpUtil.simplePost(url, reqBody);
		SimpleResponse resp = new Gson().fromJson(respStr, SimpleResponse.class);
		resp.handle("reqBody:%s",reqBody);
		return resp;
	}

}
