package com.mini.framework.util.string.gson.format;

import com.mini.framework.core.exception.ServerException;
import com.mini.framework.core.status.Status;
import com.mini.framework.util.annotation.AnnotationUtil;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.export.ExcelColumnDeclares;
import com.mini.framework.util.params.MapParams;
import com.mini.framework.util.string.RegexUtil;
import com.mini.framework.util.string.gson.format.bean.Concat;
import com.mini.framework.util.string.gson.format.bean.Fill;
import com.mini.framework.util.string.gson.format.bean.Json;
import com.mini.framework.util.string.gson.format.bean.Mapper;
import com.mini.framework.util.string.gson.format.bean.Merge;
import com.mini.framework.util.string.gson.format.bean.MergeList;
import com.mini.framework.util.string.gson.format.bean.Remove;
import com.mini.framework.util.string.gson.format.bean.Rename;
import com.mini.framework.util.string.gson.format.bean.Retain;
import com.mini.framework.util.string.gson.format.bean.Union;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.core.annotation.Order;

import java.lang.annotation.Annotation;
import java.lang.reflect.AnnotatedElement;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;


/**
 * 响应格式化之注解支持。
 * 也就是格式化这个功能上添加注解的映射。
 * @author jayheo
 * */
@Aspect
public abstract class ResponseFormatAnnotationSupport {


	//TODO 目前发现一个问题，如果不小心配置了多个  ResponseFormatAnnotationSupport 框架没有发现，导致不好排查问题。后续要处理

	public static final String defaultDataFormatKey="dataFormat";
	
	@Pointcut("@annotation(com.mini.framework.util.string.gson.format.ResponseFormat)")
	public void methodPoint(){}
	
	@Around("methodPoint()")
	@Order(100)
	public Object around(ProceedingJoinPoint  joinPoint) throws Throwable{
		findAnnotation(joinPoint, ResponseFormat.class)
				.orElseThrow(()->new ServerException("没有发现注解%s",ResponseFormat.class));

		List<ResponseFormat> responseFormats = findDeepUsefulResponseFormats(joinPoint);
		AssertUtil.assertNotFatalBug(!responseFormats.isEmpty(),"至少要有一个可用的响应注解:%s",ResponseFormat.class);
		List<ExcelColumnDeclares> excelColumnDeclaresList = findDeepExcelColumnDeclaresList(joinPoint);
		Optional<String> dataFormatParamOptional = queryDataFormatParam();
		Object originResult = joinPoint.proceed();
		return dataFormatParamOptional.map(dataFormatParam->{
			AssertUtil.assertNotFatalProgramConfig(originResult!=null,Status.Server.programConfigJava,"返回值为空的不允许使用格式处理器");
			// TODO 错配置了多次如何排查问题
			// TODO 返值是字符串的如何合理的处理，?   后续尝试通过gson测验字符器是不是能生成 map型数据测一下再使用。
			// 这里要注意一下，如果地 不小心配置了多个 ResponseFormatAnnotationSupport 可能会导致出错，因第一个格式化返回的是字符串。
			AssertUtil.assertNotFatalProgramConfig(!(originResult instanceof String),Status.Server.programConfigJava,"返回值字符串的不允许使用格式处理器,originResult:%s",originResult);
			return applyFormat(originResult,dataFormatParam,responseFormats,excelColumnDeclaresList);
		}).orElse(originResult);
	}


	private static List<ResponseFormat> findDeepUsefulResponseFormats(ProceedingJoinPoint joinPoint){
		AnnotatedElement targetMethodElement = findPointAnnotationElement(joinPoint);
		ArrayList<ResponseFormat> foundResponseFormats = AnnotationUtil.findElementDeepSortedAnnotationsDistinct(targetMethodElement, ResponseFormat.class);
		return ResponseFormat.Support.retainUsefulByWorkScheme(foundResponseFormats);
	}


	private static List<ExcelColumnDeclares> findDeepExcelColumnDeclaresList(ProceedingJoinPoint joinPoint){
		AnnotatedElement targetMethodElement = findPointAnnotationElement(joinPoint);
		return AnnotationUtil.findElementDeepSortedAnnotationsDistinct(targetMethodElement,ExcelColumnDeclares.class);
	}



	public Object applyFormat(Object origin, String dataFormatParam, List<ResponseFormat> aliasAnnotation, List<ExcelColumnDeclares> excelColumnDeclaresList){
		AssertUtil.assertNoBadReq(dataFormatParam.length()!=0, Status.BadReq.illParam, "dataFormat参数长度不能为0");
		LinkedHashMap<String, String> aliasMapper = findResponseFormatAliases(aliasAnnotation);
		String explainForApply = applyPrepareExplainAliases(dataFormatParam, aliasMapper);
		ExportResourceFromByte resourceExportFunction = showResourceExportFunction();
		AssertUtil.assertNotFatalBug(resourceExportFunction!=null,"参数 showResourceExportFunction() 不能返回空");
		GsonDataFormatBean gsonBean = GsonDataFormatBean.build(showResourceExportFunction(), excelColumnDeclaresList, origin);
		gsonBean.addRootProperty("responseFormatDescribe",createExplainDescribe(explainForApply,aliasAnnotation));

		gsonBean.applyCommandsFromExplain(explainForApply);
		String stringResult = gsonBean.createString();
		this.afterDataFormat(stringResult);
		return stringResult;
	}

	private MapParams createExplainDescribe(String explainForApply, List<ResponseFormat> aliasAnnotations){
		LinkedHashMap<String, String> aliasMapper = findResponseFormatAliases(aliasAnnotations);
		List<ExplainAliasCommandParams> list = ExplainAliasCommandParams.createFromAliasExplainMapper(aliasMapper);
		//TODO 如果还加上系统提供的命令方式就更加好了。
		return MapParams.build()
				.param("schemes",aliasAnnotations.stream().map(ResponseFormat::name).collect(Collectors.toList()))
				.param("applyCommandExplain",explainForApply)
				.param("prepareAliasExplainMapper",aliasMapper)
				.param("applyCommandList",CommandParams.resolverExplainWithDescribe(explainForApply))
				.param("prepareAliasExplainList", list)
				;
	}


	private LinkedHashMap<String, String> findResponseFormatAliases(List<ResponseFormat> aliasAnnotations){
		return aliasAnnotations.stream()
				.map(this::findResponseFormatAliases)
				.reduce(new LinkedHashMap<>(),
					//用前面的去覆盖后面的。
					(prev,next)->{
					next.putAll(prev);
					return next;
					}
				);

	}

	
	private LinkedHashMap<String, String> findResponseFormatAliases(ResponseFormat aliasAnnotation){
		String actionSplit = "/";
		FormatAlias[] aliases = aliasAnnotation.value();
		LinkedHashMap<String, String> aliasExplainMapper = new LinkedHashMap<>();
		for (FormatAlias responseFormatAlias : aliases) {
			final Object[] paramObjects = responseFormatAlias.params();
			final String alias = responseFormatAlias.alias();
			AssertUtil.assertSupport(aliasExplainMapper.get(alias)==null, "alias:[%s]重复了", alias);
			Merge[] merges = responseFormatAlias.merge();
			MergeList[] mergeLists = responseFormatAlias.mergeList();
			Remove[] removes = responseFormatAlias.remove();
			Union[] unions = responseFormatAlias.union();
			Mapper[] mappers = responseFormatAlias.mapper();
			Concat[] concats = responseFormatAlias.concat();
			Fill[] fills = responseFormatAlias.fill();
			Rename[] renames = responseFormatAlias.rename();
			Json[] jsons = responseFormatAlias.json();
			Retain[] retains = responseFormatAlias.retain();

			Map<String,Object[]> commands = new LinkedHashMap<String, Object[]>();
			commands.put("merge", merges);
			commands.put("mergeList", mergeLists);
			commands.put("remove", removes);
			commands.put("json", jsons);
			commands.put("union", unions);
			commands.put("mapper", mappers);
			commands.put("concat", concats);
			commands.put("fills", fills);
			commands.put("renames", renames);
			commands.put("retains", retains);
			Object[] keys = commands.entrySet().stream().filter(command->command.getValue().length>0).map(Entry::getKey).toArray();
			//AssertUtil.assertSupport(keys.length<=1, "注解%s alias:[%s]中只允许一种命令注解目前有:%s", FormatAlias.class,alias,Arrays.toString(keys));
			//TODO 需要写一个方法把 FormatAlias 里的内容显示出来
			String[] explainItems = responseFormatAlias.explain();
			if(explainItems.length>0){
				AssertUtil.assertSupport(keys.length==0,"alias:[%s]中如果写了explain:[%s]不允许再使用%s",alias,String.join(",", explainItems),Arrays.toString(keys));
				String explain = Stream.of(explainItems).map( explainItem -> MessageFormat.format(explainItem,paramObjects)).collect(Collectors.joining(actionSplit));
				aliasExplainMapper.put(alias, explain);
			}else{
				AssertUtil.assertSupport(keys.length>0,"alias:[%s]中找不到 explain 必须改命令注解",alias);
				List<String> explains = new ArrayList<String>();
				
				for (Merge merge : merges) {
					explains.add(showExplain(merge));
				}
				for (MergeList mergeList : mergeLists) {
					explains.add( showExplain(mergeList));
				}
				for (Remove remove : removes) {
					explains.add( showExplain(remove));
				}
				for (Union union : unions) {
					explains.add( showExplain(union));
				}
				for (Mapper mapper : mappers) {
					explains.add( showExplain(mapper));
				}
				for (Concat concat : concats) {
					explains.add( showExplain(concat));
				}
				for (Fill fill : fills) {
					explains.add( showExplain(fill));
				}
				for (Rename rename : renames) {
					explains.add( showExplain(rename));
				}
				for (Json json : jsons) {
					explains.add( showExplain(json));
				}
				for (Retain retain : retains) {
					explains.add( showExplain(retain));
				}
				aliasExplainMapper.put(alias, StringUtils.join( explains.iterator(),actionSplit));
			}
			
		}
		return aliasExplainMapper;
	}


	private String applyPrepareExplainAliases(String explain, Map<String, String> aliases){
		AssertUtil.assertMethodRequire(explain,"explain");
		String preExplain;
		do{
			preExplain = explain;
			explain = RegexUtil.splitByPointEachHandle(CommandParams.commandExplainSplitRegex, explain, (item)-> aliases.getOrDefault(item, item));
			//如果修改了就要再找一下，直接没有修改为止
		}while( !explain.equals(preExplain));
		return explain;
	}

	/**
	 * 得到当前响应格式化的参数。
	 * */
	protected abstract Optional<String> queryDataFormatParam();
	
	/**
	 * 提供一个修改后的钩子后子类可以做一些事情，比如打印日志
	 * @param ret
	 */
	protected void afterDataFormat(String ret){}
	

	/**
	 * 提供一个资源导出的方法。
	 * @return
	 */
	protected abstract ExportResourceFromByte showResourceExportFunction();



	private static  <T extends Annotation> AnnotatedElement findPointAnnotationElement(JoinPoint joinPoint) {
		Signature signature = joinPoint.getSignature();
		MethodSignature methodSignature = (MethodSignature) signature;
		return methodSignature.getMethod();
	}

	private <T extends Annotation> Optional<T> findAnnotation(JoinPoint joinPoint, Class<T> annotationClass) {
		AnnotatedElement targetMethod = findPointAnnotationElement(joinPoint);
		return Optional.ofNullable(targetMethod.getAnnotation(annotationClass));
	}
	
	public String showExplain(Merge merge){
		AssertUtil.assertMethodRequire(merge, "merge");
		String explain = String.join(",", "merge",merge.targetPath(),merge.targetKey(),merge.soursePath(),merge.sourseKey(),merge.alias());
		return explain;
	}
	
	public String showExplain(MergeList mergeList){
		AssertUtil.assertMethodRequire(mergeList, "mergeList");
		String explain = String.join(",", "mergeList",mergeList.targetPath(),mergeList.targetKey(),mergeList.soursePath(),mergeList.sourseKey(),mergeList.listAlias());
		return explain;
	}
	
	public String showExplain(Mapper mapper){
		AssertUtil.assertMethodRequire(mapper, "mapper");
		String explain = String.join(",", "mapper",mapper.targetPath(),mapper.targetKey(),mapper.soursePath(),mapper.alias());
		return explain;
	}
	
	private String showExplain(Remove remove) {
		AssertUtil.assertMethodRequire(remove, "remove");
		AssertUtil.assertMethodRequire(remove.targetKeys().length>0,"remove里的targetKeys数必须大于0");
		String explain = String.join(",", "remove",remove.targetPath(),String.join(",",remove.targetKeys()));
		return explain;
	}

	private String showExplain(Union union) {
		AssertUtil.assertMethodRequire(union, "union");
		AssertUtil.assertMethodRequire(union.retainKeys().length>0,"union里的retainKeys数必须大于0");
		String explain = String.join(",", "union",union.targetPath(),union.targetKey(),union.soursePath(),union.sourseKey(), String.join(",",union.retainKeys()));
		return explain;
	}
	
	private String showExplain(Concat concat) {
		AssertUtil.assertMethodRequire(concat, "concat");
		AssertUtil.assertMethodRequire(concat.sourseKeys().length>0,"concat里的sourseKeys数必须大于0");
		String explain = String.join(",", "concat",concat.soursePath(),concat.alias(),concat.nullValue(),concat.split(), String.join(",",concat.sourseKeys()));
		return explain;
	}

	private String showExplain(Fill fill) {
		AssertUtil.assertMethodRequire(fill, "fill");
		String explain = String.join(",", "fill",fill.targetPath(),fill.targetKey(),fill.value());
		return explain;
	}

	private String showExplain(Rename rename) {
		AssertUtil.assertMethodRequire(rename, "rename");
		String explain = String.join(",", "rename",rename.targetPath(),rename.targetKey(),rename.newName());
		return explain;
	}

	private String showExplain(Json json) {
		AssertUtil.assertMethodRequire(json, "json");
		String explain = String.join(",", "json",json.action().name(),json.targetPath(),json.targetKey(),json.alias());
		return explain;
	}

	private String showExplain(Retain retain) {
		AssertUtil.assertMethodRequire(retain, "remove");
		AssertUtil.assertMethodRequire(retain.targetKeys().length>0,"retain里的targetKeys数必须大于0");
		String explain = String.join(",", "retain",retain.targetPath(),String.join(",",retain.targetKeys()));
		return explain;
	}

}
