package com.mini.framework.util.export;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.text.MessageFormat;
import java.util.function.Function;
import java.util.stream.Stream;

import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.export.protocol.DefaultExcelCellContentSupplier;
import com.mini.framework.util.export.protocol.ExcelCellContentSupplier;
import com.mini.framework.util.export.protocol.branch.SwitchCases;

/**
 *
 * 道出表到excel功能中的列声明。
 * 主要描述其中一个列是如何生成的。
 * @author jayheo
 */
@Target(ElementType.METHOD)
@Retention(RetentionPolicy.RUNTIME)
@Documented
public @interface ExcelColumnDeclare {

	/**
	 * 排序依据 默认是100
	 * @return
	 */
	int sorter() default 100;


	/**
	 * 是不是忽略这个字段<br>
	 * 如果需要临时忽略这个列可以写成true,不会被渲染出来。默认不隐藏
	 * @return
	 */
	boolean skip() default false;
	
	/**
	 * 标题
	 * @return
	 */
	String title();
	
	/**
	 * 取数据的路径
	 * @return
	 */
	String dataPath();
	
	/**
	 * 如果没有值的时候的默认值
	 * @return
	 */
	String defaultValue() default "";

	/**
	 * 内容提供者<BR>
	 * 这里指定格式转换器，例如  日期表格，等其他格式，相对时间转换器<BR>
	 * 可以参考  ContentSupplier 中的各种类来完成对应的业务 <BR>
	 * 目前可选的类如下,也可以自定义一些 ContentSupplier 的类:<BR>
	 *<pre>
	 contentSupplier = ContentSupplier.MoneyNumber.FenOfYuan.CnyZhCn.class
	 contentSupplier = ContentSupplier.MoneyNumber.DateLong.Full.class
	 contentSupplier = ContentSupplier.MoneyNumber.DateLong.OnlyTime.class
	 contentSupplier = ContentSupplier.MoneyNumber.DateLong.OnlyDate.class
	 *</pre>
	 * 这里要注意无论不要定义接口，只能定义类，且有能被实例化的默认构造方法，因为目前是通过 newInstance 方法生成的实例。<BR>
	 * @see com.mini.framework.util.export.protocol.ContentSupplier
	 * @return
	 */
	Class<? extends ExcelCellContentSupplier> contentSupplier() default DefaultExcelCellContentSupplier.class;


	/**
	 * 在完成以前做格式化,这里要属于格式化用的表达式<br>
	 * <pre>
	 MessageFormat.format("前面的内容{0}后面的内存",origin)
	 注意要写上 {0} 如果没有的话，原来的值的没有了。
	 * </pre>
	 * @return
	 * @see MessageFormat#format(String, Object...)
	 */
	String[] formatsAfterSupplier() default {"{0}"};


	/**
	 * 一开始的分支案例选择
	 * @return
	 */
	SwitchCases firstSwitchCaseReplace () default @SwitchCases(nameCases = {});



	/**
	 * 一个静态的支持类
	 */
	static interface Support{

		/**
		 * 通过原始数据算出string 类型的结构
		 * @param declare
		 * @param origin
		 * @return
		 */
		static String contentFromOrigin(ExcelColumnDeclare declare,String origin){
			AssertUtil.assertMethodRequire(origin,"origin");
			Class<? extends ExcelCellContentSupplier> contentSupplierClass = declare.contentSupplier();
			//TODO 这里拿着 contentSupplierClass 当作一个类去 创建实例，报错了。没有作强检查。
			//TODO 如果不能从语法上检查到未，那么需要在报错时文字提示到位，看到错误能自行排查出来。
			//TODO 也可以考虑到使用 map 映射一接收处理器。

			SwitchCases switchCaseReplace = declare.firstSwitchCaseReplace();
			origin = SwitchCases.Support.applyCases(switchCaseReplace,origin);
			ExcelCellContentSupplier columnContentSupplier = ExcelCellContentSupplier.createInstance(contentSupplierClass);
			String content = columnContentSupplier.supplier(origin);
			return Stream.of(declare.formatsAfterSupplier())
					.reduce(content,(c,e)->MessageFormat.format(e,c));
		}


		/**
		 * 通过原始数据算出string 类型的结构 的提供器
		 * @param declare
		 * @return
		 */
		static Function<String,String> contentSupplierFromOrigin(ExcelColumnDeclare declare){
			return origin->contentFromOrigin(declare,origin);
		}


	}
}
