package com.mini.framework.util.http.linked;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.function.Function;

import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;

import com.mini.framework.core.exception.BadReqException;
import com.mini.framework.core.exception.standard.UnsupportedException;
import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.params.MapParams;



public class HttpRequest {
	public static final String post = "POST";
	public static final String get = "GET";
	public static final String delete = "DELETE";
	public static final String put = "PUT";
	/**
	 * get post put delete
	 */
	private HttpMethod method;

	/**
	 * 服务域名
	 */
	private String endpoint;

	/**
	 * 接口的命名空间
	 */
	private String path;

	/**
	 * 头部请求参数
	 */
	private LinkedHashMap<String, String> header = new LinkedHashMap<>();

	/**
	 * url上的请求参数
	 */
	private LinkedHashMap<String, String> query = new LinkedHashMap<>();

	/**
	 * 请求体
	 */
	private LinkedHashMap<String, String> body = new LinkedHashMap<>();



	public HttpMethod getMethod() {
		return method;
	}

	public void setMethod(HttpMethod method) {
		this.method = method;
	}

	public String getEndpoint() {
		return endpoint;
	}

	public void setEndpoint(String endpoint) {
		this.endpoint = endpoint;
	}

	public String getPath() {
		return path;
	}

	public void setPath(String path) {
		this.path = path;
	}

	public LinkedHashMap<String, String> getHeader() {
		return header;
	}

	public void setHeader(LinkedHashMap<String, String> header) {
		this.header = header;
	}

	public LinkedHashMap<String, String> getQuery() {
		return query;
	}

	public void setQuery(LinkedHashMap<String, String> query) {
		this.query = query;
	}

	public LinkedHashMap<String, String> getBody() {
		return body;
	}

	public void setBody(LinkedHashMap<String, String> body) {
		this.body = body;
	}

	// ----------------------------------


	public static  HttpRequest create() {
		return new HttpRequest();
	}

	public HttpRequest method(HttpMethod method) {
		this.method = method;
		return this;
	}

	public HttpRequest endpoint(String endpoint) {
		this.endpoint = endpoint;
		return this;
	}

	public HttpRequest path(String path) {
		this.path = path;
		return this;
	}
	
	/**
	 * 指定url
	 * @param urlPath
	 * @return
	 */
	public HttpRequest url(String urlPath){
		try {
			URL url = new URL(urlPath);
			return endpoint(url.getProtocol()  + "://"+ url.getHost()).path(url.getFile());
		} catch (MalformedURLException e) {
			throw new BadReqException(Status.BadReq.illUrlSyntax,"url:[%s]格式错误", urlPath);
		}
	}

	public HttpRequest post(String path) {
		return this.method(HttpMethod.post).path(path);
	}

	public HttpRequest get(String path) {
		return this.method(HttpMethod.get).path(path);
	}

	public HttpRequest put(String path) {
		return this.method(HttpMethod.put).path(path);
	}

	public HttpRequest delete(String path) {
		return this.method(HttpMethod.delete).path(path);
	}

	public HttpRequest bodys(Map<String, String> bodyParams) {
		bodyParams.forEach((key, value) -> body(key, value));
		return this;
	}

	public HttpRequest body(String key, String value) {
		body.put(key, value);
		return this;
	}

	public HttpRequest querys(Map<String, ? extends Object> queryParams) {
		queryParams.forEach((key, value) -> query(key, value));
		return this;
	}

	public HttpRequest query(String key, Object value) {
		query.put(key, value.toString());
		return this;
	}

	public HttpRequest headers(Map<String, String> headerParams) {
		headerParams.forEach((key, value) -> header(key, value));
		return this;
	}

	public HttpRequest header(String key, String value) {
		header.put(key, value);
		return this;
	}
	

	/*
	 * public Request<T> parseByteReselt(Function<byte[], T>
	 * byteResultParser) { this.byteResultParser = byteResultParser; return
	 * this; }
	 */


	// ----------------------------

	public void validateComplate() {
		AssertUtil.assertNoBadReq(method!=null, Status.BadReq.illParam, "method不能为空");
		AssertUtil.assertNoBadReq(path!=null, Status.BadReq.illParam, "path不能为空");
		AssertUtil.assertNoBadReq(endpoint!=null, Status.BadReq.illParam, "endpoint不能为空");
	}

	public String showUrl() {
		return MapParams.build().params(query).toUrlParams(endpoint + path);
	}

	public HttpRequestBase requestEntity() {
		validateComplate();
		switch (method) {
		case delete:
			return new HttpDelete(showUrl());
		case put:
			return new HttpPut(showUrl());
		case post:
			return new HttpPost(showUrl());
		case get:
			return new HttpGet(showUrl());
		default:
			throw new UnsupportedException("不支持的method:%s", method);
		}
	}

	public HttpResponse execute(Function<HttpRequest,Boolean> before) {
		validateComplate();
		Boolean right = before.apply(this);
		if(right){
			return HttpClientHelper.execute(this);
		}else{
			return HttpResponse.createAbort(this);
		}
	}
}