package com.mini.framework.util.reflect;

import com.mini.framework.core.exception.LogicException;
import com.mini.framework.util.asserts.AssertUtil;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * java 返回帮助类
 * @author jayheo
 */
public class ReflectUtil {


    /**
     * 获取简单的pojo 属性 非静态非final的属性。
     * @param clazz
     * @return
     */
    public static List<Field> getSimplePojoFieldFromClass(Class<?> clazz){
        return getAllFieldFromClass(clazz).stream()
                .filter(field -> !Modifier.isFinal(field.getModifiers()))
                .filter(field -> !Modifier.isStatic(field.getModifiers()))
                .filter(field -> !"serialVersionUID".equals(field.getName()))
                .collect(Collectors.toList());
    }


    /**
     * 返回类所有的属性，包含父类
     * @return
     */
    public static List<Field> getAllFieldFromClass(Class<?> clazz){
        AssertUtil.assertMethodRequire(clazz,"clazz");
        List<Field> allField = new ArrayList<>();
        Class<?>tempClass =  clazz;
        do{
            Field[] fields = tempClass.getDeclaredFields();
            allField.addAll(Arrays.asList(fields));
            tempClass = tempClass.getSuperclass();
        }while(tempClass!=null && tempClass!=Object.class);
        return allField;
    }


    public static boolean notExistClassByName(String className){
        return !existClassByName(className);
    }

    /**
     * 判断是不是存在某一个类
     * @param className 类的完整的名字
     * @return 是不是存在
     */
    public static boolean existClassByName(String className){
        AssertUtil.assertMethodRequire(className,"className");
        try{
            Class.forName(className);
            return true;
        }catch (ClassNotFoundException e){
            //ignore e
            return false;
        }
    }


    /**
     * 比较两个对象是不是属于同一个类加载器
     * @param object1
     * @param object2
     * @return
     */
    public static boolean matchOneClassLoader(Object object1, Object object2){
        AssertUtil.assertMethodRequire(object1,"object1");
        AssertUtil.assertMethodRequire(object2,"object2");
        int hashCode1 = findBelongClassLoader(object1).map(Objects::hashCode).orElse(1);
        int hashCode2 = findBelongClassLoader(object2).map(Objects::hashCode).orElse(2);
        //int hashCode2 = getBelongClassLoader(object2).hashCode();
        return hashCode1 == hashCode2;
    }


    public static ClassLoader getBelongClassLoader(Object object) {
        return findBelongClassLoader(object)
                .orElseThrow(() -> new LogicException("在获取类加载器的时候发现找不到加载器 object.getClass():[%s]", object.getClass()));
    }
    
    
    public static Optional<ClassLoader> findBelongClassLoader(Object object) {
        AssertUtil.assertMethodRequire(object, "object");
        return Optional.of(object).map(obj -> {
            if (object.getClass() == String.class) {
                return null;
            }else if (object.getClass() == Class.class) {
                return ((Class<?>) object).getClassLoader();
            } else if (object .getClass() == Field.class) {
                return ((Field) object).getDeclaringClass().getClassLoader();
            } else if (object .getClass() == Method.class) {
                return ((Method) object).getDeclaringClass().getClassLoader();
            } else if (object.getClass().equals(Object.class)){
                return null;
            }else{
                return object.getClass().getClassLoader();
            }
        });
    }
}
