package com.mini.framework.util.access;

import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.ArrayUtils;

import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;

/**
 * 批量操作权限限制
 * @author jayheo
 *
 */
public class BatchAccessUtil {


	/**
	 * 创建一个bean
	 * @param key 作用域
	 * @param expireSeconds 以当时间加多少秒得到时间
	 * @param paramList
	 * @return
	 */
	public static <T extends Object> SimpleAccessBean createAccessToken(String key,int expireSeconds,List<T> paramList){
		return createAccessToken(key, expireSeconds, toStringArray(paramList));
	}
	
	/**
	 * 创建一个bean
	 * @param key 作用域
	 * @param expireSeconds 以当时间加多少秒得到时间
	 * @param params
	 * @return
	 */
	public static SimpleAccessBean createAccessToken(String key,int expireSeconds,String... params){
		Date expire = new Date(System.currentTimeMillis() + expireSeconds *1000);
		return createAccessToken(key, expire , params);
	}
	
	/**
	 * 创建一个授权的token
	 * @param key 描述这个是用于什么
	 * @param expire 描述过期时间
	 * @param params 具体的参数序列
	 * @return
	 */
	public static SimpleAccessBean createAccessToken(String key,Date expire,String... params){
		String salt = "minidev";
		AssertUtil.assertMethodRequire(key	,"key");
		AssertUtil.assertMethodRequire(expire	,"expire");
		AssertUtil.assertMethodRequire(params	,"params");
		AssertUtil.assertSupport(Stream.of(params).filter(param->param==null).count()==0	,"不能有参数为空");
		String sum  = String.join("-",ArrayUtils.addAll(new String[]{salt,key,expire.getTime() + ""}, params));
		String token = DigestUtils.md5Hex(sum);
		return new SimpleAccessBean(key, token, expire, params);
	}
	
	private static boolean checkAccessToken(String token ,String key,Date expire,String... params){
		SimpleAccessBean realAccessBean = createAccessToken(key, expire, params);
		return realAccessBean.getAccessToken().equals(token);
	}
	
	

	/**
	 * 验证token的正确性
	 * @param token 加密的token
	 * @param key 作用域
	 * @param paramList 参数
	 */
	public static <T extends Object> void valiAccessToken(String token,String key,List<T> paramList){
		valiAccessToken(token, key, toStringArray(paramList));
	}
	

	public static <T extends Object> void valiAccessToken(String token,String key,Integer... params){
		valiAccessToken(token, key, Arrays.asList(params));
	}

	/**
	 * @param token 需要验证的token
	 * @param token
	 * @param params
	 */
	public static void valiAccessToken(String token,String key,String... params){
		AssertUtil.assertMethodRequire(key, "key");
		SimpleAccessBean accessBean = new SimpleAccessBean(token);
		AssertUtil.assertNoBadReq(key.equals(accessBean.getAccessKey()),Status.BadReq.illParam, "accessKey应该为:%s",key);
		valiAccessToken(accessBean , params);
	}
	
	/**
	 * @param accessBean 验证token的bean
	 * @param params
	 */
	private static void valiAccessToken(SimpleAccessBean accessBean,String... params){
		AssertUtil.assertMethodRequire(accessBean	,"accessBean");
		valiAccessToken(accessBean.getAccessToken(), accessBean.getAccessKey(), accessBean.getExpireDate(), params);
	}
	
	private static <T extends Object>  String[] toStringArray(List<T> paramList){
		String[] params = paramList.stream().map(Object::toString).collect(Collectors.toList()).toArray(new String[]{});
		return params;
	}
	

	private static <T extends Object> void valiAccessToken(String token ,String key,Date expire,List<T> paramList){
		valiAccessToken(token, key, expire, toStringArray(paramList));
	}
	
	/**
	 * 验证access token 基础方法
	 * @param token token加密字符串
	 * @param key token应用的领域
	 * @param expire 过期时间
	 * @param params 要验证的数据
	 */
	private static void valiAccessToken(String token ,String key,Date expire,String... params){
	
		AssertUtil.assertMethodRequire(token	,"token");
		AssertUtil.assertMethodRequire(key	,"key");
		AssertUtil.assertMethodRequire(expire	,"expire");
		AssertUtil.assertMethodRequire(params	,"params");
		Date nowDate = new Date();
		boolean check = checkAccessToken(token, key, expire, params);
		AssertUtil.assertNoBadReq(nowDate.compareTo(expire)<0,Status.BadReq.badbatchTokenExpire, "%s的batchToken过期了(%s ms)",key,nowDate.getTime()-expire.getTime());
		AssertUtil.assertNoBadReq(check,Status.BadReq.badbatchToken, "batchToken无效");
	}
}
