package com.mini.framework.util.params;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.mini.framework.util.optional.OptionalDateRange;
import com.mini.framework.util.string.StringUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.mini.framework.core.exception.standard.UnsupportedException;
import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.bean.BeanUtil;

public class MapParams extends LinkedHashMap<String, Object>{


	private Logger logger = LogManager.getLogger(MapParams.class);
	
	/**
	 * 
	 */
	private static final long serialVersionUID = 5748967435422272687L;
	
	
	//当前支持多少具体要看看 baseStatementV1的配置。
	
	protected static String USER_ID = "userId";
	protected static String ID = "id";
	protected static String CREATE_DATE = "createDate";
	protected Map<String, Type> multiFilterFieldType = new HashMap<>(); 
	protected LinkedHashMap<String, String> inFieldsKey = getInFieldsKey(10);
	
	private int nullValueCount = 0;
	
	private int orderByIndex = 0;
	
	private int betweenIndex = 0;
	
	private Map<String, String> orderByParams = new LinkedHashMap<>();
	
	public static MapParams build(String paramKey,Object paramValue){
		return build().param(paramKey, paramValue);
	}


	private LinkedHashMap<String, String> getInFieldsKey(int max) {
		LinkedHashMap<String, String> inFields = new LinkedHashMap<String, String>();
		for (int i = 1; i <= max; i++) {
			inFields.put("inField" + i, "inValues" + i);
		}
		return inFields ;
	}

	public static MapParams buildUserId(Object userId){
		return build().param(USER_ID, userId);
		
	}
	public static MapParams buildId(Object id){
		return build().param(ID, id);
	}
	
	public static MapParams build(){
		return new MapParams();
	}
	
	@Deprecated
	public <T> MapParams paramIfAbsent(String paramKey,Optional<T> paramValueOptional){
		return paramIfPresent(paramKey, paramValueOptional);
	}

	public <T> MapParams paramIfPresent(String paramKey,Optional<T> paramValueOptional){
		AssertUtil.assertMethodRequire(paramValueOptional, "paramValueOptional");
		return paramIfAbsent(paramKey, paramValueOptional.orElse(null));
	}
	
	
	@Deprecated
	public MapParams paramIfAbsent(String paramKey,Object paramValue){
		return paramIfPresent(paramKey, paramValue);
	}
	
	public MapParams paramIfPresent(String paramKey,Object paramValue){
		if(paramValue!=null){
			this.param(paramKey, paramValue);
		}
		return this;
	}
	
	/**
	 * 如果是不确定的参数 使用 #paramIfPresent
	 * @param paramKey
	 * @param paramValue
	 * @return
	 */
	public MapParams param(String paramKey,Object paramValue){
		AssertUtil.simpleAssertByStatus(paramKey!=null&&paramKey.length()>0,Status.Busy.fatalBug, "不能传一个paramKey为空的参数val:%s", paramValue);
		//TODO 碰到这个问题 如果是不确定的参数 使用 paramIfPresent
		AssertUtil.assertMethodRequire(paramValue, "paramValue");
		put(paramKey, paramValue);
		return this;
	}
	
	public MapParams params(Map<String,?> map){
		map.forEach((key,value)->paramIfAbsent(key, value));
		return this;
	}
	
	public MapParams paramsFromBean(Object bean){
		AssertUtil.assertMethodRequire(bean, "bean");
		Map<String, Object> stringKeyValue = BeanUtil.beanToMap(bean);
		return params(stringKeyValue );
	}
	
	public MapParams paramAllowNull(String paramKey,Object paramValue){
		if(paramValue!=null){
			return param(paramKey, paramValue);
		}else{
			return paramNull(paramKey);
		}
	}
	
	public MapParams paramNull(String paramKey){
		AssertUtil.simpleAssertByStatus(paramKey!=null&&paramKey.length()>0,Status.Busy.fatalBug, "不能传一个paramKey为空的参数");
		put(paramKey, null);
		return this;
	}
	
	public MapParams paramUserId(Object paramValue){
		return this.param(USER_ID, paramValue);
	}
	
	/**
	 * @param key 排序的key
	 * @return
	 */
	public MapParams descBy(String key){
		return orderBy(key,true);
	}
	
	
	/**
	 * @param key 排序的key
	 * @return
	 */
	public MapParams ascBy(String key){
		return orderBy(key,false);
	}
	
	/**
	 * @param key 排序的key
	 * @param desc 是不是要倒序，true倒序 false顺序。
	 * @return
	 */
	public MapParams orderBy(String key ,boolean desc){
		AssertUtil.assertSupport(!orderByParams.containsKey(key),"排序key:[%s]不允许使用多次,params:[%s]",key, orderByParams);
		String suffix = orderByIndex==0?"":String.valueOf(orderByIndex);
		orderByIndex++;
		String orderByKey = "orderBy" + suffix;
		String orderByDescAscKey = "descAsc" + suffix;
		param(orderByKey, key);
		param(orderByDescAscKey, desc?"desc":"asc");
		orderByParams.put(key,  desc?"desc":"asc");
		AssertUtil.assertSupport(orderByParams.size()<=10, "目前只支持10个排序key,sortParams:%s",orderByParams);
		return this;
	}
	
	/**
	 * 设置limit
	 * @param offset
	 * @param length
	 * @return
	 */
	public MapParams limit(int offset,int length){
		//TODO 检查limit length 不能少于0
		param("limit", "limit");//这里做为判断条件。
		param("offset", offset);
		param("length", length);
		return this;
	}
	
	/**
	 * sql in 的使用
	 * 如果要多使用多个inField见{@link #addInField(String, Set)}
	 * @param field
	 * @param inValues
	 * @return
	 */
	@Deprecated//建议使用  addInField
	public MapParams inField(String field,Set<? extends Object> inValues){
		AssertUtil.assertSupport(get("inField")==null, "inField参数:[%s]不支持多次使用", field);
		param("inField", field);
		param("inValues", inValues);
		return this;
	}
	
	
	/**
	 * 添加in查询，如果有提供就执行，
	 * 
	 * @param field
	 * @param inValuesOptional 有数据提供且长度大于0才会被执行，否则不作任何事情。
	 * @return
	 */
	@Deprecated
	public <T> MapParams addInFieldIfAbsent(String field,Optional<Set<T>> inValuesOptional){
		return addInFieldIfPresent(field, inValuesOptional);
	}
	
	
	public <T> MapParams addInFieldIfPresent(String field,Optional<Set<T>> inValuesOptional){
		return addInFieldIfPresent(field, inValuesOptional.orElse(null));
	}

	/**
	 * 添加in查询，如果有提供就执行，
	 * 
	 * @param field
	 * @param inValues 有数据且长度大于0才会被执行，否则不作任何事情。
	 * @return
	 */
	@Deprecated
	public MapParams addInFieldIfAbsent(String field,Set<? extends Object> inValues){
		return addInFieldIfPresent(field, inValues);
	}
	
	public MapParams addInFieldIfPresent(String field,Set<? extends Object> inValues){
		if(inValues!=null&& inValues.size()>0){
			addInField(field, inValues);
		}
		return this;
	}
	
	@Deprecated
	public MapParams addInFieldIfAbsent(String field,Set<? extends Object> inValues,boolean notIn){
		return addInFieldIfPresent(field, inValues,notIn);
	}
	
	
	public MapParams addInFieldIfPresent(String field,Set<? extends Object> inValues,boolean notIn){
		if(inValues!=null&& inValues.size()>0){
			addInField(field, inValues,notIn);
		}
		return this;
	}
	
	@Deprecated
	public MapParams addNotInFieldIfAbsent(String field,Set<? extends Object> inValues){
		return addNotInFieldIfPresent(field, inValues);
	}
	
	
	public MapParams addNotInFieldIfPresent(String field,Set<? extends Object> inValues){
		if(inValues!=null&& inValues.size()>0){
			addInField(field, inValues,true);
		}
		return this;
	}
	

	public MapParams addNotInField(String field,Set<? extends Object> inValues){
		return addInField(field, inValues, true);
	}
	
	public MapParams addInField(String field,Set<? extends Object> inValues){
		return addInField(field, inValues, false);
	}
	
	
	/**
	 * 添加 in查询
	 * @param field
	 * @param inValues
	 * @return
	 */
	public MapParams addInField(String field,Set<? extends Object> inValues,boolean notIn){
		AssertUtil.assertMethodRequire(inValues, "inValues");
		AssertUtil.assertSupport(inValues.size()>0, "属性:[%s]的in条件数不能为0", field);
		AssertUtil.assertSupport(inValues.stream().filter(Objects::isNull).count()==0,"属性:[%s]的in条件不能有空指针inValues:%s",field,inValues);
		LinkedHashMap<String, String> inFieldsKey = new LinkedHashMap<>(this.inFieldsKey);
		int index = 0;
		for (java.util.Map.Entry<String, String> entry : inFieldsKey.entrySet()) {
			boolean haveField = get(entry.getKey())!=null;
			if(haveField){
				continue;
			}else{
				paramIfPresent("notInField" + (index+++1) ,notIn);
				param(entry.getKey(),field);
				param(entry.getValue(),inValues);
				return this;
			}
		}
		throw new UnsupportedException("最多只支持%s个inField", inFieldsKey.size());
	}
	
	
	
	/**
	 * 添加日期区间限制。这里一般用于创建日期。
	 * 坚持左闭右开原则。
	 * @param field 应用时间区间的字段
	 * @param fromDate 开始时间
	 * @param toDate 结束时间
	 * @return
	 */
	public MapParams dateRange(String field,Date fromDate,Date toDate){
		return dateAfter(field,fromDate).dateBefore(field,toDate);
	}
	
	/**
	 * 添加日期区间限制。这里一般用于创建日期。
	 * 坚持左闭右开原则。
	 * @param field 应用时间区间的字段
	 * @param fromDateOptional 开始时间
	 * @param toDateOptional 结束时间
	 * @return
	 */
	public MapParams dateRangeIfPresent(String field,Optional<Date> fromDateOptional,Optional<Date> toDateOptional){
		return dateBeforeIfPresent(field,toDateOptional).dateAfterIfPresent(field,fromDateOptional);
	}

	/**
	 * 添加日期区间限制。这里一般用于创建日期。
	 * 坚持左闭右开原则。
	 * @param field 应用时间区间的字段
	 * @param dateRangeOptional 时间区间可以为空
	 * @return
	 */
	public MapParams dateRangeIfPresent(String field, OptionalDateRange dateRangeOptional){
		AssertUtil.assertMethodRequire(dateRangeOptional,"dateRangeOptional");
		AssertUtil.assertMethodRequire(field,"field");
		return dateRangeIfPresent(field,dateRangeOptional.outFromDate(),dateRangeOptional.outToDate());
	}

	/**
	 * 必须提供数据不能为空 添加日期区间限制。这里一般用于创建日期。
	 * 坚持左闭右开原则。
	 * @param field 应用时间区间的字段
	 * @param dateRangeOptional 时间区间不可以为空
	 * @return
	 */
	public MapParams dateRangeRequirePresent(String field, OptionalDateRange dateRangeOptional){
		AssertUtil.assertMethodRequire(dateRangeOptional,"dateRangeOptional");
		return dateRange(field
				,dateRangeOptional.outFromDate().orElseThrow(()->new UnsupportedException("fromDate不能为空"))
				,dateRangeOptional.outToDate().orElseThrow(()->new UnsupportedException("toDate不能为空")));
	}
	

	/**
	 * 多少 时间以后
	 * @param time 时间单位毫秒。比如5分钟以后填 5*60*1000   也可以写负数让时间线向后推
	 * @return
	 */
	public MapParams timeAfter(TimeUnit timeUnit,int time){
		Date fromDate = new Date(System.currentTimeMillis() + timeUnit.toMillis(time));
		return dateAfter(fromDate );
	}
	/**
	 * 在某时间以后
	 * @param fromDate 时间点，可以相等
	 * @return
	 * 使用新的方法 @see  {@link MapParams#dateAfter(String, Date)}
	 */
	@Deprecated
	public MapParams dateAfter(Date fromDate){
		AssertUtil.simpleAssertByStatus(fromDate!=null, Status.Busy.bug, "开始时间不能为空");
		param("dateRange", "dateRange");
		param("rangeFromDate", fromDate);
		return this;
	}
	
	
	
	/**
	 * @param field 对应的字段名
	 * @param fromDateOptional 对应字段的日期
	 * @return
	 */
	public MapParams dateAfterIfPresent(String field,Optional<Date> fromDateOptional){
		AssertUtil.assertMethodRequire(fromDateOptional, "fromDateOptional");
		fromDateOptional.ifPresent(fromDate->dateAfter(field,fromDate));
		return this;
	}
	
	
	/**
	 * @param field 对应的字段名
	 * @param toDateOptional 对应字段的日期
	 * @return
	 */
	public MapParams dateBeforeIfPresent(String field,Optional<Date> toDateOptional){
		AssertUtil.assertMethodRequire(toDateOptional, "toDateOptional");
		toDateOptional.ifPresent(toDate->dateBefore(field,toDate));
		return this;
	}
	
	
	/**
	 * @param field 对应的字段名
	 * @param fromDate 对应字段的日期
	 * @return
	 */
	public MapParams dateAfter(String field,Date fromDate){
		AssertUtil.assertMethodRequire(field, "field");
		AssertUtil.assertMethodRequire(fromDate, "fromDate");
		AssertUtil.assertSupport(get("rangeFromField")==null, "不允许使用多个时间范围参数");
		param("rangeFromDate", fromDate);
		param("rangeFromField", field);
		return this;
	}
	
	
	/**
	 * 方法名写错了，请改用  numberBetweenIfPresent
	 * 数字介于两之间，如果没有就什么都不做。
	 * 这个方法暂时只支持使用4次
	 * @param field
	 * @param from 最小值 可以等于
	 * @param to 上限 不可以等于
	 * @return
	 */
	@Deprecated
	public MapParams numberBetweenIfAbsent(String field,Integer from,Integer to){
		return numberBetweenIfPresent(field, from, to);
	}
	
	
	/**
	 * 数字介于两之间，如果没有就什么都不做。
	 * 这个方法暂时只支持使用4次
	 * @param field
	 * @param from 最小值 可以等于
	 * @param to 上限 不可以等于
	 * @return
	 */
	public MapParams numberBetweenIfPresent(String field,Integer from,Integer to){
		return numberBetweenIfPresent(field, Optional.ofNullable(from), Optional.ofNullable(to));
	}
	
	
	/**
	 * 数字介于两之间，如果没有就什么都不做。
	 * 这个方法暂时只支持使用4次
	 * @param field
	 * @param fromOptional 最小值 可以等于
	 * @param toOptional 上限 不可以等于
	 * @return
	 */
	public MapParams numberBetweenIfPresent(String field,Optional<Integer> fromOptional,Optional<Integer> toOptional){
		AssertUtil.assertMethodRequire(fromOptional,"fromOptional");
		AssertUtil.assertMethodRequire(toOptional,"toOptional");
		if(fromOptional.isPresent()|| toOptional.isPresent()){
			return numberBetween(field, fromOptional.orElse(null), toOptional.orElse(null));
		}
		return this;
	}
	
	
	
	
	/**
	 * 数字介于两之间
	 * @param field
	 * @param from 最小值 可以等于
	 * @param to 上限 不可以等于
	 * @return
	 */
	public MapParams numberBetween(String field,Integer from,Integer to){
		AssertUtil.assertMethodRequire(field, "field");
		betweenIndex++;
		AssertUtil.assertSupport(from!=null || to!=null, "from和to不能都为空");
		AssertUtil.assertSupport(betweenIndex<=4, "numberBetween方式最多只支持3个   param:%s",this);
		String suffix = String.valueOf(betweenIndex);
		if(from!=null){
			param("betweenFromField" + suffix, field);
			param("betweenFromNumber" + suffix, from);
		}
		if(to!=null){
			param("betweenToField" + suffix, field);
			param("betweenToNumber" + suffix, to);
		}
		return this;
	}

	/**
	 * 在某时间以前
	 * @param field
	 * @param toDate
	 * @return
	 */
	public MapParams dateBefore(String field,Date toDate){
		AssertUtil.assertMethodRequire(field, "field");
		AssertUtil.assertMethodRequire(toDate, "toDate");
		AssertUtil.simpleAssertByStatus(toDate!=null, Status.Busy.bug, "结束时间不能为空");
		AssertUtil.assertSupport(get("rangeToField")==null, "不允许使用多个时间范围参数");
		param("rangeToDate", toDate);
		param("rangeToField", field);
		return this;
	}
	
	/**
	 * 强制把某些属性修改成null
	 * get和set开头的 方法可能与一些实体操作的工具冲突，所以不建议使用。应该使用 
	 * @see #addSetAttrNull(String...)
	 * @param fields
	 * @return
	 */
	@Deprecated
	public MapParams setAttrNull(String... fields){
		return addSetAttrNull(fields);
	}
	
	public MapParams addSetAttrNull(String... fields){
		return addSetAttrNull(Stream.of(fields).collect(Collectors.toSet()));
	}
	/**
	 * 强制把某些属性修改成null
	 * @param fields
	 * @return
	 */
	public MapParams addSetAttrNull(Set<String> fields){
		AssertUtil.assertSupport(fields.size()>0, "不能为传参数");
		AssertUtil.assertSupport(fields.size()<5, "目前最多支持4个参数,当前参数为:%s",fields);
		int i = 1;
		for (String field : fields) {
			param("setNullAttr" + (i++), field);
		}
		return this;
	}
	
	
	/**
	 * 强制把某些属性使用null条件
	 * @param fields
	 * @return
	 */
	public MapParams andAttrNull(String... fields){
		AssertUtil.assertSupport(fields.length>0, "不能为传参数");
		AssertUtil.assertSupport(fields.length<=7, "目前最多支持7个参数,当前参数为:%s",Arrays.toString(fields));
		for (int i = 0; i < fields.length; i++) {
			String field = fields[i];
			param("andNullAttr" + (i+1), field);
		}
		return this;
	}
	
	

	/**
	 * 使用布尔决定要不要使用attr为null
	 * @param ifNull
	 * @param fields
	 * @return
	 */
	public MapParams andAttrIfNull(boolean ifNull, String... fields){
		if(ifNull){
			return andAttrNull(fields);
		}else{
			return andAttrNotNull(fields);
		}
	}
	
	
	
	/**
	 * 强制把某些属性使用非null条件
	 * @param fields
	 * @return
	 */
	public MapParams andAttrNotNull(String... fields){
		AssertUtil.assertSupport(fields.length>0, "不能为传参数");
		AssertUtil.assertSupport(fields.length<=7, "目前最多支持7个参数,当前参数为:%s",Arrays.toString(fields));
		for (int i = 0; i < fields.length; i++) {
			String field = fields[i];
			param("andNotNullAttr" + (i+1), field);
		}
		return this;
	}
	
	/**
	 * 原子加
	 * @param field 需要增加的元素
	 * @param add 增量
	 * @param limit 最多允许使用几个
	 * @return
	 */
	public MapParams atomAdd(String field,int add,int limit){
		return multiField("atomAdd", "atomAddField", "atomAddNum", field, add, limit);
	}
	

	/**
	 * 对于某个参数使用
	 * @param field
	 * @return
	 */
/*	public MapParams distinct(String field){
		String key = "distinct";
		Object oldField = get(key);
		AssertUtil.assertSupport(oldField==null, "distinct只能使用一个目前已有:[%s]不能再使用:[%s]", oldField,field);
		return param("distinct", field);
	}*/
	
	@Deprecated
	public MapParams atomAddIfAbsent(String field,Integer add,int limit){
		return atomAddIfPresent(field, add, limit);
	}
	
	
	/**
	 * 原子加
	 * @param field 需要增加的元素
	 * @param add 增量
	 * @param limit 最多允许使用几个
	 * @return
	 */
	public MapParams atomAddIfPresent(String field,Integer add,int limit){
		if(add!=null){
			multiField("atomAdd", "atomAddField", "atomAddNum", field, add, limit);
		}
		return this; 
	}
	
	private MapParams multiField(String type,String fieldPrefix,String valuePrefix,String field,Object value,int limit){
		param(type,type);
		for (int i = 1; i <= limit; i++) {
			String key = fieldPrefix + i;
			Object val = get(key);
			if(val!=null){
				continue;
			}else{
				param(key, field);
				param(valuePrefix + i,value);
				return this;
			}
		}
		throw new UnsupportedException("%s类型的使用上限为%s",type, limit);
	}
	
	/**
	 * 属性的某个值比较
	 * @param field
	 * @param value
	 * @param compare
	 * @return
	 */
	public MapParams compare(String field,Object value,CompareType compare){
		AssertUtil.assertMethodRequire(field, "field");
		AssertUtil.assertMethodRequire(value, "value");
		AssertUtil.assertMethodRequire(compare, "compare");
		param("compare",true);
		AssertUtil.assertSupport(get(compare.getName())==null, "%s不能被使用两次", compare.getName());
		param(compare.getName(),true);
		param(compare.getFieldKey(), field);
		param(compare.getValueKey(), value);
		return this;
	}
	
	
	/**
	 * 混合过滤申明字符串
	 * @param fields
	 * @return
	 */
	public MapParams multiFilterString(String... fields){
		AssertUtil.assertSupport(multiFilterFieldType!=null, "匹配后不能再申明");
		AssertUtil.assertSupport(fields.length>0, "declareStringType查询至少要传一个参数");
		for (String field : fields) {
			multiFilterFieldType.put(field, String.class);
		}
		return this;
	}
	/**
	 * 混合过滤申明数字
	 * @param fields
	 * @return
	 */
	public MapParams multiFilterNumeric(String... fields){
		AssertUtil.assertSupport(multiFilterFieldType!=null, "匹配后不能再申明");
		AssertUtil.assertSupport(fields.length>0, "declareIntType查询至少要传一个参数");
		for (String field : fields) {
			multiFilterFieldType.put(field, Integer.class);
		}
		return this;
	}
	
	/**
	 * 分解关键字
	 * TODO 对于那些  += 等 符号也应该支持，但是又要防止出来sql注入
	 * @param keyword
	 * @return
	 */
	public String[] splitKeyword(String keyword){
		keyword = keyword.replaceAll( "[^a-zA-Z0-9\u4e00-\u9fa5]", " ");
		String[] keys = StringUtils.split(keyword, null);
		keys = Arrays.asList(keys).stream().filter(key->key!=null&&key.length()>0).distinct().collect(Collectors.toList()).toArray(keys);
		//TODO 要处理一下搜索条件，例如模糊搜索的时候，同时有   昵称和昵 两个词，其实昵称是没有义意的。
		//TODO 应该限制搜索条件的数据，太多了其实没有话意义。
		AssertUtil.validateParams(keys.length<30, "解析出来的搜索条件过于复杂原条件:[%s]解析后条件为%s",keyword,Arrays.toString(keys));
		return keys;
	}
	
	/**
	 * 混合声明匹配
	 * @param keywords
	 * @return
	 */
	public MapParams multiFilterMatch(String ... keywords){
		//TODO keyword是不是合法需要检查
		List<String > orSqls = new ArrayList<String>();
		for (String keyword : keywords) {
			Optional<String> orSqlOptional = multiFilterOrSqlMatch(keyword, multiFilterFieldType);
			if(orSqlOptional.isPresent()){
				orSqls.add(orSqlOptional.get());
				String debugMsg =String.format("根据关键字符串:[%s]查询字段:%s,得到sql片段:[%s]",keyword,multiFilterFieldType,orSqlOptional.get());
				logger.debug(debugMsg);
			}
		}
		if(orSqls.size()>0){
			param("keywordMultiLike", StringUtils.join(orSqls," and ") );
		}
		multiFilterFieldType = null;
		return this;
	}
	
	public Optional<String> multiFilterOrSqlMatch(String keyword, Map<String, Type> multiFilterFieldType){
		AssertUtil.assertSupport(multiFilterFieldType.size()>0, "必须先申明才能匹配");
		AssertUtil.assertSupport(keyword != null && keyword.length()!=0, "multiFilterMatch查询关键字不能为空");
		AssertUtil.assertSupport(keyword.trim().length() == keyword.length(), "multiFilterMatch查询关键字前后不能有无效字符");
		List<String> orSqls = new ArrayList<String>();
		String[] keys =splitKeyword(keyword);
		
		for (String key : keys) {
			for (String field : multiFilterFieldType.keySet()) {
				Type fieldType = multiFilterFieldType.get(field);
				AssertUtil.assertSupport(fieldType!=null, "field:[%s]没有声明数据类型", field);
				String sql = null;
				
				if(Integer.class.equals(fieldType)){
					if(StringUtils.isNumeric(key)){
						sql = field + " = " + key;
						orSqls.add(sql);
					}
				}else if(String.class.equals(fieldType)){
					sql = field + " like '%" + key + "%'";
					orSqls.add(sql);
				}else{
					throw new UnsupportedException("不支持的过滤数据类型%s", fieldType) ;
				}
			}
		}
		String orSqlString = StringUtils.join(orSqls, " or ");
		if(orSqlString.length()==0){
			return Optional.empty();
		}else{
			orSqlString = " ( " + orSqlString +  " ) ";
			return Optional.of(orSqlString);
		}
	}
	
	
	public LinkedHashMap<String, String> toStringMap(){
		return toStringMap(key->"null");
	}


	public LinkedHashMap<String, String> toStringMap(Function<String,String> defaultValueFunction){
		LinkedHashMap<String, String> map = new LinkedHashMap<>();
		for (java.util.Map.Entry<String, Object> entry : this.entrySet()) {
			String key = entry.getKey();
			Object value = entry.getValue();
			String stringValue = Optional.ofNullable(value)
					.map(Object::toString)
					.orElseGet(()->defaultValueFunction.apply(key));
			map.put(key,stringValue);
		}
		return map;
	}
	

	public int getNullValueCount() {
		return nullValueCount;
	}
	
	/**
	 * 根据是不是require决定要不要执行
	 * @param require
	 * @param consumer
	 * @return
	 */
	public <T> MapParams runRequire(T require,Consumer<MapParams> consumer){
		return runIf(require!=null, consumer);
	}
	
	/**
	 * 如果提供了就执行
	 * @param optional
	 * @param biConsumer
	 * @return
	 */
	public <T> MapParams ifPresent(Optional<T> optional,BiConsumer<MapParams,T> biConsumer){
		optional.ifPresent( v->biConsumer.accept(this,v));
		return this;
	}
	
	/**
	 * 运行某个
	 * @param consumer
	 * @return
	 */
	public MapParams run(Consumer<MapParams> consumer){
		return runIf(true, consumer);
	}
	
	
	/**
	 * 根据表达式判断要不要执行。
	 * @param explain
	 * @param consumer
	 * @return
	 */
	public MapParams runIf(boolean explain,Consumer<MapParams> consumer){
		if(explain){
			consumer.accept(this);
		}
		return this;
	}

	@Override
	public String toString() {
		return "nullValueCount:" + nullValueCount +",map:"+ super.toString();
	}
	
	/**
	 * 转成url参数
	 * @return
	 */
	public String toUrlParams(String url){
		AssertUtil.assertMethodRequire(url, "url");
		String split = url.contains("?")?"&":"?";
		return url +split  + toUrlParams();
	}
	
	public Function<String, String> asMapFunction(){
		LinkedHashMap<String, String> smap = toStringMap();
		return key->smap.get(key);
	}
	
	public String toParamSplit(String split,String itemSplit){
		return toStringMap().entrySet().stream().map(toParamSplitFunction(split)).collect(Collectors.joining(itemSplit));
	}
	
	public static Function<Map.Entry<String, String>, String> toParamSplitFunction(String split){
		return entry->toParamSplit(entry, split);
	}
	
	public static String toParamSplit(Map.Entry<String, String> entry,String split){
		return entry.getKey() + split + entry.getValue();
	}
	
	/**
	 * 转成url参数
	 * @return
	 */
	public String toUrlParams(){
		//TODO 对于value是date等情况要特殊处理一下。
		return toUrlParams(Object::toString,Function.identity());
	}

	/**
	 * 转成url参数 并把值 encode 一下。
	 * @return
	 */
	public String toUrlParamsEncodeValue(){
		//TODO 对于value是date等情况要特殊处理一下。
		return toUrlParams(Object::toString,StringUtil::urlEncode);
	}
	
	public String toUrlParams(Function<Object,String> valueToString,Function<String,String> urlEncodeHandle){
		MapParams other = MapParams.build().params(this);
		other.forEach((key,value)->{
			if(value==null){
				other.paramNull("");
			}else {
				other.param(key, valueToString.andThen(urlEncodeHandle).apply(value));
			}
		});
		return other.toParamSplit("=", "&");
		
		//List<String> params = entrySet().stream().map(entry->entry.getKey() + "=" + entry.getValue()).collect(Collectors.toList());
		//return String.join("&", params);
	}


	/**
	 * 通过key去排序
	 * @return
	 */
	public MapParams sortedByKey(){
		MapParams other = MapParams.build().params(this);
		clear();
		other.keySet().stream()
				//TODO 会不会有空key的情况呢？
				.sorted(Comparator.comparing(String::valueOf))
				.forEach(key->this.paramIfPresent(key,other.get(key)));
		return this;
	}

	public String toSortedUrlParams(){
		return MapParams.build().params(this).sortedByKey().toUrlParams();
	}


	/**
	 * 应用这个参数去查询
	 * @param executor
	 * @param <R>
	 * @return
	 */
	public <R> R apply(Function<MapParams,R> executor) {
		return executor.apply(this);
	}
}
