package com.mini.framework.util.report.statistics.protocol;

import com.mini.framework.core.exception.ServerException;
import com.mini.framework.core.exception.standard.CustomException;
import com.mini.framework.core.exception.standard.CustomExceptionSupplier;
import com.mini.framework.core.status.Status;
import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.date.DateRange;
import com.mini.framework.util.date.DateUtil;
import com.mini.framework.util.date.TimeSectionType;
import com.mini.framework.util.type.EnumTypeShower;
import org.apache.commons.lang3.time.DateUtils;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 时间区域单元
 * @author jayheo
 */
public enum TimeRegionUnit implements EnumTypeShower<TimeRegionUnit> {

    //TODO 需要添加分钟 和 一天的4时区

    /**
     * 分钟
     */
    minute("分钟",null,100,Optional.empty(),array()),

    /**
     * 小时
     */
    hour("小时",TimeSectionType.hour,200,Optional.of(minute),array(minute)),
    /**
     *自然日
     */
    day("自然日",TimeSectionType.day,300,Optional.of(hour),array(minute,hour)),
    /**
     * 自然周，目前是以周一为起点
     */
    week("自然周",TimeSectionType.week,400,Optional.of(day),array(minute,hour,day)),
    /**
     * 自然月
     */
    month("自然月",TimeSectionType.month,500,Optional.of(day),array(minute,hour,day)),
    /**
     * 自然季
     */
    season("自然季",TimeSectionType.season,600,Optional.of(month),array(minute,hour,day,month)),
    /**
     * 自然年
     */
    year("自然年",TimeSectionType.year,700, Optional.of(season),array(minute,hour,day,month,season)),
    /**
     * 有史以来
     */
    all("有史以来",TimeSectionType.all,800,Optional.of(year),array(minute,hour,day,month,season,year)),

    ;

    private String title;

    /**
     * 按大小排一个序号
     */
    private int sizeSorter;


    /**
     * 用于组成自己的某个子元素
     */
    private Optional<TimeRegionUnit> composeElement;

    /**
     * 可以用来做累积的子元素
     */
    private TimeRegionUnit[] accumulateElements;


    /**
     * 映射sectionType
     */
    private TimeSectionType sectionUnit;


    @Override
    public void fillFieldToMap(Map map) {
        map.put("title",title);
        map.put("sizeSorter",sizeSorter);
        map.put("composeElement",composeElement);
        map.put("accumulateElements",accumulateElements);
        map.put("sectionUnit",sectionUnit);
    }

    
    public static <T> T[] array(T ... array){
        return array;
    } 


    TimeRegionUnit(String title,TimeSectionType sectionUnit, int sizeSorter
            , Optional<TimeRegionUnit> composeElementOptional
            , TimeRegionUnit[] accumulateElements) {
        this.sectionUnit = sectionUnit;
        this.title = title;
        this.sizeSorter = sizeSorter;
        this.composeElement = composeElementOptional;
        this.accumulateElements = accumulateElements;
    }


    /**
     * size的顺序
     * @return
     */
    public static Comparator<TimeRegionUnit> sorterOfSize(){
        return Comparator.comparing(TimeRegionUnit::getSizeSorter);
    }

    public Optional<TimeRegionUnit> getComposeElement() {
        return composeElement;
    }

    public TimeSectionType getSectionUnit() {
        return sectionUnit;
    }

    /**
     * 校验能不能累积成为某个超级单元
     * @param superUnit
     * @return
     */
    public TimeRegionUnit validateCanAccumulateTo(TimeRegionUnit superUnit){
        AssertUtil.assertNormal(this.canAccumulateTo(superUnit),()->new ServerException("TimeRegionUnit:[%s]不能被组成:[%s]",this,superUnit));
        return this;
    }

    /**
     * 可以组合成载个类型(父级)
     * @param superUnit
     * @return
     */
    public boolean canAccumulateTo(TimeRegionUnit superUnit){
        AssertUtil.assertMethodRequire(superUnit,"superUnit");
        return Stream.of(superUnit.accumulateElements).anyMatch(this::equals);
    }

    public String getTitle() {
        return title;
    }

    public int getSizeSorter() {
        return sizeSorter;
    }

    public TimeRegionUnit[] getAccumulateElements() {
        return accumulateElements;
    }

    /**
     * 返回最近的点
     * @param pointLimit 这个时间是截至时间的最大值
     * @return
     */
    public TimeRegionRange nearFenceRange(Date pointLimit){
        return TimeRegionRange.create(this, nearLazyFenceRangeDate(pointLimit));
    }



    public boolean matchOffsetPoint(Date point) {
        return nearLazyFenceRangeDate(point).equals(point);
    }


    /**
     * 使用去尾法返回最近的点。
     * 需要换成新的方法。
     * @see #nearLazyFenceRangeDate(Date)
     * @param pointLimit
     * @return
     */
    @Deprecated
    public Date nearFenceRangeDate(Date pointLimit){
        return nearLazyFenceRangeDate(pointLimit);
    }

    /**
     * 使用去尾法返回最近的点。
     * 以时间 20210303144612840 为例有如下。
     * 具体见测试用例。
     * <pre>
     输入值:
     20210303144612840
     输出值:
     20210303144600000|minute
     20210303140000000|hour
     20210303000000000|day
     20210301000000000|week
     20210301000000000|month
     20210101000000000|season
     20210101000000000|year
     19700101080000000|all
     * </pre>
     * @param pointLimit 这个时间是截至时间的最大值
     * @return
     */
    public Date nearLazyFenceRangeDate(Date pointLimit){
        AssertUtil.assertMethodRequire(pointLimit, "pointLimit");
        Date cleanedDate = new Date(pointLimit.getTime());
        cleanedDate = DateUtils.setMilliseconds(cleanedDate, 0);
        cleanedDate = DateUtils.setSeconds(cleanedDate, 0);
        if(!this.equalType(minute)){
            cleanedDate = DateUtils.setMinutes(cleanedDate, 0);
        }
        switch (this) {
            case all :
                cleanedDate = new Date(0);
                break;
            case week:
                cleanedDate = DateUtil.setWeek(cleanedDate, 0);
                break;
            case year:
                //TODO 如果日期是的1530360000000L对象的会不会出问题，见  DateUtilTest.setMonthError 可能是时区导致的
                cleanedDate = DateUtils.setMonths(cleanedDate, 0);
            case season:
                int seasonMonth = DateUtil.seasonIndex(cleanedDate) * 3 ;
                //当从一个大月转到一个小月的时候可能出错，所以先把日期改到第一天
                cleanedDate = DateUtils.setDays(cleanedDate, 1);
                cleanedDate = DateUtils.setMonths(cleanedDate, seasonMonth);
            case month:
                cleanedDate = DateUtils.setDays(cleanedDate, 1);
            case day:
                cleanedDate = DateUtils.setHours(cleanedDate, 0);
            case hour:
            case minute:
                break;
            default:
                throw new ServerException(Status.Server.programConfigJava,"日期单元类型:[%s]无效必须处理", this);
        }
        return cleanedDate;
    }

    /**
     * 做为元素的检查器
     * @return
     */
    public static Predicate<TimeRegionUnit> elementUsefulPredicate(){
        return unit->!unit.equalAnyType(all);
    }


    /**
     * 使用去进一法返回最近的点。
     * @param pointLimit
     * @return
     */
    public Date nearGreedyFenceRangeDate(Date pointLimit){
        Date nearLazy = nearLazyFenceRangeDate(pointLimit);
        if(nearLazy.equals(pointLimit)){
            return nearLazy;
        }else {
            return addTimeUnit(nearLazy,1);
        }
    }


    /**
     * 显示从大到小获取一个可以作为元素的序列。
     * @param maxUnit
     * @return
     */
    public static List<TimeRegionUnit> showElementUsefulDownSort(TimeRegionUnit minUnit,TimeRegionUnit maxUnit){
        return Stream.of(TimeRegionUnit.values())
                .sorted(TimeRegionUnit.sorterOfSize().reversed())
                .filter(TimeRegionUnit.elementUsefulPredicate())
                .filter(unit->TimeRegionUnit.sorterOfSize().compare(maxUnit,unit)>=0)
                .filter(unit->TimeRegionUnit.sorterOfSize().compare(minUnit,unit)<=0)
                .collect(Collectors.toList());
    }


    /**
     * 校验时间是不是一个整时间点
     * @param dateRange
     * @param exceptionSupplier
     * @param <E>
     * @throws E
     */
    public <E extends CustomException> void validateNativeDateRightPoint(DateRange dateRange, CustomExceptionSupplier<E> exceptionSupplier)throws E{
        validateNativeDateRightPoint(dateRange.getMinDate(),exceptionSupplier);
        validateNativeDateRightPoint(dateRange.getMaxDate(),exceptionSupplier);
    }


    /**
     * 校验时间是不是一个整时间点
     * @param datePoint
     * @param exceptionSupplier
     * @param <E>
     * @throws E
     */
    public <E extends CustomException> void validateNativeDateRightPoint(Date datePoint,CustomExceptionSupplier<E> exceptionSupplier)throws E{
        Date rightPoint = nearLazyFenceRangeDate(datePoint);
        exceptionSupplier.applyAssert(rightPoint.equals(datePoint)
                ,"range中unit:[%s]当前offset:[%s]与正确的offset:[%s]不一致",this,datePoint.getTime(),rightPoint.getTime());
    }




    public Date addTimeUnit( Date oldDate, int amount){
        return addTimeUnit(this,oldDate,amount);
    }


    public static Date addTimeUnit(TimeRegionUnit timeUnit, Date oldDate, int amount){

        Date newDate = null;
        switch (timeUnit) {
            case minute:
                newDate = DateUtils.addMinutes(oldDate,amount);
                break;
            case hour:
                newDate = DateUtils.addHours(oldDate,amount);
                break;
            case day:
                newDate = DateUtils.addDays(oldDate,amount);
                break;
            case week:
                newDate = DateUtils.addWeeks(oldDate,amount);
                break;
            case month:
                newDate = DateUtils.addMonths(oldDate,amount);
                break;
            case season:
                newDate = DateUtils.addMonths(oldDate,amount*3);
                break;
            case year:
                newDate = DateUtils.addYears(oldDate,amount);
                break;
            case all:
                //从业务上来看  all添加时间是没有意义的，不是没有开发好，上层应该避免all出现使用这个方法的情形
                throw new ServerException(Status.Server.programConfigJava,"不支支持all的添加单元动作");
            default:
                //如果代码到了这里说明，存在一个类型没有处理，需要处理一下，一般是新增时间类型会出现这个问题。
                throw new ServerException(Status.Server.programConfigJavaFramework,"类型为:[%s]的enum没有处理需要立即修改程序", timeUnit);
        }
        return newDate;
    }




    /**
     * 根据一个子区间找出所有的子区间
     * @param oneElementRange
     * @return
     */
    public List<TimeRegionRange> findElementRegionFences(TimeRegionRange oneElementRange){
        TimeRegionUnit elementUnit = oneElementRange.getUnit();
        elementUnit.validateCanAccumulateTo(this);
        TimeRegionRange superRange = this.nearFenceRange(oneElementRange.getOffset());
        return oneElementRange.getUnit().regionFences(superRange.getOffset(),superRange.mapperToDate());
    }

    public List<TimeRegionRange> regionFences(Date fromDate, Date toDate){
        //TODO 需要确认一下，toDate闭区间还是开区间
        //TODO 需要确认一下，toDate限制的是开始时间还是结束时间。
        List<TimeRegionRange> fences = new ArrayList<>();
        Date pointDate = nearLazyFenceRangeDate(fromDate);
        TimeRegionRange pointFence = TimeRegionRange.create(this, pointDate);
        do {
            fences.add(pointFence);
            pointFence = pointFence.nextRegionRange();
        } while (pointFence.getOffset().before(toDate));
        return fences.stream()
                .filter(fence -> !fence.getOffset().before(fromDate))
                //TODO 需要确认一下，toDate限制的是开始时间还是结束时间。
                .filter(fence -> fence.getOffset().before(toDate))
                .collect(Collectors.toList());
    }


}
