package com.mini.framework.util.report.statistics.ranking.bean;


import com.mini.framework.util.asserts.AssertUtil;
import com.mini.framework.util.date.DateRange;
import com.mini.framework.util.date.DateUtil;
import com.mini.framework.util.report.statistics.protocol.RankingDimension;
import com.mini.framework.util.report.statistics.protocol.TimeRegionUnit;
import org.apache.commons.lang3.builder.ToStringBuilder;

import java.util.Date;
import java.util.Optional;
import java.util.UUID;

/**
 * 自然时间排序方案  排行榜的表
 */
public class NativeDateScoreRankingScheme {



    private RankingDimension dimension;


    /**
     * 参数序列化的字符串
     */
    private String paramsSerialize;
    /**
     * 参数序列化的字符串
     */
    private String paramsSerializeMd5;

    /**
     * 排行榜使用的key<br>
     * 如果使用redis等来存希望加上前缀 redis的key后缀 真实使用的时候还要加上项目规范的redis key<br>
     * TODO 从便于查询的角度来看，这个key应该使用数据。
     */
    private String rankingKey;

    /**
     * 时间区域的起点。
     */
    private Date regionOffset;

    /**
     * 统计的时间单元
     */
    private TimeRegionUnit regionUnit;


    /**
     * 区域的联合key，它能代表某个区域。<br>
     * 这个参数用于做时间 的联合条件。方便批量查询
     * @see #regionUnit
     * @see #regionOffset
     */
    private String regionUnionKey;

    /**
     * 触达的时间位置点。也就是说明，目前在这个时间点前的数据已触达，不包含当前时间点。
     */
    private Date touchUpperLimit;


    /**
     * 最大缓存的数量<BR>
     * 排行榜聚合缓存不能保留太多。
     */
    private Integer cacheItemLimit;

    /**
     * 是不是触达完整的时间范围<br>
     * 这里一个冗余字段
     */
    private Boolean touchFullyScope;


    /**
     * 根据基本属性创建，一般用于初始化scheme
     * @param regionUnit
     * @param regionOffset
     * @param paramsSerialize
     * @return
     */
    public static NativeDateScoreRankingScheme createByBasicField(RankingDimension dimension, TimeRegionUnit regionUnit, Date regionOffset,String paramsSerializeMd5
            ,int cacheItemLimit, String paramsSerialize) {
        NativeDateScoreRankingScheme instance = new NativeDateScoreRankingScheme();
        //写入基础参数
        instance.dimension = dimension;
        instance.paramsSerialize = paramsSerialize;
        instance.paramsSerializeMd5 = paramsSerializeMd5;
        instance.regionOffset = regionOffset;
        instance.regionUnit = regionUnit;
        instance.cacheItemLimit = cacheItemLimit;
        //随机数
        instance.rankingKey = UUID.randomUUID().toString();
        //计算出冗余参数
        instance.regionUnionKey = instance.countRegionUnionKey();
        instance.touchUpperLimit = instance.regionOffset;
        instance.touchFullyScope = false;
        return instance;
    }



    /**
     * 计算出需要触达的时间范围。<br>
     * eg. 按 21年1月15日来算，目前已统计 到了  10点整。现在要统计到 14点整，那么会返回的时间区间是 [10点,14点)
     * @param touchUpperPoint 已触达的时间点
     * @return
     */
    public Optional<DateRange> countRequireTouchRange(Date touchUpperPoint){
        AssertUtil.assertMethodRequire(touchUpperPoint,"touchUpperPoint");
        Date offset = this.touchUpperLimit!=null?this.touchUpperLimit: regionOffset;
        Date rangeLimitDate = regionUnit.addTimeUnit(regionOffset, 1);
        // 不能大于当前确认的时间范围
        Date limit = rangeLimitDate.before(touchUpperPoint)?rangeLimitDate:touchUpperPoint;
        return Optional.of(DateRange.create(offset,limit))
                .filter(DateRange::haveTimeScope)
                ;
    }


    /**
     * 是否已触达的时间上限
     * @param touchUpperPoint
     * @return
     */
    public boolean ifTouchRangeUpperLimit(Date touchUpperPoint){
        AssertUtil.assertMethodRequire(touchUpperPoint,"touchUpperPoint");
        Date rangeLimitDate = regionUnit.addTimeUnit(regionOffset, 1);
        return !rangeLimitDate.after(touchUpperPoint);
    }


    /**
     * 生成时间区域唯一的key
     * @return
     */
    public String countRegionUnionKey(){
        return countRegionUnionKey(regionUnit, regionOffset);
    }

    /**
     * 生成时间区域唯一的key
     * @param regionUnit
     * @param regionOffset
     * @return
     */
    public static String countRegionUnionKey(TimeRegionUnit regionUnit,Date regionOffset) {

        //TODO 应该与 TimeRegionRange中的unitOffsetUnionString联合使用
        return String.format("%s-%s", regionUnit, DateUtil.toSecondsString(regionOffset));
    }

    public boolean ifNotTouchFullyScope(){
        return ifNotTouchFullyScope();
    }

    /**
     * 是不是触达了 完整的时间范围。
     * @return
     */
    public boolean ifTouchFullyScope(){
        return Optional.ofNullable(touchFullyScope).orElse(false);
    }


    public String getRankingKey() {
        return rankingKey;
    }

    public void setRankingKey(String rankingKey) {
        this.rankingKey = rankingKey;
    }

    public Date getRegionOffset() {
        return regionOffset;
    }

    public void setRegionOffset(Date regionOffset) {
        this.regionOffset = regionOffset;
    }

    public TimeRegionUnit getRegionUnit() {
        return regionUnit;
    }

    public void setRegionUnit(TimeRegionUnit regionUnit) {
        this.regionUnit = regionUnit;
    }

    public Date getTouchUpperLimit() {
        return touchUpperLimit;
    }

    public void setTouchUpperLimit(Date touchUpperLimit) {
        this.touchUpperLimit = touchUpperLimit;
    }

    public String getRegionUnionKey() {
        return regionUnionKey;
    }

    public void setRegionUnionKey(String regionUnionKey) {
        this.regionUnionKey = regionUnionKey;
    }

    public Boolean getTouchFullyScope() {
        return touchFullyScope;
    }

    public void setTouchFullyScope(Boolean touchFullyScope) {
        this.touchFullyScope = touchFullyScope;
    }

    public String getParamsSerialize() {
        return paramsSerialize;
    }

    public void setParamsSerialize(String paramsSerialize) {
        this.paramsSerialize = paramsSerialize;
    }

    public String getParamsSerializeMd5() {
        return paramsSerializeMd5;
    }

    public void setParamsSerializeMd5(String paramsSerializeMd5) {
        this.paramsSerializeMd5 = paramsSerializeMd5;
    }

    public RankingDimension getDimension() {
        return dimension;
    }

    public void setDimension(RankingDimension dimension) {
        this.dimension = dimension;
    }

    public Integer getCacheItemLimit() {
        return cacheItemLimit;
    }

    public void setCacheItemLimit(Integer cacheItemLimit) {
        this.cacheItemLimit = cacheItemLimit;
    }
    

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

}
