package com.mini.framework.util.log.trace;

import java.util.Map;
import java.util.Optional;

import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.ThreadContext;
import org.apache.logging.log4j.util.ReadOnlyStringMap;

import com.mini.framework.util.log.MiniLogLevel;

/**
 * 日志跟踪门面<br>
 * 这里负责向线程里写跟踪信息，标记开始。<br>
 * @author zhxie
 * TODO 要把作用域从本地线程换到当前工作线程，跟踪作用域它依赖于本地线程，但是又没有那么宽。比如多个请求可能会复用同一个地本线程。
 */
public class LinkTraceFace {
	
	
	/**
	 * 服务id(两位)
	 * */
	private static String serverId;
	/**
	 * 集群id(一位) 
	 * */
	private static String colonyIndex;
	
	
	
	private static final Logger logger = LogManager.getLogger(LinkTraceFace.class);
	private static final String LINK_FEATHRE_LOG = "LINK_FEATHRE_LOG";
	private static final String MILLI_SECOND_FORMAT = "yyyyMMddHHmmssSSS";
	static final String BEGIN_POINT = "BEGIN_POINT";  
	static final String BEGIN_POINT_RANDOM = "BEGIN_POINT_RANDOM";  
	public final static String TRACE_ID = "ti";
	/**
	 * 注入用户id到日志里
	 */
	public static final String logUserPatternKey = "userKey";
	/**
	 * 注入商家id到日志里
	 */
	public static final String logSiteIdPatternKey = "siteId";
	
	
	/**
	 * 注入app index 到日志里
	 */
	public static final String logAppIndexPatternKey = "appIndex";
	
	
	
	/**
	 * 在链路开始打点  同添加特征
	 */
	public static void linkBeginPoint(String feature){
		if(feature!=null){
			ThreadContext.put(LINK_FEATHRE_LOG , feature);
		}
		setBeginPoint();
		resetTraceId();
	}
	
	
	/**
	 * 清除链接跟踪
	 */
	public static void clearTrace(){
		//logger.log(MiniLogLevel.getFrameWorkLog(),"trace end链路跟踪结束");
		ThreadContext.remove(TRACE_ID);
		ThreadContext.remove(BEGIN_POINT);
		ThreadContext.remove(BEGIN_POINT_RANDOM);
		ThreadContext.remove(logAppIndexPatternKey);
		ThreadContext.remove(logSiteIdPatternKey);
		ThreadContext.remove(logUserPatternKey);
	}
	
	/**
	 *  System.setProperty("Log4jContextSelector", "org.apache.logging.log4j.core.async.AsyncLoggerContextSelector");
	 */
	public static void setAsyncLogSystemProperty(){
		System.setProperty("Log4jContextSelector", "org.apache.logging.log4j.core.async.AsyncLoggerContextSelector");
	}
	
	
	/**
	 * 重新初始化TraceId
	 */
	private static void resetTraceId(){
		Long time = getBeginPoint();
		String pointTimeString = null;
		if(time!=null){//这时默认认为time不为空时，rendom也同时不为空 方法setBeginPoint() 会保证这个逻辑
			String random = ThreadContext.get(BEGIN_POINT_RANDOM);
			pointTimeString = DateFormatUtils.format(time, MILLI_SECOND_FORMAT) + random;
			
		}
		String feature = ThreadContext.get(LINK_FEATHRE_LOG);
		String traceId = (feature==null?"":serverId + colonyIndex +  feature)//如果不为空就加上服务id和集群id 
				+ (pointTimeString==null?"":pointTimeString);
		String oldTraceId = ThreadContext.get(TRACE_ID);
		if(oldTraceId != null && oldTraceId.length()!=0){//如果存在非空的oldTraceId
			if(!oldTraceId.equals(traceId)){//并且和现在的又一样那么需要做一个记录
				logger.log(MiniLogLevel.getFrameWorkLog(),String.format("change trace id target:[%s],old:[%s]",traceId,oldTraceId));
			}
		}
		ThreadContext.put(TRACE_ID, traceId);
		//logger.log(MiniLogLevel.getFrameWorkLog(),"trace begin链路跟踪开始");
	}
	
	
	private static long setBeginPoint(){
		long time = System.currentTimeMillis();
		String random = RandomStringUtils.randomNumeric(6);
		ThreadContext.put(BEGIN_POINT_RANDOM,random);
		ThreadContext.put(BEGIN_POINT, String.valueOf(time));
		return time;
	}
	
	/**
	 * 获取链路开始点，如果出现在任何错误返回null 上次一定要忽略这个null
	 */
	private static Long getBeginPoint(){
		String beginPointStr = ThreadContext.get(BEGIN_POINT);
		if(StringUtils.isNumeric(beginPointStr)){
			long beginPoint = Long.parseLong(beginPointStr);
			long year = 1000L*60*60*24*365;
			if(beginPoint > year*30 && beginPoint <year*300){
				//在30到300年间的都算正常时间1970年为计数起点
				return beginPoint;
			}else{
				//ingore 忽略这个情况，一般的不会放一个不符合格式的数据到里面。
			}
		}
		return null;
	}

	/**
	 * 标记用户key比如用户id
	 * @param userKey
	 */
	public static void markUserKey(String userKey){
		if(userKey!=null){
			ThreadContext.put(logUserPatternKey,userKey);
		}
	}
	
	/**
	 * 标记商家id
	 * @param siteId
	 */
	public static void markSiteId(String siteId){
		if(siteId!=null){
			ThreadContext.put(logSiteIdPatternKey,siteId);
		}
	}

	/**
	 * 标记app index
	 * @param appIndex
	 */
	public static void markAppIndex(String appIndex){
		if(appIndex!=null){
			ThreadContext.put(logAppIndexPatternKey, appIndex);
		}
	}
	
	public static Optional<Long> countThreadUserTime() {
		return countThreadUserTime(ThreadContext.getThreadContextMap().getReadOnlyContextData());
	}
	public static Optional<Long> countThreadUserTime(ReadOnlyStringMap readOnlyThreadContextMap) {
		return countThreadUserTime(readOnlyThreadContextMap.toMap());
	}

	/**
	 * 计算链路消耗时间 这里要注意，如果一开始打点的时间不准备，这个值就没有参考话话话意义
	 */
	public static Optional<Long> countThreadUserTime(Map<String, String> readOnlyThreadContextMap) {
		long now = System.currentTimeMillis();
		String beginPointStr = readOnlyThreadContextMap.get(BEGIN_POINT);
		if (StringUtils.isNumeric(beginPointStr)) {
			long beginPoint = Long.parseLong(beginPointStr);
			long userTime = now - beginPoint;
			if (userTime >= 0) {
				return Optional.ofNullable(userTime);
			}
		}
		return Optional.empty();// 如果没有得到耗时则设置成0
	}
	
	/**
	 * 把跟踪id显示出来，返回给调用方
	 * @return
	 */
	public static String showTraceId(){
		return ThreadContext.get(TRACE_ID);
	}
	

	public static void setServerId(String serverId) {
		LinkTraceFace.serverId = serverId;
	}

	public static void setColonyIndex(String colonyIndex) {
		LinkTraceFace.colonyIndex = colonyIndex;
	}
	
	public static String getServerId() {
		return serverId;
	}

	public static String getColonyIndex() {
		return colonyIndex;
	}


	/**
	 * 系统启动时用的
	 */
	public static void pointSystemBoot(){
		linkBeginPoint("900001");
	}
	/**
	 * 系统定时定时任务
	 * */
	public static void pointSystemQuartz(){
		linkBeginPoint("900002");
	}
	/**
	 * 临时用的，当不知道用什么的时候用这个吧。
	 * */
	public static void pointTemporary(){
		linkBeginPoint("900003");
	}
	
	/**
	 * 在不知道用什么Feature的时候用这个方法
	 */
	public static void pointUnknownFeature(){
		linkBeginPoint("900104");
	}
	
	/**
	 * 在强调必须使用Feature的时候用这个方法
	 */
	public static void pointRequiredFeature(){
		linkBeginPoint("900005");
	}
	
	/**
	 * 在强调必须使用Feature的时候用这个方法
	 */
	public static void pointInitWorkFeature(){
		linkBeginPoint("900006");
	}
}
