package com.mini.framework.util.oss.bean.work.mark;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Stream;

import com.mini.framework.util.bean.desc.Desc;
import com.mini.framework.util.oss.bean.OssImageProcessParams;
import com.mini.framework.util.oss.bean.work.type.OriginType;
import com.mini.framework.util.oss.bean.work.type.TextWordsType;

/**
 * https://help.aliyun.com/document_detail/44957.html?spm=a2c4g.11186623.6.1255.2bf116756fHoPJ
 * 文本水印处理
 * @author jayheo
 *
 */
@Desc("图片上的文本水印")
public class TextMarkOssImageWork extends MarkOssImageWork {
	
	private static final int textMaxLength = 20;
	private static final String textSpace = "　";
	private static final String textEndWith = "…";

	/**
	 * 参数意义：文字水印的文字的颜色
	 * 
	 * 参数的构成必须是：六个十六进制数，如：000000表示黑色。 000000每两位构成RGB颜色， FFFFFF表示的是白色
	 * 
	 * 默认值：000000黑色
	 */
	@Desc("6位颜色默认是000000")
	private String color;

	/**
	 * 文字的文本
	 */
	@Desc("文字的文本")
	private String text;

	@Desc("水印文本的字体")
	private TextWordsType type;

	/**
	 * 参数意义：文字水印的文字大小(px)
	 * 
	 * 取值范围：(0，1000]
	 * 
	 * 默认值：40
	 */
	@Desc("水印文本的子号(0,1000]默认40")
	private Integer size;

	/**
	 * 参数意义：文字水印的阴影透明度
	 * 
	 * 取值范围：[0,100]
	 */
	@Desc("文字水印的阴影透明度[0,100]")
	private Integer shadow;

	/**
	 * 参数意义：文字顺时针旋转角度
	 * 
	 * 取值范围：[0,360]
	 */
	@Desc("文字顺时针旋转角度[0,360]")
	private Integer rotate;

	/**
	 * 参数意义：进行水印铺满的效果
	 * 
	 * 取值范围：[0,1]，1表示铺满，0表示效果无效
	 */
	@Desc("进行水印铺满的效果 [0,1] 1表示铺满，0表示效果无效")
	private Integer fill;
	
	@Desc("最大限制多少个字，超过了使用…代替")
	private Integer lengthLimit;

	public String getColor() {
		return color;
	}

	public void setColor(String color) {
		this.color = color;
	}

	public String getText() {
		return text;
	}

	public void setText(String text) {
		this.text = text;
	}

	public TextWordsType getType() {
		return type;
	}

	public void setType(TextWordsType type) {
		this.type = type;
	}

	public Integer getSize() {
		return size;
	}

	public void setSize(Integer size) {
		this.size = size;
	}

	public Integer getShadow() {
		return shadow;
	}

	public void setShadow(Integer shadow) {
		this.shadow = shadow;
	}

	public Integer getRotate() {
		return rotate;
	}

	public void setRotate(Integer rotate) {
		this.rotate = rotate;
	}

	public Integer getFill() {
		return fill;
	}

	public void setFill(Integer fill) {
		this.fill = fill;
	}
	
	
	public Integer getLengthLimit() {
		return lengthLimit;
	}

	public void setLengthLimit(Integer lengthLimit) {
		this.lengthLimit = lengthLimit;
	}
	
	/**
	 * 显示出合适的文档
	 * @return
	 */
	public String showUsefulText(){
		String text = Optional.ofNullable(this.text).orElse(textSpace );
		int length = Stream.of(lengthLimit,textMaxLength).filter(Objects::nonNull).mapToInt(Integer::valueOf).filter(v->v>0).min().getAsInt();
		text = text.length()>length?(text.substring(0, length-1) +textEndWith):text;
		return text;
	}

	public OssImageProcessParams mapperProcessParams(Function<String, String> sureSameBucket){
		String text = showUsefulText();
		return super.mapperProcessParams(sureSameBucket)
				.paramBase64("text", text)
				.paramIfVal("color", color)
				.paramIfVal("size", size)
				.paramIfVal("shadow", shadow)
				.paramIfVal("rotate", rotate)
				.paramIfVal("fill", fill)
				.paramIfVal("type", type.base64())
				;
	}

	public TextMarkOssImageWork() {
		super();
	}

	public TextMarkOssImageWork(String color, String text, TextWordsType type, Integer size, Integer shadow, Integer rotate,
			Integer fill,Integer lengthLimit) {
		super();
		this.color = color;
		this.text = text;
		this.type = type;
		this.size = size;
		this.shadow = shadow;
		this.rotate = rotate;
		this.fill = fill;
		this.lengthLimit = lengthLimit;
	}
	

	public static List<TextMarkOssImageWork> createExamples(){
		List<TextMarkOssImageWork> examples = new ArrayList<TextMarkOssImageWork>();
		TextMarkOssImageWork example = new TextMarkOssImageWork("000000", "${这里写文本占位符}", TextWordsType.droidsansfallback, null, 10, 12, 0,8);
		example.fillParams(80, OriginType.nw, 100, 100, 100)
		.fillParams(50.0,"增加一个文本水印，需要自行修改占位符");
		examples.add(example);
		return examples ;
	}
}
